﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.ComponentModel;
using Terasoluna.SelectableValues;
using System.Reflection;
using System.Drawing.Design;
using System.Collections;

namespace Terasoluna.TourSample.Client.Common.View
{
    /// <summary>
    /// 
    /// CheckedItemsに対して画面データと複合データバインディングできるようにした
    /// 拡張CheckedListBoxの簡易実装
    /// </summary>
    /// ComplexBindingProperties属性の付与により、データソース画面からの貼り付けが可能になる
    [System.ComponentModel.ComplexBindingProperties("CheckedDataSource", "CheckedDataMember")]
    public class CheckedItemsBindableCheckedListBox : CheckedListBox
    {
        private bool checking = false;
        private bool listChanging = false;
        private BindingSource bindingSource = null;

        public CheckedItemsBindableCheckedListBox()
        {
            ItemCheck += new ItemCheckEventHandler(CheckedItemsBindableCheckedListBox_ItemCheck);
        }


        /// <summary>
        /// CheckedItemsとバインドされる画面データのプロパティ名
        /// </summary>
        public string StringValuePropertyName { get; set; }

        /// <summary>
        /// 複合バインディングの対象となる画面データのリスト（データソース）
        /// </summary>
        /// AttributeProvider属性の付与により、BindingSourceを選択するエディタになる
        [AttributeProvider(typeof(IListSource))]
        public object CheckedDataSource
        {
            get
            {
                return bindingSource;
            }
            set
            {
                //BindingSourceのみ対応
                BindingSource newDataSource = value as BindingSource;
                if (newDataSource != null)
                {
                    bindingSource = newDataSource;
                    bindingSource.ListChanged += new ListChangedEventHandler(bindingSource_ListChanged);
                    OnCheckedDataSourceChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// CheckedListBoxのItemCheckイベントに登録するデリゲート
        /// </summary>
        void CheckedItemsBindableCheckedListBox_ItemCheck(object sender, ItemCheckEventArgs e)
        {
            if (listChanging)
            {
                //bindingSource_ListChanged実行中は動作させない
                return;
            }
            if (bindingSource == null)
            {
                return;
            }
            checking = true;

            bool isChecked = (e.NewValue == CheckState.Checked);
            int index = e.Index;
            SelectableValue selectableValue = Items[index] as SelectableValue;
            if (selectableValue == null)
            {
                return;
            }
            //データソース内の画面データのリストを取得
            IList viewDataList = bindingSource.List;
            bool alreadyChecked = false;
            for (int i = 0; i < viewDataList.Count; i++)
            {
                PropertyInfo property = viewDataList[i].GetType().GetProperty(StringValuePropertyName);
                if (property != null && property.PropertyType.Equals(typeof(string)))
                {
                    if (string.Equals(property.GetValue(viewDataList[i], null), selectableValue.StringValue))
                    {
                        if (!isChecked)
                        {
                            bindingSource.RemoveAt(i);
                            i--;
                        }
                        else
                        {
                            alreadyChecked = true;
                        }                    
                    }
                }
            }
            if (isChecked && !alreadyChecked)
            {
                ///BindingSourceに要素を追加
                object newItem = bindingSource.AddNew();
                ///SelectedDataMemberプロパティで指定した値を持つ
                ///画面データプロパティに、SelectableValue.StringValueの値をセット
                PropertyInfo property = newItem.GetType().GetProperty(StringValuePropertyName);
                if (property != null && property.PropertyType.Equals(typeof(string)))
                {
                    property.SetValue(newItem, selectableValue.StringValue, null);
                }
            }
            checking = false;
            /// SelectedDataSourceChangedイベントを通知
            OnCheckedDataSourceChanged(EventArgs.Empty);
        }

        /// <summary>
        /// BindingSourceのListChangedイベントに登録するデリゲート
        /// </summary>
        void bindingSource_ListChanged(object sender, ListChangedEventArgs e)
        {
            if (checking)
            {
                //CheckedItemsBindableCheckedListBox_ItemCheck実行中は動作させない
                return;
            }
            listChanging = true;
            for (int i = 0; i < Items.Count; i++)
            {
                SelectableValue selectableValue = Items[i] as SelectableValue;
                SetItemChecked(i, IsChecked(selectableValue));
            }
            listChanging = false;
        }

        private bool IsChecked(SelectableValue selectableValue)
        {
            bool isChecked = false;
            //データソース内の画面データのリストを取得
            IList viewDataList = bindingSource.List;
            if (viewDataList != null)
            {
                ///画面データのリストに含まれるものを選択状態にする
                foreach (object viewData in viewDataList)
                {
                    ///SelectedDataMemberプロパティで指定した値を持つ
                    ///画面データプロパティの値と一致するものを選択状態にする
                    PropertyInfo property = viewData.GetType().GetProperty(StringValuePropertyName);
                    if (property != null && property.PropertyType.Equals(typeof(string)))
                    {
                        object value = property.GetValue(viewData, new object[0]);
                        if (value != null && string.Equals(selectableValue.StringValue, value.ToString(), StringComparison.Ordinal))
                        {
                            isChecked = true;
                        }
                    }
                }
            }
            return isChecked;
        }

        /// <summary>
        /// PropertyNameChangedパターン
        /// </summary>
        public event EventHandler CheckedDataSourceChange;

        protected virtual void OnCheckedDataSourceChanged(EventArgs e)
        {
            EventHandler handler = CheckedDataSourceChange;
            if (handler != null)
            {
                handler(this, e);
            }
        }


        /// <summary>
        /// 現状、使用していない
        /// </summary>
        [Editor("System.Windows.Forms.Design.DataMemberListEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(UITypeEditor))]
        public string CheckedDataMember { get; set; }
    }
}
