﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.ComponentModel;
using Terasoluna.Windows.Forms.Events;
using Terasoluna.Threading;
using Terasoluna.Logging;

namespace Terasoluna.TourSample.Client.Common.IO
{
    /// <summary>
    /// 進捗情報を通知する機能をもつストリームのサンプル実装
    /// </summary>
    public class NotifyProgressInfoFileStream : Stream
    {
        /// ログ
        private ILogger logger = LogManager.GetLogger();
        /// ファイル
        private readonly FileStream fileStream;
        /// 進捗情報管理オブジェクト
        private IEventProcProgressManager progressManager;
        //Disposeされたか
        private bool disposed;

        public NotifyProgressInfoFileStream(FileStream fileStream)
        {
            this.fileStream = fileStream;
            disposed = false;
            /// 進捗情報管理オブジェクトの取得            
            this.progressManager = InvocationScope.Current.GetContext<EventProcessContext>().ProgressManager;
        }

        public override bool CanRead
        {
            get { return fileStream.CanRead; }
        }

        public override bool CanSeek
        {
            get { return false; }
        }

        public override bool CanWrite
        {
            get { return false; }
        }

        public override void Flush() {
            fileStream.Flush();
        }

        public override long Length
        {
            get {return fileStream.Length;}
        }

        public override long Position
        {
            get { return fileStream.Position; }
            set { throw new NotSupportedException(); }
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            if (disposed)
            {
                throw new ObjectDisposedException("すでにファイルはDisposeされています");
            }
            int result = fileStream.Read(buffer, offset, count);
            int percentage = (int)(Position * 100 / Length);
            if (logger.IsTraceEnabled)
            {
                logger.Trace(string.Format("進捗率={0}%", percentage));
            }
            if (progressManager != null) 
            {
                ///進捗状況の通知
                ///送信率を0%～100%で通知する
                progressManager.SetProgress(new ProgressInfo(percentage));
            }
            return result;
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            throw new NotSupportedException();
        }

        public override void SetLength(long value)
        {
           throw new NotSupportedException();
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            throw new NotSupportedException();
        }

        /// Disposeパターン
        protected override void Dispose(bool disposing)
        {
            if (!disposed)
            {
                try
                {
                    if (disposing)
                    {
                        fileStream.Dispose();
                    }
                }
                finally
                {
                    base.Dispose(disposing);
                }
                disposed = true;
            }
        }
    }
}
