﻿// Copyright (c) 2011, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using Microsoft.Practices.Unity;
using System.Reflection;
using Terasoluna.Reflection;
using System.ServiceModel;
using Terasoluna.ServiceModel;

namespace Terasoluna.BizLogic
{
    /// <summary>
    /// <para>WCF の Proxy で通信するビジネスロジックを実行する機能を表します。</para>
    /// </summary>
    public class WcfProxyBizLogicExecutorRev : IBizLogicExecutor
    {
        /// <summary>
        /// <para><see cref="Terasoluna.BizLogic.IBizLogicInfoManager"/> のインスタンス名です。</para>
        /// </summary>
        /// <remarks>
        /// <para>この定数の値は "WcfProxy" です。</para>
        /// </remarks>
        public const string InstanceName = "WcfProxy";

        /// <summary>
        /// <para>ビジネスロジック情報管理機能のインスタンスを取得します。</para>
        /// </summary>
        /// <value>ビジネスロジック情報管理機能のインスタンス。</value>
        [Dependency(InstanceName)]
        public IBizLogicInfoManager InfoManager { get; set; }

        #region IBizLogicExecutor メンバ

        /// <summary>
        /// <para>ビジネスロジックを実行します。</para>
        /// </summary>
        /// <param name="info">ビジネスロジック情報。</param>
        /// <param name="input">ビジネスロジック実行時の入力情報。</param>
        /// <returns>ビジネスロジックの実行結果。</returns>
        /// <exception cref="System.ArgumentNullException">
        /// <list type="bullet">
        /// <item>引数 [<paramref name="info"/>] の値が <c>null</c> の場合。</item>
        /// <item>引数 [<paramref name="info"/>] が保持するプロパティ [<see cref="Terasoluna.BizLogic.BizLogicInfo.BizLogicType"/>] の値が <c>null</c> の場合。</item>
        /// <item>引数 [<paramref name="info"/>] が保持するプロパティ [<see cref="Terasoluna.BizLogic.BizLogicInfo.BizLogicName"/>] の値が <c>null</c> の場合。</item>
        /// <item>引数 [<paramref name="info"/>] が保持するプロパティ [<see cref="Terasoluna.BizLogic.BizLogicInfo.MethodName"/>] の値が <c>null</c> の場合。</item>
        /// </list>
        /// </exception>
        public object ExecuteBizLogic(BizLogicInfo info, object input)
        {
            ArgumentChecker.CheckNotNull("info", info);
            ArgumentChecker.CheckNotNull("info.BizLogicType", info.BizLogicType);
            ArgumentChecker.CheckNotNull("info.BizLogicName", info.BizLogicName);
            ArgumentChecker.CheckNotNull("info.MethodName", info.MethodName);

            object bizLogic = InfoManager.GetBizLogic(info);

            ICommunicationObject proxy = (ICommunicationObject)bizLogic;

            MethodInfo method = InfoManager.GetBizLogicMethod(info);

            object[] arguments = InfoManager.BuildArguments(info, input);

            object output = null;
            try
            {
                output = ReflectionUtility.InvokeMethod(proxy, method, arguments);
                proxy.Close();
            }
            catch (FaultException<TerasolunaSoapFault>)
            {
                try
                {
                    proxy.Close();
                }
                catch (Exception)
                {
                    proxy.Abort();
                }
                throw;
            }
            catch (Exception)
            {
                proxy.Abort();
                throw;
            }

            return output;
        }

        #endregion
    }
}
