﻿// <著作権>
// Copyright (c) 2007, NTT DATA Corporation.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using NTTDATA.TERASOLUNA.FW.Web;
using System.IO;
using NTTDATA.TERASOLUNA.FW.Web.Controller.Multipart;

namespace MultipartUploadUT
{
    ///<summary>
    /// MultipartItemFactory のテストクラスです。
    ///</summary>
    ///<author>
    /// 串野亮
    /// 菅井隆雄
    ///</author>
    [TestClass()]
    public class MultipartItemFactoryTest
    {

        private TestContext testContextInstance;

        // boundary
        private static readonly string BOUNDARY = "boundary";

        // エンコード: UTF-8
        private static readonly Encoding ENCODING = new UTF8Encoding();

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// MultipartItemFactory() のテスト
        ///</summary>
        ///<remarks>
        /// インスタンスが生成されることを確認するテスト。
        ///</remarks>
        public void ConstructorTest01()
        {
            // 準備
            MultipartItemFactory target = null;

            // 実行
            target = new MultipartItemFactory();

            // 検証
            Assert.IsNotNull(target);

        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamがnullの場合、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateItemTest01()
        {
            // 準備
            MultipartItemFactory target = new MultipartItemFactory();
            target.Boundary = BOUNDARY;
            target.Encoding = ENCODING;
            Stream partStream = null;

            try
            {
                // 失敗
                target.CreateItem(partStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException)
            {
                // 成功
            }
            catch (Exception ex)
            {
                // 失敗
                Assert.Fail("予期せぬエラーです。" + ex.ToString());
            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamが既にDisposeされている場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory.txt")]
        public void CreateItemTest02()
        {
            // 準備
            MultipartItemFactory target = new MultipartItemFactory();
            target.Boundary = BOUNDARY;
            target.Encoding = ENCODING;
            Stream partStream = new FileStream("MultipartItemFactory.txt", FileMode.Open, FileAccess.Read);
            partStream.Dispose();
            
            try
            {
                // 失敗
                target.CreateItem(partStream);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidRequestException)
            {
                // 成功
            }
            catch (Exception ex)
            {
                // 失敗
                Assert.Fail("予期せぬエラーです。" + ex.ToString());
            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamのヘッダーにおいて、キーと値の区切りがない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem03.txt")]
        public void CreateItemTest03()
        {
            // 準備
            MultipartItemFactory target = new MultipartItemFactory();
            target.Boundary = BOUNDARY;
            target.Encoding = ENCODING;
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem03.txt", FileMode.Open, FileAccess.Read))
            {
                try
                {
                    // 失敗
                    target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、区切り文字が見つからない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem04.txt")]
        public void CreateItemTest04()
        {
            // 準備
            MultipartItemFactory target = new MultipartItemFactory();
            target.Boundary = BOUNDARY;
            target.Encoding = ENCODING;
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem04.txt", FileMode.Open, FileAccess.Read))
            {

                try
                {
                    // 失敗
                    target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }
            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、終端文字列の後にデータが存在した場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem05.txt")]
        public void CreateItemTest05()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem05.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory05",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory05", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、終端文字列の後にデータが存在しない場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory.txt")]
        public void CreateItemTest06()
        {
            // 準備
            using (Stream partStream
                = new FileStream("MultipartItemFactory.txt",
                                 FileMode.Open,
                                 FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory", item.Name);
                Assert.AreEqual("MultipartItemFactory", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダーに"Content-Disposition"が存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem07.txt")]
        public void CreateItemTest07()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem07.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Disposition"(小文字)が定義されている場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem08.txt")]
        public void CreateItemTest08()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem08.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory08",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory08", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、
        /// ヘッダー"Content-Disposition"(小文字 + 大文字)が定義されている場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem09.txt")]
        public void CreateItemTest09()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem09.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory09",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory09", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、
        /// ヘッダー"Content-Disposition"(大文字)が定義されている場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem10.txt")]
        public void CreateItemTest10()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem10.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory10",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory10", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダーが"Conent-Disposition"を含む文字列の場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem11.txt")]
        public void CreateItemTest11()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem11.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"が存在しない場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem12.txt")]
        public void CreateItemTest12()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem12.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 実行
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"content-type"(小文字)の値が
        /// "application/octet-stream"(小文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem13.txt")]
        public void CreateItemTest13()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem13.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartFileItem item = result as MultipartFileItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory13;filename=MultipartItemFactory13.txt",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/octet-stream", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory13", item.Name);
                Assert.AreEqual("MultipartItemFactory13.txt", item.Filename);
                Assert.IsNotNull(item.OutputStream);
                Assert.IsFalse(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"content-type"(小文字)の値が
        /// "Application/Octet-Stream"(小文字+大文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem14.txt")]
        public void CreateItemTest14()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem14.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartFileItem item = result as MultipartFileItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory14;filename=MultipartItemFactory14.txt",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "Application/Octet-Stream", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory14", item.Name);
                Assert.AreEqual("MultipartItemFactory14.txt", item.Filename);
                Assert.IsNotNull(item.OutputStream);
                Assert.IsFalse(item.IsText);
            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"(小文字)の値が
        /// "application/octet-stream"(大文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem15.txt")]
        public void CreateItemTest15()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem15.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartFileItem item = result as MultipartFileItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory15;filename=MultipartItemFactory15.txt",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "APPLICATION/OCTET-STREAM", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory15", item.Name);
                Assert.AreEqual("MultipartItemFactory15.txt", item.Filename);
                Assert.IsNotNull(item.OutputStream);
                Assert.IsFalse(item.IsText);
            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"(小文字)の値が
        /// "application/octet-stream"を含む文字列の場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem16.txt")]
        public void CreateItemTest16()
        {
            // 準備
            using (Stream partStream
                = new FileStream("MultipartItemFactory_CreateItem16.txt",
                                 FileMode.Open,
                                 FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;
                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"(小文字)の値が
        /// "application/x-www-form-urlencoded"(小文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem17.txt")]
        public void CreateItemTest17()
        {
            // 準備
            using (Stream partStream
                = new FileStream("MultipartItemFactory_CreateItem17.txt",
                                 FileMode.Open,
                                 FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory17",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory17", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"content-type"(小文字)の値が
        /// "application/x-www-form-urlencoded"(小文字+大文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem18.txt")]
        public void CreateItemTest18()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem18.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory18",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "Application/X-Www-Form-UrlEncoded", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory18", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"content-type"(小文字)の値が
        /// "application/x-www-form-urlencoded"(大文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem19.txt")]
        public void CreateItemTest19()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem19.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory19",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "APPLICATION/X-WWW-FORM-URLENCODED", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory19", item.Name);
                Assert.AreEqual("あいうえお", item.Text);
                Assert.IsTrue(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"(小文字)の値が
        /// "application/x-www-form-urlencoded"を含む文字列の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem20.txt")]
        public void CreateItemTest20()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem20.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"content-type"(小文字)の値が
        /// "application/x-www-form-urlencoded"、"application/octet-stream"以外の文字列の場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem21.txt")]
        public void CreateItemTest21()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem21.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"(小文字+大文字)の値が
        /// "application/octet-stream"(小文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem22.txt")]
        public void CreateItemTest22()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem22.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartFileItem item = result as MultipartFileItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory22;filename=MultipartItemFactory22.txt",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/octet-stream", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory22", item.Name);
                Assert.AreEqual("MultipartItemFactory22.txt", item.Filename);
                Assert.IsFalse(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"(大文字)の値が
        /// "application/octet-stream"(小文字)の場合、
        /// 正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem23.txt")]
        public void CreateItemTest23()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem23.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartFileItem item = result as MultipartFileItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory23;filename=MultipartItemFactory23.txt",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/octet-stream", item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("MultipartItemFactory23", item.Name);
                Assert.AreEqual("MultipartItemFactory23.txt", item.Filename);
                Assert.IsFalse(item.IsText);

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダー"Content-Type"を含む文字列の値が
        /// "application/octet-stream"(小文字)の場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem24.txt")]
        public void CreateItemTest24()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem24.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダーに同じキーが複数存在する場合、
        /// 例外(InvalidRequestException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem25.txt")]
        public void CreateItemTest25()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem25.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                try
                {
                    // 失敗
                    IMultipartItem result = target.CreateItem(partStream);
                    Assert.Fail("到達してはいけないコードです。");
                }
                catch (InvalidRequestException)
                {
                    // 成功
                }
                catch (Exception ex)
                {
                    // 失敗
                    Assert.Fail("予期せぬエラーです。" + ex.ToString());
                }

            }
        }

        ///<summary>
        /// CreateItem (Stream) のテスト
        ///</summary>
        ///<remarks>
        /// 引数pastStreamで、ヘッダーに"Cotent-Disposition","Content-Type"以外のヘッダーも
        /// 存在する場合、正常に動作することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"MultipartUploadUT\Files\MultipartItemFactory_CreateItem26.txt")]
        public void CreateItemTest26()
        {
            // 準備
            using (Stream partStream = new FileStream(
                "MultipartItemFactory_CreateItem26.txt", FileMode.Open, FileAccess.Read))
            {
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = BOUNDARY;
                target.Encoding = ENCODING;

                // 実行
                IMultipartItem result = target.CreateItem(partStream);
                MultipartTextItem item = result as MultipartTextItem;

                // 検証
                Assert.IsNotNull(item);
                Assert.AreSame(ENCODING, item.Encoding);
                Assert.AreEqual(
                    "form-data; name=MultipartItemFactory26",
                    item.HeaderList["CONTENT-DISPOSITION"]);
                Assert.AreEqual(
                    "application/x-www-form-urlencoded",
                    item.HeaderList["CONTENT-TYPE"]);
                Assert.AreEqual("20", item.HeaderList["LENGTH"]);
                Assert.AreEqual("MultipartItemFactory26", item.Name);
                Assert.IsTrue(item.IsText);

            }
        }

         ///<summary>
        ///  Boundary のテスト
        ///</summary>
        ///<remarks>
        /// 引数Boundaryに、空文字列でない任意の文字列を設定し、
        /// 空文字列でない任意の文字列を取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void BoundaryTest01()
        {
            //準備
            MultipartItemFactory target = new MultipartItemFactory();
            string val = "boundary";
            target.Boundary= val;

            //検証
            Assert.AreEqual(val,target.Boundary);
        }


        ///<summary>
        ///  Boundary のテスト
        ///</summary>
        ///<remarks>
        /// 引数Boundaryにnullを設定した場合、
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void BoundaryTest02()
        {
        
            try
            {
                //失敗
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = null;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException)
            {
                //成功
            }
            catch(Exception e)
            {
                //失敗
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }
        }


        ///<summary>
        ///  Boundary のテスト
        ///</summary>
        ///<remarks>
        /// 引数Boundaryに空文字列を設定した場合、
        /// 例外(ArgumentException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void BoundaryTest03()
        {
            try
            {
                //失敗
                MultipartItemFactory target = new MultipartItemFactory();
                target.Boundary = "";
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException)
            {
                //成功
            }
            catch (Exception e)
            {
                //失敗
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }
        }


        ///<summary>
        ///  Encoding のテスト
        ///</summary>
        ///<remarks>
        /// Encodingで、not nullを設定し、
        /// Encodingでnot nullを取得することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EncodingTest01()
        {
            //準備
            MultipartItemFactory target = new MultipartItemFactory();
            Encoding val = new UTF8Encoding();
            target.Encoding = val;


            //検証
            Assert.AreEqual(val, target.Encoding);
        }


        ///<summary>
        /// Encoding のテスト
        ///</summary>
        ///<remarks>
        /// 引数Encodingで、nullを設定した場合
        /// 例外(ArgumentNullException)が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void EncodingTest02()
        {
            try
            {
                //失敗
                MultipartItemFactory target = new MultipartItemFactory();
                target.Encoding = null;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException)
            {
                //成功
            }
            catch (Exception e)
            {
                //失敗
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }
        }
    }
}
