﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TV = TERASOLUNA.Fw.Common.Validation;
using System.Configuration;
using System.IO;
using EL = Microsoft.Practices.EnterpriseLibrary.Validation;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.CommonUT.Validation;
using System.Data;
using System.Reflection;
using System.Security.AccessControl;
using TERASOLUNA.Fw.Common.Validation;
using TERASOLUNA.Fw.CommonUT.Validation.VabValidatorTestData;
using TERASOLUNA.Fw.CommonUT.Validation.VabValidatorTestData.xsd._001;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.CommonUT.Validation.VabValidatorTestData.xsd._002;


namespace TERASOLUNA.Fw.CommonUT.Validation{
    /// <summary>
    /// VabValidatorの単体試験コードです。
    /// </summary>
    [TestClass]
    public class VabValidatorTest
    {
        Guid id = Guid.NewGuid();

        #region 追加テスト属性
        //
        // テストを作成しているときに、次の追加属性を使用することができます:
        //
        // クラス内の最初のテストを実行する前にコードを実行するには、ClassInitialize を使用します
        // [ClassInitialize()]
        // public static void MyClassInitialize(TestContext testContext) { }
        //
        // クラス内のすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用します
        // [ClassCleanup()]
        // public static void MyClassCleanup() { }
        //
         //各テストを実行する前にコードを実行するには、TestInitialize を使用します 
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        
        // 各テストを実行した後にコードを実行するには、TestCleanup を使用します
        //[TestCleanup()]
        //public void MyTestCleanup() {
        //}
        
        #endregion

        #region テスト初期化

        /////<summary>
        ///// テスト初期化を行う
        /////</summary>
        [ClassInitialize()]
        public static void VabValidatorTestInitialize(TestContext testContext)
        {
            ConfigurationManager.AppSettings["ValidatorTypeName"] =
                "TERASOLUNA.Fw.Common.Validation.VabValidator,TERASOLUNA.Fw.Common";

        }
        #endregion

        /// <summary>
        /// ValidationFilePathのテスト
        /// </summary>
        /// <remarks>
        /// ValidationFilePathに null を設定した場合、
        /// エラーとならないことを確認するテスト。
        /// </remarks>
        [TestMethod]
        public void ValidationFilePathTest_01()
        {
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            try
            {
                validator.ValidationFilePath = null;
                Assert.AreEqual<string>(validator.ValidationFilePath, null);
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>
        /// ValidationFilePathのテスト
        /// </summary>
        /// <remarks>
        /// ValidationFilePath に空文字列を設定した場合、
        /// エラーとならないことを確認するテスト。
        /// </remarks>
        [TestMethod]
        public void ValidationFilePathTest_02()
        {
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            try
            {
                validator.ValidationFilePath = string.Empty;
                Assert.AreEqual<string>(string.Empty, validator.ValidationFilePath);
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>
        /// ValidationFilePathのテスト
        /// </summary>
        /// <remarks>
        /// ValidationFilePath にファイルシステムで許されない長い文字列を設定した場合、
        /// エラーとならないことを確認するテスト。
        /// </remarks>
        [TestMethod]
        public void ValidationFilePathTest_03()
        {
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            StringBuilder sb = new StringBuilder();
            sb.Append(@"c:\");
            for (int i = 0; i < 244; i++)
            {
                sb.Append("a");
            }

            sb.Append(@"\12345678.txt");
            string longFilePath = sb.ToString();

            try
            {
                validator.ValidationFilePath = longFilePath;
                Assert.AreEqual<string>(validator.ValidationFilePath, longFilePath);
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>
        /// ValidationFilePath のテスト
        /// </summary>
        /// <remarks>
        /// ValidationFilePath に正常値を設定した場合、
        /// 設定した値が取得できることを確認するテスト。
        /// </remarks>
        [TestMethod]
        public void ValidationFilePathTest_04()
        {
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();

            validator.ValidationFilePath = @"c:\test\file\path";

            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));
            Assert.AreEqual<string>(validator.ValidationFilePath, @"c:\test\file\path");

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>
        /// ValidationFilePath のテスト
        /// </summary>
        /// <remarks>
        /// ValidationFilePath の初期値が null であることを確認するテスト。
        /// </remarks>
        [TestMethod]
        public void ValidationFilePathTest_05()
        {
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();

            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));
            Assert.IsNull(validator.ValidationFilePath);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>
        /// RuleSet のテスト
        /// </summary>
        /// <remarks>
        /// RuleSet に設定したnull・空文字列でない文字列を、
        /// RuleSet で取得することができることを確認するテスト。
        /// </remarks>
        [TestMethod]
        public void RuleSetTest_01()
        {
            // 準備
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));
            TERASOLUNA_Fw_Common_Validation_VabValidatorAccessor accessor = new TERASOLUNA_Fw_Common_Validation_VabValidatorAccessor(validator as VabValidator);
            accessor._ruleSet = null;
            string expected = "TERASOLUNA";

            // 実施
            validator.RuleSet = expected;

            // 結果
            Assert.AreEqual(expected, validator.RuleSet);
        }
        
        /// <summary>Validate(DataSet dataSet)のテスト。引数がNull。</summary>
        /// <remarks>
        /// 引数にnullを渡した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        /// </remarks>        
        [TestMethod()]
        public void ValidateTest_01()
        {
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            try
            {
                TV.ValidationResult result = validator.Validate(null);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                //成功
                Assert.AreEqual("dataSet", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "dataSet"));  
        }

        /// <summary>Validate(DataSet dataSet)のテスト。各テーブルは0行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// </para>
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <para>
        ///  + Table1(VABValidatiorTestDataSet001.Table1Row) 
        /// </para>
        /// <para>
        /// 　- Column1:string 
        /// </para>
        /// <para>
        /// 　- Column2:int 
        /// </para>
        /// <para>
        ///  + Table2(VABValidatiorTestDataSet001.Table2Row) 
        /// </para>
        /// <para>
        ///    - Column1:string 
        /// </para>
        /// <para>
        ///    - Column2:date 
        /// </para>
        /// </item>
        /// <item>
        /// <description>全て一行もデータを持たない</description>
        /// </item>
        /// </list>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// <item>
        /// <description>実行されれば必ずエラーを返す。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config", @"input\DS01")]
        public void ValidateTest_02()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            Assert.AreEqual<int>(0, dataSet.Table1.Count);
            Assert.AreEqual<int>(0, dataSet.Table2.Count);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー有。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p> + Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て一行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、
        /// エラーとなる</description>
        /// </item>
        /// <item>
        /// <description>戻り値のIsValidはfalse</description>
        /// </item>
        /// <item>
        /// <description>MessageInfoは一件。ValidationMessageInfoのインスタンス</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoに格納されたメッセージは「VABValidatiorTestDataSet001_fail」である</description>
        /// </item>
        /// <item>
        /// <description>XPathはTable1[1]/Column1である</description>
        /// </item>
        /// <item>
        /// <description>keyには"Column1"が返る</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoの使用しない項目は初期値のままである</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config",
                        @"input\DS01")]
        public void ValidateTest_03()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(1, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message = (ValidationMessageInfo)results.Errors[0];

            // メッセージ内容が正しいか？
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", message.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message.ErrorPath);

            // メッセージで使用しない項目は初期値のままか？
            Assert.AreEqual("Column1", message.Key);
            Assert.IsNull(message.ErrorTitle);
            Assert.IsNull(message.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー有。3行。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p> + Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て3行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、
        /// エラーとなる。</description>
        /// </item>
        /// <item>
        /// <description>MessageInfoは3件。ValidationMessageInfoのインスタンス。</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoに格納されたメッセージは全て「VABValidatiorTestDataSet01_fail」</description>
        /// </item>
        /// <item>
        /// <description>XPathはTable1[1]/Column1～Table[3]/Column1である。</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoの使用しない項目は初期値のままである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config",
                        @"input\DS01")]
        public void ValidateTest_04()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1r1c1", 256);
            dataSet.Table1.AddTable1Row("t1r2c1", 256);
            dataSet.Table1.AddTable1Row("t1r3c1", 256);
            dataSet.Table2.AddTable2Row("t2r1c1", DateTime.Now);
            dataSet.Table2.AddTable2Row("t2r2c1", DateTime.Now);
            dataSet.Table2.AddTable2Row("t2r3c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(3, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message0 = (ValidationMessageInfo)results.Errors[0];
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", message0.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message0.ErrorPath);

            Assert.AreEqual("Column1", message0.Key);
            Assert.IsNull(message0.ErrorTitle);
            Assert.IsNull(message0.ErrorTypeName);

            Assert.IsInstanceOfType(results.Errors[1], typeof(ValidationMessageInfo));
            ValidationMessageInfo message1 = (ValidationMessageInfo)results.Errors[1];
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", message1.Message);
            Assert.AreEqual<string>("Table1[2]/Column1", message1.ErrorPath);

            Assert.AreEqual("Column1", message1.Key);
            Assert.IsNull(message1.ErrorTitle);
            Assert.IsNull(message1.ErrorTypeName);

            Assert.IsInstanceOfType(results.Errors[2], typeof(ValidationMessageInfo));
            ValidationMessageInfo message2 = (ValidationMessageInfo)results.Errors[2];
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", message2.Message);
            Assert.AreEqual<string>("Table1[3]/Column1", message2.ErrorPath);

            Assert.AreEqual("Column1", message2.Key);
            Assert.IsNull(message2.ErrorTitle);
            Assert.IsNull(message2.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証対象データがDBNull。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNull。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティがDBNull。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがArgumentExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config",
                        @"input\DS01")]
        public void ValidateTest_05()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();
            dataSet.Table1.AddTable1Row(null, 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                string message = MessageManager.GetMessage("E_STRONG_TYPING_EXCEPTION");
                Assert.AreEqual(message, e.Message);
                Assert.IsNotNull(e.InnerException);
                Assert.AreEqual<Type>(typeof(TargetInvocationException), e.InnerException.GetType());

                Assert.IsNotNull(e.InnerException.InnerException);
                Assert.AreEqual<Type>(typeof(StrongTypingException), e.InnerException.InnerException.GetType());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(TerasolunaException), "E_STRONG_TYPING_EXCEPTION"));
        }


        /// <summary>Validate(DataSet dataSet)のテスト。検証対象データがDBNull。設定ファイルに設定なし。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNull。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS05\VabValidationTest.config</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>検証対象となるプロパティがDBNull。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS05\VabValidationTest.config",
                        @"input\DS05")]
        public void ValidateTest_06()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS05\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();
            dataSet.Table1.AddTable1Row(null, 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。入力値検証設定ファイルがNull。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>ValidationFilePathにnullを設定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがInvalidOperationExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        public void ValidateTest_07()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));
            validator.ValidationFilePath = null;

            // 設定
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 成功
                string message = MessageManager.GetMessage("E_VALIDATION_FILE_UNDEFINED");
                Assert.AreEqual(message, e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(InvalidOperationException), "E_VALIDATION_FILE_UNDEFINED"));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。入力値検証設定ファイルが空文字。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>ValidationFilePathに空文字を設定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがInvalidOperationExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        public void ValidateTest_08()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));
            validator.ValidationFilePath = string.Empty;

            // 設定
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 成功
                string message = string.Format(MessageManager.GetMessage("E_VALIDATION_FILE_UNDEFINED"));
                Assert.AreEqual(message, e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(InvalidOperationException), "E_VALIDATION_FILE_UNDEFINED"));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。存在しない設定ファイルパス指定。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>ValidationFilePathに存在しないファイルパスを指定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがFileNotFoundExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        public void ValidateTest_09()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            string fileName = null;
            do
            {
                fileName = @"c:\" + System.Guid.NewGuid().ToString() + @"\dumyFile.xml";
            } while (File.Exists(fileName));

            validator.ValidationFilePath = fileName;

            // 設定
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FileNotFoundException e)
            {
                // 成功 
                Assert.AreEqual(fileName, e.FileName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }   
            
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(FileNotFoundException), "E_VALIDATION_CONFIG_FILE_NOT_FOUND", fileName));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。ファイル名に長すぎる文字列指定。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>ValidationFilePathに長すぎる文字列を指定。
        /// (実質的には「存在しないファイルパス」と同等だが、関数によっては別の例外を投げるので別途項目を切る)</description>
        /// </item>
        /// <item>
        /// <description>パス長248文字未満、フルパス長さ260文字未満、なので、c:\.....\(248文字) + 12345678.txtとする。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがFileNotFoundExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod]
        public void ValidateTest_10()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            StringBuilder sb = new StringBuilder();
            sb.Append(@"c:\");
            for (int i = 0; i < 244; i++)
            {
                sb.Append("a");
            }
            sb.Append(@"\12345678.txt");
            string longFilePath = sb.ToString();

            // 設定
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            validator.ValidationFilePath = longFilePath;

            try
            {
                TV.ValidationResult result = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (FileNotFoundException e)
            {
                //成功
                Assert.AreEqual(longFilePath, e.FileName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(FileNotFoundException), "E_VALIDATION_CONFIG_FILE_NOT_FOUND", longFilePath));
        
        }

        /// <summary>Validate(DataSet dataSet)のテスト。ファイルシステムが許容しないファイル名が指定された場合、入力値検証時にエラーとなることを確認します。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>"C:\temp\"ディレクトリに、"VabValidationTest.config"ファイルを配置する。</description>
        /// </item>
        /// <item>
        /// <description>実行ユーザのそのファイルに対する読み取り権限を削除する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        public void ValidateTest_11()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"C:\temp\VabValidationTest.config";
            validator.ValidationFilePath = filePath;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);
            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                ConfigurationErrorsException exception = e.InnerException as ConfigurationErrorsException;
                Assert.IsInstanceOfType(exception, typeof(ConfigurationErrorsException));
                Assert.AreEqual(filePath, exception.Filename);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(TerasolunaException), "E_VALIDATION_CONFIG_INFO_INVALID"));

        }

        /// <summary>Validate(DataSet dataSet)のテスト。不正な文字を含むValidationFilePath。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがデータを持つ。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>ValidationFilePathの値
        /// <code>"input\DS00\/-&lt;&gt;=!|~VabValidationTest.config"</code></description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがTerasolunaExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        public void ValidateTest_12()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"input\DS00\/-><=!|~VabValidationTest.config";
            validator.ValidationFilePath = filePath;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                Exception exception = e.InnerException;
                Assert.IsInstanceOfType(exception, typeof(ArgumentException));
                Assert.AreEqual("パスに無効な文字が含まれています。", exception.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(TerasolunaException), "E_INVALID_FILE_PATH", filePath));

        }

        /// <summary>Validate(DataSet dataSet)のテスト。RuleSetがNull。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>RuleSet に null を設定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する</description>
        /// </item>
        /// <item>
        /// <description>例外クラスが InvalidOperationException である。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS05\VabValidationTest.config",
                        @"input\DS05")]
        public void ValidateTest_13()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"input\DS05";
            validator.ValidationFilePath = filePath;
            validator.RuleSet = null;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 成功
                string message = string.Format(MessageManager.GetMessage("E_VALIDATION_RULESET_UNDEFINED"));
                Assert.AreEqual(message, e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(InvalidOperationException), "E_VALIDATION_RULESET_UNDEFINED"));
        }


        /// <summary>Validate(DataSet dataSet)のテスト。RuleSetが空文字列。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>RuleSet に空文字列を設定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する</description>
        /// </item>
        /// <item>
        /// <description>例外クラスが InvalidOperationException である。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        public void ValidateTest_14()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"input\DS05";
            validator.ValidationFilePath = filePath;
            validator.RuleSet = string.Empty;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 成功
                string message = string.Format(MessageManager.GetMessage("E_VALIDATION_RULESET_UNDEFINED"));
                Assert.AreEqual(message, e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(InvalidOperationException), "E_VALIDATION_RULESET_UNDEFINED"));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。RuleSetがnot null。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>RuleSet に not null を設定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS16\VabValidationTest.config",
                        @"input\DS16")]
        public void ValidateTest_15()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS16\VabValidationTest.config";
            validator.RuleSet = "ruleSet";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。RuleSetがnot null。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>なし</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>RuleSet に not null を設定。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、
        /// エラーとなる</description>
        /// </item>
        /// <item>
        /// <description>戻り値のIsValidはfalse</description>
        /// </item>
        /// <item>
        /// <description>MessageInfoは一件。ValidationMessageInfoのインスタンス</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoに格納されたメッセージは「VABValidatiorTestDataSet016_fail」である</description>
        /// </item>
        /// <item>
        /// <description>XPathはTable1[1]/Column1である</description>
        /// </item>
        /// <item>
        /// <description>keyには"Column1"が返る</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoの使用しない項目は初期値のままである</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS16\VabValidationTest.config",
                        @"input\DS16")]
        public void ValidateTest_16()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS16\VabValidationTest.config";
            validator.RuleSet = "TERASOLUNA";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(1, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message = (ValidationMessageInfo)results.Errors[0];

            // メッセージ内容が正しいか？
            Assert.AreEqual<string>("VABValidatiorTestDataSet016_fail", message.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message.ErrorPath);

            // メッセージで使用しない項目は初期値のままか？
            Assert.AreEqual("Column1", message.Key);
            Assert.IsNull(message.ErrorTitle);
            Assert.IsNull(message.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }
        /// <summary>Validate(DataSet dataSet)のテスト。入力値検証設定ファイルが不正な形式。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>使用する入力値検証設定ファイルはValidation\VabValidatorTestData\input\DS10\VabValidationTest.config</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>指定した入力値検証設定ファイルが存在するが形式が不正。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがTerasolunaExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS10\VabValidationTest.config",
                        @"input\DS10")]
        public void ValidateTest_17()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"input\DS10\VabValidationTest.config";
            validator.ValidationFilePath = filePath;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                string fullPath = Path.GetFullPath(filePath);
                ConfigurationErrorsException exception = e.InnerException as ConfigurationErrorsException;
                Assert.IsInstanceOfType(exception, typeof(ConfigurationErrorsException));
                Assert.AreEqual(exception.Filename, fullPath);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(TerasolunaException), "E_VALIDATION_CONFIG_INFO_INVALID"));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。入力値検証設定ファイルがXML形式ではない。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>使用する入力値検証設定ファイルはValidation\VabValidatorTestData\input\DS12\VabValidationTest.config</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>設定ファイルが不正な形式。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがTerasolunaExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS12\VabValidationTest.config",
                        @"input\DS12")]
        public void ValidateTest_18()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"input\DS12\VabValidationTest.config";
            validator.ValidationFilePath = filePath;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);
         
            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                string fullPath = Path.GetFullPath(filePath);
                ConfigurationErrorsException exception = e.InnerException as ConfigurationErrorsException;
                Assert.IsInstanceOfType(exception, typeof(ConfigurationErrorsException));
                Assert.AreEqual(exception.Filename, fullPath);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(TerasolunaException), "E_VALIDATION_CONFIG_INFO_INVALID"));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。対象型の記述無し。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS05\VabValidationTest.config</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>対応する型に対する検証ルールの記述がない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS05\VabValidationTest.config",
                        @"input\DS05")]
        public void ValidateTest_19()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS05\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証ルールなし。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS07\VabValidationTest.config</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>ルールセット"Default"が存在するが検証ルール内容が記述されていない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーとならない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS07\VabValidationTest.config",
                        @"input\DS07")]
        public void ValidateTest_20()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS07\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            Assert.AreEqual<int>(0, dataSet.Table1.Count);
            Assert.AreEqual<int>(0, dataSet.Table2.Count);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));

        }

        /// <summary>Validate(DataSet dataSet)のテスト。Validatorのロード失敗。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS13\VabValidationTest.config</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>指定したValidatorクラスが存在しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生すること。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスの型がTerasolunaExceptionであること。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS13\VabValidationTest.config",
                        @"input\DS13")]
        public void ValidateTest_21()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            string filePath = @"input\DS13\VabValidationTest.config";
            validator.ValidationFilePath = filePath;
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                string message = MessageManager.GetMessage("E_VALIDATION_CONFIG_INFO_INVALID");
                Assert.AreEqual(message, e.Message);
                Exception exception = e.InnerException;
                Assert.IsInstanceOfType(exception, typeof(ArgumentException));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(VabValidator), typeof(TerasolunaException), "E_VALIDATION_CONFIG_INFO_INVALID"));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。ルールセット内に検証対象カラム情報なし。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column3に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS15\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// <item>
        /// <description>決してエラーとならない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS15\VabValidationTest.config",
                        @"input\DS15")]
        public void ValidateTest_22()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS15\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }


        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー有。3カラム対象。1カラムエラー。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS14\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、エラーとなる。</description>
        /// </item>
        /// <item>
        /// <description>1件の検証エラーが発生する。</description>
        /// </item>
        /// <item>
        /// <description>メッセージは「VABValidatiorTestDataSet001_fail」である。</description>
        /// </item>
        /// <item>
        /// <description>XPathはTable1[1]/Column1である。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS14\VabValidationTest.config",
                        @"input\DS14")]
        public void ValidateTest_23()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS14\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(1, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message = (ValidationMessageInfo)results.Errors[0];

            // メッセージ内容が正しいか？
            Assert.AreEqual<string>("VABValidatiorTestDataSet001_t1c1_fail", message.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message.ErrorPath);

            // メッセージで使用しない項目は初期値のままか？
            Assert.AreEqual("Column1", message.Key);
            Assert.IsNull(message.ErrorTitle);
            Assert.IsNull(message.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。ValidatorからTestException。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがデータを持つ。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS09\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用する検証クラスはDummyValidator04_WithTestException</description>
        /// </item>
        /// <item>
        /// <description>使用するValidatorが例外をスロー。(TestException)</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>対応する検証ルールが存在する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがTestExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS09\VabValidationTest.config",
                        @"input\DS09")]
        public void ValidateTest_24()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS09\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            ValidationResult results = null;
            try
            {
                // 検証実行
                results = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TestException)
            {
                // 成功
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー無。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS04\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator02_WithNoError</description>
        /// </item>
        /// <item>
        /// <description>決してエラーとならない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS04\VabValidationTest.config",
                        @"input\DS04")]
        public void ValidateTest_25()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS04\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。3カラムでエラー。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS02\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(テーブル1のColumn1、Column2、テーブル2のColumn1)に対して検証を行い、正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが3件発生する。</description>
        /// </item>
        /// <item>
        /// <description>それぞれテーブル1のカラム1、カラム2、テーブル2のカラム1に対して発生する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS02\VabValidationTest.config",
                        @"input\DS02")]
        public void ValidateTest_26()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS02\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1r1c1", 256);
            dataSet.Table2.AddTable2Row("t2r1c1", DateTime.Now);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(3, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message0 = (ValidationMessageInfo)results.Errors[0];
            Assert.AreEqual<string>("VABValidatiorTestDataSet001_t1c1_fail", message0.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message0.ErrorPath);

            Assert.AreEqual("Column1", message0.Key);
            Assert.IsNull(message0.ErrorTitle);
            Assert.IsNull(message0.ErrorTypeName);

            Assert.IsInstanceOfType(results.Errors[1], typeof(ValidationMessageInfo));
            ValidationMessageInfo message1 = (ValidationMessageInfo)results.Errors[1];
            Assert.AreEqual<string>("VABValidatiorTestDataSet001_t1c2_fail", message1.Message);
            Assert.AreEqual<string>("Table1[1]/Column2", message1.ErrorPath);

            Assert.AreEqual("Column2", message1.Key);
            Assert.IsNull(message1.ErrorTitle);
            Assert.IsNull(message1.ErrorTypeName);

            Assert.IsInstanceOfType(results.Errors[2], typeof(ValidationMessageInfo));
            ValidationMessageInfo message2 = (ValidationMessageInfo)results.Errors[2];
            Assert.AreEqual<string>("VABValidatiorTestDataSet001_t2c2_fail", message2.Message);
            Assert.AreEqual<string>("Table2[1]/Column2", message2.ErrorPath);

            Assert.AreEqual("Column2", message2.Key);
            Assert.IsNull(message2.ErrorTitle);
            Assert.IsNull(message2.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。ネストされた検証結果。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>使用する入力値検証設定ファイルはValidation\VabValidatorTestData\input\DS08\VabValidationTest.config。</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator05_With1ErrorAnd1NegatedError</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>検証対象となる型とルールセットに対応した検証ルールが存在する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>入力値検証エラーが1件、テーブル1、カラム1に対して発生する。</description>
        /// </item>
        /// <item>
        /// <description>1件のネストしたエラーを含む。</description>
        /// </item>
        /// <item>
        /// <description>メッセージは「VABValidatiorTestDataSet01_fail」、子が「NestedResult01」</description>
        /// </item>
        /// <item>
        /// <description>エラーパスは、全て同一のもの、「Table1[1]/Column1」が格納される。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS08\VabValidationTest.config",
                        @"input\DS08")]
        public void ValidateTest_27()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS08\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            TV.ValidationResult result = validator.Validate(dataSet);

            Assert.IsNotNull(result);
            Assert.IsFalse(result.IsValid);
            Assert.IsNotNull(result.Errors);
            Assert.AreEqual<int>(2, result.Errors.Count);

            TV.ValidationMessageInfo messageInfo1 = (TV.ValidationMessageInfo)result.Errors[0];
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", messageInfo1.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", messageInfo1.ErrorPath);
            Assert.AreEqual("Column1", messageInfo1.Key);
            Assert.IsNull(messageInfo1.ErrorTitle);

            TV.ValidationMessageInfo messageInfo2 = (TV.ValidationMessageInfo)result.Errors[1];
            Assert.AreEqual<string>("NestedResult01", messageInfo2.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", messageInfo2.ErrorPath);
            Assert.AreEqual("Column1", messageInfo2.Key);
            Assert.IsNull(messageInfo2.ErrorTitle);
            
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。ネストされた検証結果。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>使用する入力値検証設定ファイルはValidation\VabValidatorTestData\input\DS11\VabValidationTest.config。</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator06_With1ErrorAnd3NegatedError</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>検証対象となる型とルールセットに対応した検証ルールが存在する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>入力値検証エラーが3件、テーブル1、カラム2に対して発生する。</description>
        /// </item>
        /// <item>
        /// <description>3件のネストしたエラーを含む。</description>
        /// </item>
        /// <item>
        /// <description>メッセージは親が「VABValidatiorTestDataSet001_fail」、子がそれぞれ「NestedResult01」～3</description>
        /// </item>
        /// <item>
        /// <description>エラーパスは、全て同一のもの、「Table1[1]/Column1」。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS11\VabValidationTest.config",
                        @"input\DS11")]
        public void ValidateTest_28()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS11\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            TV.ValidationResult result = validator.Validate(dataSet);

            Assert.IsNotNull(result);
            Assert.IsFalse(result.IsValid);
            Assert.IsNotNull(result.Errors);
            Assert.AreEqual<int>(4, result.Errors.Count);

            TV.ValidationMessageInfo messageInfo1 = (TV.ValidationMessageInfo)result.Errors[0];
            Assert.AreEqual<string>("VABValidatiorTestDataSet001_fail", messageInfo1.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", messageInfo1.ErrorPath);
            Assert.AreEqual("Column1", messageInfo1.Key);
            Assert.IsNull(messageInfo1.ErrorTitle);

            TV.ValidationMessageInfo messageInfo2 = (TV.ValidationMessageInfo)result.Errors[1];
            Assert.AreEqual<string>("NestedResult01", messageInfo2.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", messageInfo2.ErrorPath);
            Assert.AreEqual("Column1", messageInfo2.Key);
            Assert.IsNull(messageInfo2.ErrorTitle);

            TV.ValidationMessageInfo messageInfo3 = (TV.ValidationMessageInfo)result.Errors[2];
            Assert.AreEqual<string>("NestedResult02", messageInfo3.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", messageInfo3.ErrorPath);
            Assert.AreEqual("Column1", messageInfo3.Key);
            Assert.IsNull(messageInfo3.ErrorTitle);

            TV.ValidationMessageInfo messageInfo4 = (TV.ValidationMessageInfo)result.Errors[3];
            Assert.AreEqual<string>("NestedResult03", messageInfo4.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", messageInfo4.ErrorPath);
            Assert.AreEqual("Column1", messageInfo4.Key);
            Assert.IsNull(messageInfo4.ErrorTitle);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。DoValidateDataSetからTestException。各テーブルは1行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>使用する入力値検証設定ファイルはValidation\VabValidatorTestData\input\DS11\VabValidationTest.config。</description>
        /// </item>
        /// <item>
        /// <description>使用するValidatorはValidator06。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>検証対象となる型とルールセットに対応した検証ルールが存在する。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>例外が発生する。</description>
        /// </item>
        /// <item>
        /// <description>例外クラスがTestExceptionである。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS11\VabValidationTest.config",
                        @"input\DS11")]
        public void ValidateTest_29()
        {
            // 生成
            TV.IValidator validator = new TERASOLUNA.Fw.CommonUT.Validation.VabValidatorStub01();
            Assert.AreEqual<Type>(validator.GetType(), typeof(TERASOLUNA.Fw.CommonUT.Validation.VabValidatorStub01));

            // 設定
            validator.ValidationFilePath = @"input\DS11\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            TV.ValidationResult result = null;
            try
            {
                // 検証実行
                result = validator.Validate(dataSet);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TestException)
            {
                // 成功
            }
            
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckError(typeof(VabValidator), typeof(TestException), "種類 'TERASOLUNA.Fw.CommonUT.Validation.VabValidatorTestData.TestException' の例外がスローされました。"));
            
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー有。各テーブルは1行のデータを持つ。
        /// 各テーブルには、1行のデータが追加・削除され、未コミットである。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p> + Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て一行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>一行のデータが削除済みである。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、
        /// エラーとなる</description>
        /// </item>
        /// <item>
        /// <description>戻り値のIsValidはfalse</description>
        /// </item>
        /// <item>
        /// <description>MessageInfoは一件。ValidationMessageInfoのインスタンス</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoに格納されたメッセージは「VABValidatiorTestDataSet001_fail」である</description>
        /// </item>
        /// <item>
        /// <description>XPathはTable1[1]/Column1である</description>
        /// </item>
        /// <item>
        /// <description>keyには"Column1"が返る</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoの使用しない項目は初期値のままである</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config",
                        @"input\DS01")]
        public void ValidateTest_30()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 追加した後、削除する。
            dataSet.Table1.AddTable1Row("t1c1", 0);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            dataSet.AcceptChanges();

            dataSet.Table1.Rows[1].Delete();
            dataSet.Table2.Rows[1].Delete();

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(1, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message = (ValidationMessageInfo)results.Errors[0];

            // メッセージ内容が正しいか？
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", message.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message.ErrorPath);

            // メッセージで使用しない項目は初期値のままか？
            Assert.AreEqual("Column1", message.Key);
            Assert.IsNull(message.ErrorTitle);
            Assert.IsNull(message.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー無。各テーブルは1行のデータを持つ。
        /// 各テーブルには、1行のデータが追加・削除され、未コミットである。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS04\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator02_WithNoError</description>
        /// </item>
        /// <item>
        /// <description>決してエラーとならない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS04\VabValidationTest.config",
                        @"input\DS04")]
        public void ValidateTest_31()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS04\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 追加した後、削除する。
            dataSet.Table1.AddTable1Row("t1c1", 0);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            dataSet.AcceptChanges();

            dataSet.Table1.Rows[1].Delete();
            dataSet.Table2.Rows[1].Delete();

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー有。各テーブルは1行のデータを持つ。
        /// 各テーブルには、1行のデータが追加・削除され、コミットされている。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p> + Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て一行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>一行のデータが削除済みである。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、
        /// エラーとなる</description>
        /// </item>
        /// <item>
        /// <description>戻り値のIsValidはfalse</description>
        /// </item>
        /// <item>
        /// <description>MessageInfoは一件。ValidationMessageInfoのインスタンス</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoに格納されたメッセージは「VABValidatiorTestDataSet001_fail」である</description>
        /// </item>
        /// <item>
        /// <description>XPathはTable1[1]/Column1である</description>
        /// </item>
        /// <item>
        /// <description>keyには"Column1"が返る</description>
        /// </item>
        /// <item>
        /// <description>ValidationMessageInfoの使用しない項目は初期値のままである</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config",
                        @"input\DS01")]
        public void ValidateTest_32()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 追加した後、削除する。
            dataSet.Table1.AddTable1Row("t1c1", 0);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            dataSet.AcceptChanges();

            dataSet.Table1.Rows[1].Delete();
            dataSet.Table2.Rows[1].Delete();

            dataSet.AcceptChanges();

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsFalse(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(1, results.Errors.Count);

            // メッセージが正常に設定されているか？
            Assert.IsInstanceOfType(results.Errors[0], typeof(ValidationMessageInfo));
            ValidationMessageInfo message = (ValidationMessageInfo)results.Errors[0];

            // メッセージ内容が正しいか？
            Assert.AreEqual<string>("VABValidatiorTestDataSet01_fail", message.Message);
            Assert.AreEqual<string>("Table1[1]/Column1", message.ErrorPath);

            // メッセージで使用しない項目は初期値のままか？
            Assert.AreEqual("Column1", message.Key);
            Assert.IsNull(message.ErrorTitle);
            Assert.IsNull(message.ErrorTypeName);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。検証エラー無。各テーブルは1行のデータを持つ。
        /// 各テーブルには、1行のデータが追加・削除され、コミットされている。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <p>+ Table1(VABValidatiorTestDataSet001.Table1Row)</p>
        /// <p> 　- Column1:string</p>
        /// <p>   - Column2:int</p>
        /// <p> + Table2(VABValidatiorTestDataSet001.Table2Row)</p>
        /// <p>   - Column1:string</p>
        /// <p>   - Column2:date</p>
        /// </item>
        /// <item>
        /// <description>全て1行のデータを持つ</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがDBNullではない。</description>
        /// </item>
        /// <item>
        /// <description>全てのカラムがNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS04\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator02_WithNoError</description>
        /// </item>
        /// <item>
        /// <description>決してエラーとならない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 前提
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータセットに含まれるDataRowの型に対応した記述が設定ファイルに存在する。</description>
        /// </item>
        /// <item>
        /// <description>検証対象となるプロパティはDBNullではない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>引数に渡したデータテーブルの特定のプロパティ(Column1)に対して検証を行い、正常に終了する。</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS04\VabValidationTest.config",
                        @"input\DS04")]
        public void ValidateTest_33()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS04\VabValidationTest.config";
            VABValidatiorTestDataSet001 dataSet = new VABValidatiorTestDataSet001();

            // 検証対象データ
            dataSet.Table1.AddTable1Row("t1c1", 256);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            // 追加した後、削除する。
            dataSet.Table1.AddTable1Row("t1c1", 0);
            dataSet.Table2.AddTable2Row("t2c1", DateTime.Now);

            dataSet.AcceptChanges();

            dataSet.Table1.Rows[1].Delete();
            dataSet.Table2.Rows[1].Delete();

            dataSet.AcceptChanges();

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。各テーブルは0行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// </para>
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <para>
        ///  + Table1(VABValidatiorTestDataSet001.Table1Row) 
        /// </para>
        /// <para>
        /// 　- Column1:string 
        /// </para>
        /// <para>
        /// 　- Column2:int 
        /// </para>
        /// <para>
        ///  + Table2(VABValidatiorTestDataSet001.Table2Row) 
        /// </para>
        /// <para>
        ///    - Column1:string 
        /// </para>
        /// <para>
        ///    - Column2:date 
        /// </para>
        /// </item>
        /// <item>
        /// <description>全て一行もデータを持たない</description>
        /// </item>
        /// </list>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// <item>
        /// <description>実行されれば必ずエラーを返す。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config", @"input\DS01")]
        public void ValidateTest_34()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatorTestDataSet002 dataSet = new VABValidatorTestDataSet002();
            dataSet.BeginInit();

            // 検証対象データ
            Assert.AreEqual<int>(0, dataSet.Table1.Count);
            Assert.AreEqual<int>(0, dataSet.Table2.Count);

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }

        /// <summary>Validate(DataSet dataSet)のテスト。各テーブルは0行のデータを持つ。</summary>
        /// <remarks>
        /// <para>
        /// 検証対象
        /// </para>
        /// <list type="bullet">
        /// <item>
        /// <description>VABValidatiorTestDataSet001データセット</description>
        /// <para>
        ///  + Table1(VABValidatiorTestDataSet001.Table1Row) 
        /// </para>
        /// <para>
        /// 　- Column1:string 
        /// </para>
        /// <para>
        /// 　- Column2:int 
        /// </para>
        /// <para>
        ///  + Table2(VABValidatiorTestDataSet001.Table2Row) 
        /// </para>
        /// <para>
        ///    - Column1:string 
        /// </para>
        /// <para>
        ///    - Column2:date 
        /// </para>
        /// </item>
        /// <item>
        /// <description>全て一行もデータを持たない</description>
        /// </item>
        /// <item>
        /// <description>全てテーブルはBeginInitが実行されており、初期設定が完了していない</description>
        /// </item>
        /// </list>
        /// <para>
        /// 検証内容
        /// <list type="bullet">
        /// <item>
        /// <description>Table1.Column1に対して設定</description>
        /// </item>
        /// <item>
        /// <description>設定ファイルはValidation\VabValidatorTestData\input\DS01\VabValidationTest.config</description>
        /// </item>
        /// <item>
        /// <description>利用するValidatorはDummyValidator01_With1Error</description>
        /// </item>
        /// <item>
        /// <description>実行されれば必ずエラーを返す。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// 結果
        /// <list type="bullet">
        /// <item>
        /// <description>正常に終了する</description>
        /// </item>
        /// <item>
        /// <description>検証エラーが発生しない</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        [TestMethod()]
        [DeploymentItem(@"Validation\VabValidatorTestData\input\DS01\VabValidationTest.config", @"input\DS01")]
        public void ValidateTest_35()
        {
            // 生成
            TV.IValidator validator = TV.ValidatorFactory.CreateValidator();
            Assert.AreEqual<Type>(validator.GetType(), typeof(VabValidator));

            // 設定
            validator.ValidationFilePath = @"input\DS01\VabValidationTest.config";
            VABValidatorTestDataSet002 dataSet = new VABValidatorTestDataSet002();
            dataSet.BeginInit();

            // 検証対象データ
            Assert.AreEqual<int>(0, dataSet.Table1.Count);
            Assert.AreEqual<int>(0, dataSet.Table2.Count);
            dataSet.Table1.BeginInit();
            dataSet.Table2.BeginInit();

            // 検証実行
            ValidationResult results = validator.Validate(dataSet);

            // 結果オブジェクトが正常に返っているか？
            Assert.IsNotNull(results);
            Assert.IsTrue(results.IsValid);
            Assert.IsNotNull(results.Errors);
            Assert.AreEqual<int>(0, results.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(VabValidator), LogLevel.INFO));
        }
    }
}