﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Configuration.Conversion;
using System.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Configuration.Conversion
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ClearViewCollection のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ClearViewCollection 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ClearViewCollectionTest
    {
        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        ///<summary>
        /// ClearViewCollection () のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しのコンストラクタを呼び出した場合、インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            //実行
            ClearViewCollection target = new ClearViewCollection();

            //検証
            Assert.IsNotNull(target);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// CollectionType のテスト
        ///</summary>
        ///<remarks>
        /// CollectionTypeを取得した場合、値がBasicMapであることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CollectionTypeTest01()
        {
            //準備
            ClearViewCollection target = new ClearViewCollection();
            ConfigurationElementCollectionType expectedType = ConfigurationElementCollectionType.BasicMap;

            //実行
            ConfigurationElementCollectionType actualType = target.CollectionType;

            //検証
            Assert.AreEqual(expectedType, actualType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// IsReadOnly のテスト
        ///</summary>
        ///<remarks>
        /// IsReadOnlyを取得した場合、値がfalseであることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void IsReadOnlyTest01()
        {
            //準備
            ClearViewCollection target = new ClearViewCollection();

            //実行
            bool actual = target.IsReadOnly();

            //検証
            Assert.IsFalse(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ClearViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// 要素を追加した場合に例外が発生せず、正しく値が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest01()
        {
            //準備
            ClearViewCollection target = new ClearViewCollection();
            string expectedTableName = "tabel1";
            ClearViewElement expectedElement = new ClearViewElement(expectedTableName);
            int expectCount = 1;

            //実行
            target.Add(expectedElement);

            //検証
            Assert.AreEqual(expectCount, target.Count);
            Assert.AreEqual(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ClearViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// 要素を連続で追加した場合に例外が発生せず、FIFOで要素が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest02()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement expectedElement1 = new ClearViewElement("table1");
            ClearViewElement expectedElement2 = new ClearViewElement("table2");
            int expectedCount = 2;

            // 実行
            target.Add(expectedElement1);
            target.Add(expectedElement2);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreEqual(expectedElement1, target[0]);
            Assert.AreEqual(expectedElement2, target[1]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ClearViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// 同一要素を複数追加した場合、要素がひとつだけ追加されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest03()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement expectedElement = new ClearViewElement("table1");
            ClearViewElement duplicateElement = new ClearViewElement("table1");
            int expectedCount = 1;

            // 実行
            target.Add(expectedElement);
            target.Add(duplicateElement);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreEqual(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ClearViewElement) のテスト
        ///</summary>
        ///<remarks>
        /// nullを追加した場合、ArgumentNullExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest04()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();

            try
            {
                // 実行
                target.Add(null);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("element", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClearViewCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// 追加された要素が正しく返却できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemTest01()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement expectedElement = new ClearViewElement();
            int expectedCount = 1;

            // 実行
            target.Add(expectedElement);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        /// <remarks>
        ///　複数個追加された要素が正しく返却できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ItemTest02()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            string expectedTableName1 = "tabel1";
            string expectedTableName2 = "tabel2";
            string expectedTableName3 = "tabel3";
            ClearViewElement expectedElement1 = new ClearViewElement(expectedTableName1);
            ClearViewElement expectedElement2 = new ClearViewElement(expectedTableName2);
            ClearViewElement expectedElement3 = new ClearViewElement(expectedTableName3);
            int expectedCount = 3;

            // 実行
            target.Add(expectedElement1);
            target.Add(expectedElement2);
            target.Add(expectedElement3);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement1, target[0]);
            Assert.AreSame(expectedElement2, target[1]);
            Assert.AreSame(expectedElement3, target[2]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[string name] のテスト
        ///</summary>
        ///<remarks>
        /// 複数個追加された要素がキー指定で返却できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemTest03()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement expectedElement1 = new ClearViewElement("table1");
            ClearViewElement expectedElement2 = new ClearViewElement("table2");
            ClearViewElement expectedElement3 = new ClearViewElement("table3");
            int expectedCount = 3;
            target.Add(expectedElement1);
            target.Add(expectedElement2);
            target.Add(expectedElement3);

            // 実行
            object actual = target["table2"];
            int actualCount = target.Count;

            // 検証
            Assert.AreEqual(expectedCount, actualCount);
            Assert.AreSame(expectedElement2, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにnullを指定した場合にArgumentNullExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest04()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement element1 = new ClearViewElement("table1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[null];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (NullReferenceException)
            {
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスに-1を指定した場合にConfigurationErrorsExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest05()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement element1 = new ClearViewElement("table1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[-1];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにサイズより大きい値を指定した場合にConfigurationErrorsExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest06()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement element1 = new ClearViewElement("table1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[1];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[string name] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスに空文字を指定した場合にnullが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest07()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement element1 = new ClearViewElement("table1");
            target.Add(element1);

            // 実行
            object actual = target[string.Empty];

            // 検証
            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[string name] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにキーでない文字列を指定した場合にnullが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest08()
        {
            // 準備
            ClearViewCollection target = new ClearViewCollection();
            ClearViewElement element1 = new ClearViewElement("table1");
            target.Add(element1);

            // 実行
            object actual = target["terasoluna"];

            // 検証
            Assert.IsNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// CreateNewElement のテスト
        ///</summary>
        ///<remarks>
        /// CreateNewElement()を呼び出した場合、インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateNewElementTest01()
        {
            //準備
            ClearViewCollectionImpl target = new ClearViewCollectionImpl();

            //実行
            ConfigurationElement actual = target.CreateNewElement();

            //検証
            Assert.IsNotNull(actual);
            Assert.IsInstanceOfType(actual, typeof(ClearViewElement));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
        }

        ///<summary>
        /// GetElementKey のテスト
        ///</summary>
        ///<remarks>
        /// 引数がClearViewElementでないインスタンスの場合、
        /// ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest01()
        {
            //準備
            ClearViewCollectionImpl target = new ClearViewCollectionImpl();
            ConvertElement notClearViewElement = new ConvertElement();

            try
            {
                //実行
                target.GetElementKey(notClearViewElement);
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", notClearViewElement.GetType().FullName, typeof(ClearViewElement).FullName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClearViewCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClearViewCollection), typeof(ArgumentException), "E_INVALID_CAST", notClearViewElement.GetType().FullName, typeof(ClearViewElement).FullName));
        }

        ///<summary>
        /// ClearViewCollectionを継承するクラスです。
        ///</summary>
        class ClearViewCollectionImpl : ClearViewCollection
        {
            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new ConfigurationElement CreateNewElement()
            {
                return base.CreateNewElement();
            }

            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new string ElementName
            {
                get
                {
                    return base.ElementName;
                }
            }

            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new object GetElementKey(ConfigurationElement element)
            {
                return base.GetElementKey(element);
            }

            ///<summary>
            /// 基底クラスの同名のメソッドを呼び出します。
            ///</summary>
            public new void BaseAdd(ConfigurationElement element)
            {
                base.BaseAdd(element);
            }
        }
    }
}
