// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Conversion;
using System.Configuration;
using TERASOLUNA.Fw.CommonUT.Conversion.ConverterFactoryTestData;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;

namespace TERASOLUNA.Fw.ClientUT.Conversion
{
    /// <summary>
    /// ConverterFactoryのテストクラスです。
    /// </summary>
    [TestClass()]
    public class ConverterFactoryTest
    {
        private TestContext testContextInstance;

        private string _converterFactoryTypeName = "ConverterFactoryTypeName";

        private string _converterTypeName = "ConverterTypeName";

        /// <summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        /// </summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory = null;
            ConfigurationManager.AppSettings[_converterFactoryTypeName] = null;
            ConfigurationManager.AppSettings[_converterTypeName] = null;
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory = null;
            ConfigurationManager.AppSettings[_converterFactoryTypeName] = null;
            ConfigurationManager.AppSettings[_converterTypeName] = null;
        }
        //
        #endregion


        /// <summary>
        /// CreateConverter() のテスト
        /// </summary>
        /// <remarks>
        /// _factory が null, _converterType が null, 
        /// 構成ファイルに設定する ConverterFactoryTypeName と ConverterTypeName の記述がない場合、
        /// IConverter を実装した既定の DataSetConverter が返却されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateConverterTest01()
        {
            //実行
            IConverter target = ConverterFactory.CreateConverter();

            //検証
            Assert.IsInstanceOfType(target, typeof(DataSetConverter));
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory, typeof(ConverterFactory));

            ConverterFactory factory = TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory;
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor accessor= new TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor(factory);
            Assert.AreEqual(typeof(DataSetConverter), accessor._converterType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConverterFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateConverter() のテスト
        /// </summary>
        /// <remarks>
        /// _factory が not null, _converterType が not null, 
        /// 構成ファイルに設定する ConverterFactoryTypeName と ConverterTypeName の記述がないの場合、
        /// IConverter を実装した既定の DataSetConverter が返却されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateConverterTest02()
        {
            //準備
            ConverterFactory factory = new ConverterFactory();
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory = factory;
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor factoryAccessor = new TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor(factory);
            factoryAccessor._converterType = typeof(DataSetConverter);

            //実行
            IConverter target = ConverterFactory.CreateConverter();

            //検証
            Assert.IsInstanceOfType(target, typeof(DataSetConverter));
            Assert.AreSame(factory, TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory);

            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor accessor = new TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor(factory);
            Assert.AreSame(typeof(DataSetConverter), accessor._converterType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConverterFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateConverter() のテスト
        /// </summary>
        /// <remarks>
        /// 構成ファイルに設定する ConverterFactoryTypeName の記述があるが、
        /// ConverterFactory 派生クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateConverterTest03()
        {
            //準備
            ConfigurationManager.AppSettings[_converterFactoryTypeName] = typeof(ConverterFactory_Stub01).AssemblyQualifiedName;

            try
            {
                //実行
                IConverter target = ConverterFactory.CreateConverter();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                string exceptionMessage = MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION",
                    typeof(ConverterFactory_Stub01).AssemblyQualifiedName);
                Assert.AreEqual(exceptionMessage, e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConverterFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateConverter() のテスト
        /// </summary>
        /// <remarks>
        /// 構成ファイルに設定する ConverterFactoryTypeName と ConverterTypeName の記述があり、
        /// それぞれのクラスのインスタンスを生成できる場合、
        /// IConverter 実装クラスのインスタンスが返却されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateConverterTest04()
        {
            //準備
            ConfigurationManager.AppSettings[_converterFactoryTypeName] = typeof(ConverterFactory_Stub02).AssemblyQualifiedName;

            //実行
            IConverter target = ConverterFactory.CreateConverter();

            //検証
            Assert.IsInstanceOfType(target, typeof(ConverterFactory_IConverterImpl01));
            ConverterFactory factory = TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory;
            Assert.IsInstanceOfType(factory, typeof(ConverterFactory_Stub02));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConverterFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateConverter() のテスト
        /// </summary>
        /// <remarks>
        /// 構成ファイルに設定する ConverterFactoryTypeName の記述なし、
        /// ConverterTypeName の記述があるが IConverter 実装クラスのインスタンスを生成できない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateConverterTest05()
        {
            //準備
            ConfigurationManager.AppSettings[_converterTypeName] = typeof(ConverterFactory_IConverterImpl02).AssemblyQualifiedName; ;

            try
            {
                //実行
                IConverter target = ConverterFactory.CreateConverter();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 成功
                string exceptionMessage = MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION",
                    typeof(ConverterFactory_IConverterImpl02).AssemblyQualifiedName);
                Assert.AreEqual(exceptionMessage, e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConverterFactory), LogLevel.INFO));
        }

        /// <summary>
        /// CreateConverter() のテスト
        /// </summary>
        /// <remarks>
        /// 構成ファイルに設定する ConverterFactoryTypeName の記述なし、
        /// ConverterTypeName の記述があり IConverter 実装クラスのインスタンスを生成できる場合、
        /// IConverter 実装クラスのインスタンスが返却されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateConverterTest06()
        {
            //準備
            ConfigurationManager.AppSettings[_converterTypeName] = typeof(ConverterFactory_IConverterImpl01).AssemblyQualifiedName;

            //実行
            IConverter target = ConverterFactory.CreateConverter();

            //検証
            Assert.IsInstanceOfType(target, typeof(ConverterFactory_IConverterImpl01));
            Assert.IsInstanceOfType(TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory, typeof(ConverterFactory));
            ConverterFactory factory = TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor._factory;
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor accessor = new TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor(factory);
            Assert.AreEqual(typeof(ConverterFactory_IConverterImpl01), accessor._converterType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ConverterFactory), LogLevel.INFO));
        }

        /// <summary>
        /// ConverterType プロパティのテスト
        /// </summary>
        [TestMethod()]
        public void ConverterTypeTest01()
        {
            // 準備
            ConverterFactory factory = new ConverterFactory();
            TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor accessor =
                new TERASOLUNA_Fw_Client_Conversion_ConverterFactoryAccessor(factory);
            Type expected = typeof(Object);

            // 実施
            accessor.ConverterType = expected;
            Type actual = accessor.ConverterType;

            // 確認
            Assert.AreEqual(expected, actual);
        }
    }
}
