// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Net;
using System.Data;
using System.Configuration;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Client;
using TERASOLUNA.Fw.Client.Communication;
using TERASOLUNA.Fw.ClientUT.Communication;
using TERASOLUNA.Fw.ClientUT.Communication.CommunicatorBaseData;

namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.CommunicatorBase&lt;P, R&gt;テスト クラスです。
    ///</summary>
    [TestClass()]
    public class CommunicatorBaseTest
    {

        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            _baseUrl = ConfigurationManager.AppSettings["BaseUrl"];
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        [TestCleanup()]
        public void MyTestCleanup()
        {
            ConfigurationManager.AppSettings["BaseUrl"] = _baseUrl;
        }
        //
        #endregion

        #region フィールド変数

        /// <summary>
        /// target_ExecuteProgressChanged の引数です。
        /// </summary>
        private ExecuteProgressChangedEventArgs eventProgressChangedArgs = null;

        /// <summary>
        /// target_ExecuteProgressChanged の呼び出し回数カウント用引数です。
        /// </summary>
        private int count = 0;

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://localhost/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NORMAL_URL = ConfigurationManager.AppSettings["TestNormalUrl"];

        /// <summary>
        /// アプリケーション構成ファイルより取得するURLの初期値設定値です。
        /// </summary>
        private string _baseUrl = null;

        #endregion

        #region イベントハンドラ

        /// <summary>
        /// ReportProgressChangedTest で利用するイベントハンドラです。
        /// </summary>
        private void target_ExecuteProgressChanged(object sender, ExecuteProgressChangedEventArgs e)
        {
            eventProgressChangedArgs = e;
            count++;
        }

        #endregion


        ///<summary>
        /// CommunicateBase01 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="paramData"/> に <c>null</c> が設定されると 
        /// <exception cref="ArgumentNullException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest01()
        {
            // 準備
            DataSet paramData = null;
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeaders.Add("key1", "value1");            
            target.Address = null;            
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;

            try
            {
                // 確認
                target.Communicate(paramData,requestHeaders);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                //成功
                Assert.AreEqual("paramData", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "paramData"));  
        }


        ///<summary>
        /// CommunicateBase02 のテスト
        ///</summary>
        ///<remarks>
        /// ParamDataに非nullを設定した場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest02()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeaders.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            
            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeaders);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeaders, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase03 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="paramData"/> に <c>null</c> が設定されると 
        /// <exception cref="ArgumentNullException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest03()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = null;
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            target.Address = null;            
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            
            try
            {
                // 確認
                target.Communicate(paramData, requestHeaders);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                //成功
                Assert.AreEqual("requestHeaders", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "requestHeaders"));
        }


        ///<summary>
        /// CommunicateBase04 のテスト
        ///</summary>
        ///<remarks>
        /// requestHeadersに要素数を設定しなかった場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest04()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeaders);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeaders, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(0, senderStub._headerCollection.Count);


            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }

        ///<summary>
        /// CommunicateBase05テスト
        ///</summary>
        ///<remarks>
        /// requestHeadersに要素数3個を設定した場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest05()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();            

            requestHeader.Add("key1", "value1");
            requestHeader.Add("Key2", "Value2");
            requestHeader.Add("Key3", "Value3");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;

            target.Cancelled = false;
            
            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreSame(senderStub._request,receiverStub._request);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(paramData,senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(3, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }

        ///<summary>
        /// CommunicateBase06テスト
        ///</summary>
        ///<remarks>
        /// key値に空文字列が含まれる設定した場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest06()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add(string.Empty, "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase07テスト
        ///</summary>
        ///<remarks>
        /// value値にnullが設定されている場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest07()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", null);
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase08テスト
        ///</summary>
        ///<remarks>
        /// value値に空文字列を含む値が設定されている場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest08()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", string.Empty);
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase09テスト
        ///</summary>
        ///<remarks>
        /// 型パラメータ<TParam>に<string>が設定されている場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest09()
        {
            // 準備
            string paramData = "TERASOLUNA";
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl03 target = new CommunicatorBaseImpl03();
            CommunicatorBase_IHttpSenderStub02 senderStub = new CommunicatorBase_IHttpSenderStub02();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);
            
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));

        }


        ///<summary>
        /// CommunicateBase10 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="HttpSender"/> に <c>null</c> が設定されると 
        /// <exception cref="InvalidOperationException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest10()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", "value1");            
            target.Address = null;            
            target.HttpSender = null;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;

            try
            {
                // 実施
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 確認
                string message = MessageManager.GetMessage("E_SENDER_IS_NOT_HOLD");
                Assert.AreEqual(message, e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(InvalidOperationException), "E_SENDER_IS_NOT_HOLD"));
        }


        ///<summary>
        /// CommunicateBase11 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="Receiver"/> に <c>null</c> が設定されると 
        /// <exception cref="InvalidOperationException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest11()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
 
            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;            
            target.HttpReceiver = null;
            target.Cancelled = false;
            
            try
            {
                // 実施
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 確認
                string message = MessageManager.GetMessage("E_RECEIVER_IS_NOT_HOLD");
                Assert.AreEqual(message, e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(InvalidOperationException), "E_RECEIVER_IS_NOT_HOLD"));
        }


        ///<summary>
        /// CommunicateBase12テスト
        ///</summary>
        ///<remarks>
        /// <paramref name="Address"/> に <c>空文字列</c> が設定される場合に 
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest12()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            string Address = string.Empty;
            target.Address = Address;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            
            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase13 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="Address"/> に <c>不正なURL形式</c> が設定されると 
        /// <exception cref="TerasolunaException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest13()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", "value1");            
            string address = "http//terasoluna.co.jp";
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Address = address;
            target.Cancelled = false;          

            try
            {
                // 実施
                target.Communicate(paramData, requestHeader);                
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // 確認
                UriFormatException exception = e.InnerException as UriFormatException;
                Assert.IsInstanceOfType(exception, typeof(UriFormatException));
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_INVALID_URL",address), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(TerasolunaException), "E_COMMUNICATION_INVALID_URL", address));
        }


        ///<summary>
        /// CommunicateBase14 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="Address"/> に <c>HTTPスキームでないURL</c> が設定されると 
        /// <exception cref="TerasolunaException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest14()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            string address = "ftp://terasoluna.co.jp";
            target.Address = address;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;       

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                string message = MessageManager.GetMessage("E_COMMUNICATION_NOT_HTTP_URL", address);
                Assert.AreEqual(message, e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(TerasolunaException), "E_COMMUNICATION_NOT_HTTP_URL", address));
        }


        ///<summary>
        /// CommunicateBase15テスト
        ///</summary>
        /// <paramref name="Address"/> に <c>正常な形式URL</c> が設定される場合に 
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest15()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            string address = _baseUrl;
            target.Address = address;
            string expectedAddress = "http:terasoluna.ne.jp";
            ConfigurationManager.AppSettings["BaseUrl"] = expectedAddress;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;         

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(address, senderStub._request.RequestUri.ToString());
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);           
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase16テスト
        ///</summary>
        /// <paramref name="Cancelled"/> に <c>true</c> が設定される場合に 
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest16()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = true;
            

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase17 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="RequestTimeout"/> に <c>-1未満の値</c> が設定されると 
        /// <exception cref="ArgumentOutOfRangeException">が発生することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest17()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            string address = null;
            target.Address = address;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            target.RequestTimeout = -2222;

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentOutOfRangeException)
            {
                 //成功
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase18テスト
        ///</summary>
        ///<remarks>
        /// <paramref name="RequestTimeout"/> に <c>0</c> が設定されると 
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest18()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            target.RequestTimeout = 0;

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeader);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreEqual(0, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(_baseUrl, senderStub._request.RequestUri.ToString());
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeader, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase19 のテスト
        ///</summary>
        ///<remarks>
        /// 通信先のサーバでエラーが発生すると 
        /// <exception cref="ServerException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest19()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub02 receiverStub = new CommunicatorBase_IHttpReceiverStub02();

            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException)
            {
                //成功
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// CommunicateBase20 のテスト
        ///</summary>
        ///<remarks>
        /// 通信処理でキャンセルされた場合 
        /// <exception cref="CommunicationException">が発生することを確認するテスト。
        /// CommunicatorBase_IHttpSenderStub03で通信キャンセルの処理を行う。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest20()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub03 senderStub = new CommunicatorBase_IHttpSenderStub03();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");        
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            
            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {          
                //成功
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_EXCEPTION"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(CommunicationException), "E_COMMUNICATION_EXCEPTION"));
        }


        ///<summary>
        /// CommunicateBase21 のテスト
        ///</summary>
        ///<remarks>
        /// ConfigurationManager.AppSettings["BaseUrl"]からURLを取得できなかった場合
        /// <exception cref="TerasolunaException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest21()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            ConfigurationManager.AppSettings["BaseUrl"] = null;
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(typeof(TerasolunaException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_BASE_URL_NOT_FOUND"),e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(TerasolunaException), "E_COMMUNICATION_BASE_URL_NOT_FOUND"));
        }


        ///<summary>
        /// CommunicateBase22テスト
        ///</summary>
        ///<remarks>
        /// ConfigurationManager.AppSettings["BaseUrl"]からURLを取得できなかった場合
        /// <exception cref="TerasolunaException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest22()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            ConfigurationManager.AppSettings["BaseUrl"] = string.Empty;
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();
            
            requestHeader.Add("key1", "value1");
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreEqual(typeof(TerasolunaException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_BASE_URL_NOT_FOUND"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(TerasolunaException), "E_COMMUNICATION_BASE_URL_NOT_FOUND"));
        }


        ///<summary>
        /// CommunicateBase23テスト
        ///</summary>
        ///<remarks>
        /// ConfigurationManager.AppSettings["BaseUrl"]から不正なURLを取得した場合
        /// <exception cref="TerasolunaException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest23()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            string expectedAddress = "http:tearasoluna.ne.jp";
            ConfigurationManager.AppSettings["BaseUrl"] = expectedAddress;
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.IsInstanceOfType(e.InnerException, typeof(UriFormatException));
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_INVALID_URL",expectedAddress), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(TerasolunaException), "E_COMMUNICATION_INVALID_URL", expectedAddress));
       }


       ///<summary>
       /// CommunicateBase24テスト
       ///</summary>
       ///<remarks>
       /// ConfigurationManager.AppSettings["BaseUrl"]からHTTPスキームでないURLを取得した場合
       /// <exception cref="TerasolunaException">が発生することを確認するテスト。
       ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest24()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeader = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeader.Add("key1", "value1");
            string expectedAddress = "ftp://terasoluna.co.jp";
            ConfigurationManager.AppSettings["BaseUrl"] = expectedAddress;
            target.Address = null;
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;
            

            try
            {
                target.Communicate(paramData, requestHeader);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                string message = MessageManager.GetMessage("E_COMMUNICATION_NOT_HTTP_URL", expectedAddress);
                Assert.AreEqual(message, e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicatorBase<DataSet>), typeof(TerasolunaException), "E_COMMUNICATION_NOT_HTTP_URL", expectedAddress));
        }

        ///<summary>
        /// CommunicateBase25 のテスト
        ///</summary>
        ///<remarks>
        /// Urlにhttpsを設定した場合に
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void CommunicateBaseTest25()
        {
            // 準備
            DataSet paramData = new DataSet();
            IDictionary<string, string> requestHeaders = new Dictionary<string, string>();
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 senderStub = new CommunicatorBase_IHttpSenderStub01();
            CommunicatorBase_IHttpReceiverStub01 receiverStub = new CommunicatorBase_IHttpReceiverStub01();

            requestHeaders.Add("key1", "value1");
            target.Address = "https://terasoluna.co.jp/";
            target.HttpSender = senderStub;
            target.HttpReceiver = receiverStub;
            target.Cancelled = false;

            // 実施
            CommunicationResult actualCommunicationResult = target.Communicate(paramData, requestHeaders);

            // 確認
            Assert.AreSame(receiverStub._communicationResult, actualCommunicationResult);
            Assert.AreEqual(100000, senderStub._request.Timeout);
            Assert.IsFalse(senderStub._request.AllowAutoRedirect);
            Assert.AreEqual(@"https://terasoluna.co.jp/", senderStub._request.RequestUri.ToString());
            Assert.AreSame(senderStub._request, receiverStub._request);
            Assert.AreSame(paramData, senderStub._paramData);
            Assert.AreSame(requestHeaders, senderStub._headerCollection);
            Assert.AreSame(target, senderStub._reporter);
            Assert.AreSame(target, receiverStub._reporter);
            Assert.AreEqual(1, senderStub._headerCollection.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }

        ///<summary>
        /// Cancel01 のテスト
        ///</summary>
        ///<remarks>
        /// Cacelledプロパティにtrueが設定されていることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CancelTest01()
        {
            // 準備
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            HttpWebRequest request = null;
            bool val = false;

            PrivateObject privateObject = new PrivateObject(target, new PrivateType(typeof(CommunicatorBase<DataSet>)));
            privateObject.SetField("_request", request); 
            privateObject.SetField("_cancelled", val);

            // 実施
            target.Cancel();

            //検証
            Assert.AreEqual(target.Cancelled, val);
            
            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// Cancel02 のテスト
        ///</summary>
        ///<remarks>
        ///Cacelledプロパティにtrueが設定すると
        /// <exception cref="WebException">が発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CancelTest02()
        {
            //準備
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);
            bool val = true;

            PrivateObject privateObject = new PrivateObject(target, new PrivateType(typeof(CommunicatorBase<DataSet>)));
            privateObject.SetField("_request", request);
            privateObject.SetField("_cancelled", val);

            target.Cancel();

            //検証
            try
            {
                request.GetResponse();
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException e)
            {
                Assert.AreEqual(WebExceptionStatus.RequestCanceled, e.Status);
            }
            Assert.AreEqual(target.Cancelled, val);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }

        ///<summary>
        /// ReportProgressChanged01 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="ExecuteProgressChangedEventArgs"/> に <c>null</c> が設定される場合 
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void ReportProgressChanged01()
        {
            //準備
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            ExecuteProgressChangedEventArgs e = null;
            target.ProgressChanged += new ExecuteProgressChangedEventHandler(target_ExecuteProgressChanged);
       
            //実施
            target.ReportProgressChanged(e);
            
            //検証
            Assert.IsNull(eventProgressChangedArgs);
            Assert.AreEqual(1, count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        ///<summary>
        /// ReportProgressChanged02 のテスト
        ///</summary>
        ///<remarks>
        /// <paramref name="ExecuteProgressChangedEventArgs"/> に <c>not null</c> が設定される場合 
        /// 正常に動作することを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void ReportProgressChanged02()
        {
            //準備
            int expectedProgressPercentage = 1;
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            ExecuteProgressChangedEventArgs val = new ExecuteProgressChangedEventArgs(expectedProgressPercentage);
            target.ProgressChanged += new ExecuteProgressChangedEventHandler(target_ExecuteProgressChanged);
            
            // 実施
            target.ReportProgressChanged(val);

            //検証
            Assert.AreEqual(expectedProgressPercentage, eventProgressChangedArgs.ProgressPercentage);
            Assert.AreEqual(1, count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        /// <summary>
        /// Addressプロパティ のテスト
        ///</summary>
        ///<remarks>
        /// _addressが返却されることを確認するテスト
        /// </remarks>
        [TestMethod()]
        public void AddressTest()
        {
            //準備
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            string expectedAddress = "http://terasoluna.co.jp";

            //実行
            target.Address = expectedAddress;
            string actualAddress = target.Address;

            //検証
            Assert.AreEqual(expectedAddress, actualAddress);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// RequestTimeOutプロパティ のテスト
        ///</summary>
        ///<remarks>
        /// _requestTimeOutが返却されることを確認するテスト
        /// </remarks>
        [TestMethod()]
        public void RequestTimeOutTest()
        {
            //準備
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            int expectedRequestTimeOut = 50;

            //実行
            target.RequestTimeout = expectedRequestTimeOut;
            int actualRequestTimeOut = target.RequestTimeout;

            //検証
            Assert.AreEqual(expectedRequestTimeOut, actualRequestTimeOut);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        /// <summary>
        /// Cancelledプロパティ のテスト
        ///</summary>
        ///<remarks>
        /// _cancelledが返却されることを確認するテスト
        /// </remarks>
        [TestMethod()]
        public void CancelledTest()
        {
            //準備
            CommunicatorBaseImpl02 target = new CommunicatorBaseImpl02();
            bool expectedCancelled = false;

            //実行
            target.Cancelled = expectedCancelled;
            bool actualCancelled = target.Cancelled;

            //検証
            Assert.AreEqual(expectedCancelled, actualCancelled);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Senderプロパティ のテスト
        ///</summary>
        ///<remarks>
        /// _senderが返却されることを確認するテスト
        /// </remarks>
        [TestMethod()]
        public void SenderTest()
        {
            //準備
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpSenderStub01 expectedSenderStub = new CommunicatorBase_IHttpSenderStub01();

            //実行
            target.HttpSender = expectedSenderStub;
            IHttpSender<DataSet> actualSender = target.HttpSender;

            //検証
            Assert.AreEqual(expectedSenderStub, actualSender);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }


        /// <summary>
        /// Receiverプロパティ のテスト
        ///</summary>
        ///<remarks>
        /// _cancelledが返却されることを確認するテスト
        /// </remarks>
        [TestMethod()]
        public void ReceiverTest()
        {
            //準備
            CommunicatorBaseImpl01 target = new CommunicatorBaseImpl01();
            CommunicatorBase_IHttpReceiverStub01 expectedreceiverStub = new CommunicatorBase_IHttpReceiverStub01();

            //実行
            target.HttpReceiver = expectedreceiverStub;
            IHttpReceiver actualReceiver = target.HttpReceiver;

            //検証
            Assert.AreEqual(expectedreceiverStub, actualReceiver);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicatorBase<DataSet>), LogLevel.INFO));
        }
    }
}
