﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections;
using System.Data;
using System.Windows.Forms;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Forms
{
    /// <summary>
    /// <see cref="FormForwarder.FormClosing"/> で用いられる <see cref="EventArgs"/> の派生クラスです。
    /// </summary>
    /// <remarks>
    /// 遷移先の画面の FormClosing イベント、 FormClosed イベントの情報を遷移元に伝達します。
    /// </remarks>
    public class ForwardableFormCloseEventArgs : EventArgs
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ForwardableFormCloseEventArgs));

        /// <summary>
        /// オリジナルの <see cref="EventArgs"/> のインスタンスを保持します。
        /// </summary>
        private EventArgs _innerEventArgs = null;

        /// <summary>
        /// ダイアログボックスとしての実行結果を示す <see cref="DialogResult"/> の
        /// 値を保持します。
        /// </summary>
        private DialogResult _dialogResult = DialogResult.None;

        /// <summary>
        /// イベント発生元画面の持つコレクションを保持します。
        /// </summary>
        private IDictionary _items = null;

        /// <summary>
        /// イベント発生元画面の画面 <see cref="DataSet"/> を保持します。
        /// </summary>
        private DataSet _viewData = null;

        /// <summary>
        /// <see cref="ForwardableFormCloseEventArgs"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// <para>
        /// <paramref name="IForwardable"/> の値を取り出してプロパティに格納します。
        /// </para>
        /// </remarks>
        /// <param name="forwardable">FormClosing イベント、 FormClosed イベントを発生させた元の画面クラスのインスタンス。</param>
        /// <param name="innerEventArgs">元のイベントから渡されたイベント情報。</param>
        /// <exception cref="TerasolunaException">
        /// <paramref name="forwardable"/> が <see cref="Form"/> を継承していません。
        /// </exception>
        public ForwardableFormCloseEventArgs(IForwardable forwardable, EventArgs innerEventArgs)
            : base()
        {
            if (forwardable != null)
            {
                _items = forwardable.Items;
                _viewData = forwardable.ViewData;

                Form form = forwardable as Form;
                if (form == null)
                {
                    string message = string.Format(
                        Properties.Resources.E_INVALID_CAST, forwardable.GetType().FullName, typeof(Form).FullName);
                    TerasolunaException exception = new TerasolunaException(message);
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(message, exception);
                    }
                    throw exception;
                }

                _dialogResult = form.DialogResult;
            }

            _innerEventArgs = innerEventArgs;
        }

        /// <summary>
        /// フォームのダイアログ結果を取得します。
        /// </summary>
        /// <value>
        /// フォームのダイアログ結果。
        /// </value>
        public DialogResult DialogResult
        {
            get
            {
                return _dialogResult;
            }
        }

        /// <summary>
        /// イベント発生元画面の持つコレクションを取得します。
        /// </summary>
        /// <value>
        /// イベント発生元画面の持つコレクション。
        /// </value>
        public IDictionary Items
        {
            get
            {
                return _items;
            }
        }

        /// <summary>
        /// イベント発生元が発行する <see cref="EventArgs"/> を取得します。
        /// </summary>
        /// <value>
        /// イベント発生元が発行する <see cref="EventArgs"/> 。
        /// </value>
        public EventArgs InnerEventArgs
        {
            get
            {
                return _innerEventArgs;
            }
        }

        /// <summary>
        /// イベント発生元画面の画面 <see cref="DataSet"/> を取得します。
        /// </summary>
        /// <value>
        /// イベント発生元画面の画面 <see cref="DataSet"/> 。
        /// </value>
        public DataSet ViewData
        {
            get
            {
                return _viewData;
            }
        }
    }
}
