﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License. 

using System;
using System.IO;
using System.Text;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// ファイルをマルチパートデータとして格納するためのデータ型です。
    /// </summary>
    public class MultipartFileElement : MultipartElement
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(MultipartFileElement));

        /// <summary>
        /// マルチパートデータのヘッダにContent-Typeの値として利用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>定数の値は "application/octet-stream" です。</para>
        /// </remarks>
        protected static readonly string CONTENT_TYPE_VALUE = "application/octet-stream";

        /// <summary>
        /// マルチパートデータのヘッダにContent-Dispositionの値を設定する際に使用する文字列です。
        /// </summary>
        /// <remarks>
        /// <para>定数の値は "form-data; name={0}; filename={1}" です。</para>
        /// </remarks>
        protected static readonly string FORM_DATA_FORMAT = "form-data; name={0}; filename={1}";

        /// <summary>
        /// アップロード対象のファイルのファイルパスを格納するフィールドです。
        /// </summary>
        private string _uploadFilePath = null;

        /// <summary>
        /// <see cref="MultipartElement"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// <para><paramref name="name"/>を引数とするコンストラクタです。</para>
        /// </remarks>
        /// <param name="name">マルチパートの要素名です。</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="name"/>がnull参照です。
        /// </exception>
        /// <exception cref="ArgumentException">
        /// <paramref name="name"/>が空文字です。
        /// </exception>
        public MultipartFileElement(string name)
            : base(name)
        {
        }

        /// <summary>
        /// アップロード対象のファイルのファイルパスを取得または設定します。
        /// </summary>
        /// <value>
        /// アップロード対象のファイルのファイルパスです。
        /// </value>
        /// <exception cref="ArgumentNullException">
        /// ファイルパスが <c>null</c> 参照です。
        /// </exception>
        /// <exception cref="ArgumentException">
        /// ファイルパスが空文字です。
        /// </exception>
        public string UploadFilePath
        {
            get
            {
                return _uploadFilePath;
            }
            set
            {
                if (value == null)
                {
                    ArgumentNullException exception = new ArgumentNullException("UploadFilePath");
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(string.Format(Properties.Resources.E_NULL_PROPERTY_VALUE, "UploadFilePath"), exception);
                    }
                    throw exception;
                }

                if (value.Length == 0)
                {
                    string message = string.Format(Properties.Resources.E_EMPTY_PROPERTY_VALUE, "UploadFilePath");
                    ArgumentException exception = new ArgumentException(message);
                    if (_log.IsErrorEnabled)
                    {
                        _log.Error(message, exception);
                    }
                    throw exception;
                }

                _uploadFilePath = value;
            }
        }

        /// <summary>
        /// ファイルデータを読み出すための<see cref="Stream"/>を返却します。
        /// <paramref name="encoding"/> は使用しません。
        /// </summary>
        /// <returns>ファイルデータを読み出すための<see cref="Stream"/>です。</returns>
        /// <param name="encoding">本メソッドでは、使用しません。</param>
        /// <exception cref="InvalidOperationException">
        /// <see cref="UploadFilePath"/> が <c>null</c> 参照です。
        /// </exception>
        /// <exception cref="FileNotFoundException">
        /// 指定したファイルが存在しません。
        /// </exception>
        /// <exception cref="UnauthorizedAccessException">
        /// ファイルにアクセスできません。
        /// </exception>
        public override Stream CreateBodyStream(Encoding encoding)
        {
            if (this.UploadFilePath == null)
            {
                InvalidOperationException exception = 
                    new InvalidOperationException(Properties.Resources.E_COMMUNICATION_NO_FILENAME);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;

            }

            // ファイル存在チェック
            if (!File.Exists(this.UploadFilePath))
            {
                FileNotFoundException exception = new FileNotFoundException(string.Format(Properties.Resources.E_COMMUNICATION_FILE_NOT_FOUND, this.UploadFilePath), this.UploadFilePath);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }

            return new FileStream(this.UploadFilePath, FileMode.Open, FileAccess.Read);
        }

        /// <summary>
        /// マルチパートデータのヘッダ情報を HTTP ヘッダ形式に整形して返却します。
        /// </summary>
        /// <remarks>
        /// <para>ヘッダには "Content-Disposition" 、 "Content-Type" が含まれます。</para>
        /// <para>Content-Disposition ヘッダに格納される値は、 "form-data; name=[マルチパート要素名];filename=[ファイルパス]"
        /// となります。マルチパート要素名は、コンストラクタで与えられた文字列です。ファイルパスは
        /// <see cref="MultipartFileElement.UploadFilePath"/> で指定された文字列です。</para>
        /// <para>Content-Type ヘッダに格納される値は、 "application/octet-stream" となります。</para>
        /// <para>サーバ・アプリケーションで Commons FileUpload ライブラリ使用する場合には、使用するバージョンによっていくつかの
        /// 制約があります。</para>
        /// <para>バージョン 1.1.1 では、マルチパートデータのヘッダに以下の制約があります。</para>
        /// <para>
        /// <list type="bullet">
        /// <item>
        /// <description>ヘッダは一行(キー / 値)で1024バイトまでです。</description>
        /// </item>
        /// <item>
        /// <description>ヘッダ部は最大 10240 バイトまでです。</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>バージョン 1.2 では、マルチパートデータのヘッダに以下の制約があります。</para>
        /// <para>
        /// <list type="bullet">
        /// <item>
        /// <description>ヘッダ部は最大 10240 バイトまでです。</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        /// <returns>マルチパートデータのヘッダとなる文字列です。</returns>
        public override string CreateHeader()
        {
            StringBuilder sb = new StringBuilder();

            string headerFormat = HEADER_FORMAT_STRING;

            sb.Append(string.Format(headerFormat, 
                                    CONTENT_DISPOSITION,
                                    string.Format(FORM_DATA_FORMAT, Name, this.UploadFilePath)));
            sb.Append(NEWLINE_STRING);
            sb.Append(string.Format(headerFormat, 
                                    CONTENT_TYPE,
                                    CONTENT_TYPE_VALUE));

            return sb.ToString();
        }
    }
}
