﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections.Generic;
using TERASOLUNA.Fw.Common;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// 通信機能を提供するインターフェイスです。
    /// </summary>
    /// <remarks>
    /// <see cref="ICommunicator{Tparam}"/> 実装クラスは、
    /// <see cref="ICommunicator{Tparam}.Communicate"/> メソッドを実装します。
    /// </remarks>
    /// <typeparam name="TParam">サーバに送信するデータオブジェクトの型。</typeparam>
    public interface ICommunicator<TParam> : ICancelable, IProgressChangedEventInvoker
        where TParam : class
    {
        /// <summary>
        /// 通信先のURLを取得または設定します。
        /// </summary>
        /// <value>
        /// 通信先のURL。
        /// </value>
        string Address
        {
            get;
            set;
        }

        /// <summary>
        /// リクエストタイムアウト時間(ミリ秒)を取得または設定します。
        /// </summary>
        /// <value>
        /// リクエストタイムアウト時間(ミリ秒)。
        /// </value>
        int RequestTimeout
        {
            get;
            set;
        }

        /// <summary>
        /// 通信処理を実行します。このメソッドの戻り値が null 参照となることはありません。
        /// </summary>
        /// <remarks>
        /// <see cref="ICommunicator{Tparam}.Communicate"/> は、<see cref="Address"/> プロパティに設定された URL に対して、
        /// 渡されたパラメータに基づき通信処理を行います。
        /// </remarks>
        /// <param name="paramData">送信データを格納したオブジェクト。</param>
        /// <param name="requestHeaders">送信時に HTTP ヘッダとして利用するヘッダ文字列のコレクション。</param>
        /// <returns>通信処理結果を格納した <see cref="CommunicationResult"/>。</returns>
        /// <exception cref="ArgumentNullException">
        /// 以下のような場合に例外をスローします。
        /// <list type="bullet">
        /// <item>
        /// <paramref name="paramData"/> が null 参照です。
        /// </item>
        /// <item>
        /// <paramref name="requestHeaders"/> が null 参照です。
        /// </item>
        /// </list>
        /// </exception>
        /// <exception cref="ArgumentException">
        /// 通信処理で必要なデータが不正です。
        /// </exception>
        /// <exception cref="InvalidOperationException">
        /// 以下のような場合に例外をスローします。
        /// <list type="bullet">
        /// <item><see cref="CommunicatorBase{TParam}.Sender"/> が null 参照です。</item>
        /// <item><see cref="CommunicatorBase{TParam}.Receiver"/> が null 参照です。</item>
        /// </list>
        /// </exception>
        /// <exception cref="ServerException">
        /// 通信先のサーバでエラーが発生しました。
        /// </exception>
        /// <exception cref="CommunicationException">
        /// サーバとの通信中にエラーが発生しました。または通信がキャンセルされました。
        /// </exception>
        /// <exception cref="TerasolunaException">
        /// 通信先 URL の設定が不正です。
        /// </exception>
        CommunicationResult Communicate(TParam paramData, IDictionary<string, string> requestHeaders);
    }
}

