// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Net;
using System.Xml;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common.Validation;
using System.Text;
using System.Net.Mime;
using System.Threading;

namespace TERASOLUNA.Fw.Client.Communication
{
    /// <summary>
    /// f[^M@\񋟂 <see cref="IHttpReceiver"/> ̊NXłB
    /// </summary>
    /// <remarks>
    /// <para>
    /// ̃NX͒ۃNXłB{Iȏt[ƒ萔AHTTP X|X̃GeBeB{fB
    /// XML`Ŏ擾郁\bhƁATERASOLUNA Server Framework ɑΉG[̎񋟂
    /// ܂B
    /// </para>
    /// </remarks>
    public abstract class ReceiverBase : IHttpReceiver
    {
        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ReceiverBase));

        /// <summary>
        /// ʐMf[^ XML dł邱Ƃ\łB
        /// </summary>
        protected static readonly string TEXT_XML = "text/xml";

        /// <summary>
        /// G[ӏ\ XPath ێXMLm[h̃m[h\łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "error-field" łB
        /// </remarks>
        protected static readonly string ERROR_FIELD = "error-field";

        /// <summary>
        /// T[oŔG[ʂ擾ۂɃL[Ƃėp镶łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "exception" łB
        /// </remarks>
        protected static readonly string ERROR_TYPE = "exception";

        /// <summary>
        /// ʐMɃT[oz肵ȂOƂʕłB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "serverException" łB
        /// </remarks>
        public static readonly string SERVER_EXCEPTION = "serverException";

        /// <summary>
        /// ʐMɃT[o̓`FbNG[ƂʕłB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "serverValidateException" łB
        /// </remarks>
        public static readonly string SERVER_VALIDATE_EXCEPTION = "serverValidateException";

        /// <summary>
        /// ʐMɃT[õVXeG[܂ނ̑̃G[ʃR[h\łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "exception" łB
        /// </remarks>
        protected static readonly string EXCEPTION = "exception";

        /// <summary>
        /// ʐMɃT[o̓̓`FbNG[ʃR[h\łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "validateException" łB
        /// </remarks>
        protected static readonly string VALIDATE_EXCEPTION = "validateException";

        /// <summary>
        /// G[ێ XML m[h̃m[h\łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "error" łB
        /// </remarks>
        protected static readonly string ERROR_ELEMENT = "error";

        /// <summary>
        /// G[R[hێ XML m[h̃m[h\łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "error-code" łB
        /// </remarks>
        protected static readonly string ERROR_CODE = "error-code";

        /// <summary>
        /// G[bZ[Wێ XML m[h̃m[h\łB
        /// </summary>
        /// <remarks>
        /// ̒萔̒l "error-message" łB
        /// </remarks>
        protected static readonly string ERROR_MESSAGE = "error-message";

        /// <summary>
        /// ɑMf[^( byte )̃ftHglłB
        /// </summary>
        /// <remarks>萔̒l 8192 łB</remarks>
        private const int BUFFER_SIZE = 1024 * 8;

        /// <summary>
        /// ɑMf[^( byte )łB
        /// </summary>
        private int _bufferSize = BUFFER_SIZE;

        /// <summary>
        /// ɑMf[^( byte )\l擾܂͐ݒ肵܂B
        /// </summary>
        /// <remarks>
        /// 0 ȉ̒lݒ肵ꍇA <see cref="BUFFER_SIZE"/> ̒lp܂B
        /// </remarks>
        public int BufferSize
        {
            get
            {
                return _bufferSize;
            }
            set
            {
                if (value <= 0)
                {
                    _bufferSize = BUFFER_SIZE;
                }
                else
                {
                    _bufferSize = value;
                }
            }
        }

        /// <summary>
        /// <paramref name="request"/>œnꂽ<see cref="HttpWebResponse"/>pĎMs܂B
        /// </summary>
        /// <remarks>
        /// <para>ʐMŔ <see cref="WebException"/> ̓Lb`ɃX[܂B܂AŗO
        /// ꍇAʏ <see cref="CommunicationException"/> ɃbvăX[悤ɂ܂BR[fBO
        /// ~Xɂ <see cref="ArgumentNullException"/>  <see cref="ArgumentException"/> At@CVXe
        /// ݒt@Cُ̈ɂObvKv͂܂B</para>
        /// <para>
        /// ̃\bhہA <paramref name="response"/>  <see cref="HttpWebResponse.GetResponseStream"/> 
        /// \bh𗘗pēǂݍ݃Xg[̎擾s܂BXg[ <see cref="ReceiverBase.ReceiveResponse"/> 
        /// \bhŕKv͂܂BĂяoXg[ӔC𕉂܂B</para>
        /// <para>ʏA <see cref="CommunicationResult.ResultData"/> ɂ͓df[^i[f[^Zbgi[܂B
        /// oCit@CMꍇȂǁAi[dȂꍇłA<c>null</c>QƂݒ肷邱Ƃ
        /// ĂBꍇɂ͋<see cref="DataSet"/>CX^X𐶐Đݒ肵܂B
        /// </para>
        /// </remarks>
        /// <exception cref="WebException">
        /// ʐMG[܂B܂́ANGXgLZ܂B
        /// </exception>
        /// <exception cref="CommunicationException">
        /// MɓŃG[܂B
        /// </exception>
        /// <param name="response">MsX|XIuWFNgB</param>
        /// <param name="reporter">is󋵒ʒms<see cref="IProgressChangeReporter"/>CX^XB</param>
        /// <returns>Mʂێ <see cref="CommunicationResult"/>CX^XB</returns>
        protected abstract CommunicationResult ReceiveResponse(HttpWebResponse response,
                                                               IProgressChangeReporter reporter);

        /// <summary>
        /// <paramref name="request"/> œnꂽ <see cref="HttpWebRequest"/> CX^X
        /// <see cref="HttpWebResponse"/> 擾A <see cref="ReceiveResponse"/> \bhpĎMs܂B
        /// </summary>
        /// <para>
        /// ʏÃ\bh𒼐ڃI[o[Ch邱Ƃ͔ĂB <see cref="Receive"/> \bh
        /// <see cref="HttpWebResponse"/> 擾AA<see cref="ReceiveResponse"/> ۃ\bhւƏ
        /// Ϗ܂B<see cref="HttpWebResponse"/> ̉͂̃\bh̏Iɍs܂B̓ɖ
        /// Ȃ΁A <see cref="ReceiveResponse"/> \bhŏ̖{̂Lq܂B
        /// </para>
        /// <param name="request">
        /// MsNGXgIuWFNgB
        /// </param>
        /// <param name="reporter">
        /// is󋵒ʒms <see cref="IProgressChangeReporter"/> NX̃CX^XB
        /// </param>
        /// <returns>Mʂi[ <see cref="CommunicationResult"/>B</returns>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="request"/>  null QƂłB
        /// </exception>
        /// <exception cref="WebException">
        /// T[oƂ̒ʐMɃG[܂B܂͒ʐMLZ܂B
        /// </exception>
        /// <exception cref="ServerException">
        /// ʐM̃T[oŃG[܂B
        /// </exception>
        /// <exception cref="CommunicationException">
        /// MɓŃG[܂B
        /// </exception>
        public virtual CommunicationResult Receive(HttpWebRequest request,
                                                   IProgressChangeReporter reporter)
        {
            if (request == null)
            {
                ArgumentNullException exception = new ArgumentNullException("request");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "request"),exception);
                }
                throw exception;
            }

            CommunicationResult result = null;

            using (HttpWebResponse response = (HttpWebResponse)request.GetResponse())
            {
                CheckResponseError(response);

                result = ReceiveResponse(response, reporter);
            }

            return result;
        }

        /// <summary>
        /// <paramref name="response"/>  HTTP X|X̃{fB擾A<see cref="XmlDocument"/> 
        /// ܂B
        /// </summary>
        /// <remarks>
        /// <para>
        /// <paramref name="response"/>  HTTP X|X̃{fB擾A<see cref="XmlDocument"/> 
        /// ܂B XML f[^ł͂ȂꍇAOƂȂ܂B
        /// </para>
        /// <para>
        /// <paramref name="reporter"/>  <c>null</c> QƂł͂ȂꍇA
        /// <see cref="ReceiverBase.BufferSize"/> vpeBɐݒ肳ꂽobt@TCỸf[^iftHg8 KB jT[oɑM閈ɁA
        /// <see cref="IProgressChangeReporter.ReportProgressChanged"/> Cxg𔭐܂B
        /// </para>
        /// </remarks>
        /// <param name="response">MsX|XIuWFNgB</param>
        /// <param name="reporter">
        /// is󋵒ʒms <see cref="IProgressChangeReporter"/> NX̃CX^XB
        /// </param>
        /// <returns>
        /// T[oMXMLdێ <see cref="XmlDocument"/> CX^XB
        /// </returns>
        /// <exception cref="CommunicationException">
        /// MɓŃG[܂B
        /// </exception>
        protected virtual XmlDocument GetXmlFromResponseBody(WebResponse response, IProgressChangeReporter reporter)
        {
            // f[^XMLłȂ(content-type"text/xml"łȂ)ꍇ
            string contentType = response.ContentType;
            if (string.IsNullOrEmpty(contentType))
            {
                CommunicationException exception
                    = new CommunicationException(Properties.Resources.E_COMMUNICATION_NOT_XML);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }

            contentType = contentType.Split(';')[0];
            if (string.Compare(TEXT_XML, contentType, true) != 0)
            {
                CommunicationException exception
                    = new CommunicationException(Properties.Resources.E_COMMUNICATION_NOT_XML);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(exception.Message, exception);
                }
                throw exception;
            }
            int localBufferSize = BufferSize;
            XmlDocument receiveXml = new XmlDocument();
            byte[] buffer = new byte[localBufferSize];
            using (Stream responseStream = response.GetResponseStream())
            {
                int percentage = 0;
                long receiveSize = 0L;
                long totalSize = response.ContentLength;

                using (MemoryStream stream = new MemoryStream())
                {
                    int readSize = 0;

                    // obt@TCYM閈ɃCxgʒms
                    readSize = responseStream.Read(buffer, 0, localBufferSize);
                    while (readSize > 0)
                    {
                        stream.Write(buffer, 0, readSize);
                        receiveSize += readSize;

                        // |[^[݂Ă
                        if (reporter != null)
                        {
                            if (totalSize > 0)
                            {
                                percentage = CalcReceivePercentage(receiveSize, totalSize);
                            }
                            reporter.ReportProgressChanged(new ExecuteProgressChangedEventArgs(percentage));

                        }

                        readSize = responseStream.Read(buffer, 0, localBufferSize);
                    }

                    stream.Position = 0;

                    try
                    {
                        receiveXml.Load(stream);
                    }
                    catch (XmlException e)
                    {
                        CommunicationException exception
                            = new CommunicationException(Properties.Resources.E_COMMUNICATION_NOT_XML, e);
                        if (_log.IsErrorEnabled)
                        {
                            _log.Error(exception.Message, exception);
                        }
                        throw exception;
                    }
                }
            }
            return receiveXml;
        }

        /// <summary>
        /// T[õG[擾A <see cref="ServerException"/> ֊i[ĕԂ܂B
        /// </summary>
        /// <remarks>
        /// <para>
        /// T[oŃG[ĂꍇA HTTP X|X̃{fB XML `Ƃĉ͂A
        /// <c>error</c>vf擾ăG[쐬܂BG[́A<c>error</c> vfɊ܂܂qvf̒lp
        /// ݒ肳 <see cref="MessageInfo"/> ܂ <see cref="ValidationMessageInfo"/> ̃CX^X̃XgłB
        /// G[͖߂lł <see cref="ServerException"/>̃CX^XɊi[܂B 
        /// </para>
        /// <para><paramref name="errorType"/> ̒l "validateException" łꍇA͒l؃G[Ƃ܂B</para>
        /// <para>͒l؃G[̏ꍇA<c>error</c> vfɊ܂܂qvf̒lpāA <see cref="ValidationMessageInfo"/>
        /// ̃CX^X쐬܂B<c>error</c> vf̎qvfƁA <see cref="ValidationMessageInfo"/> CX^X
        /// i[̊֘AɂẮAȉ̒ʂƂȂ܂B</para>
        /// <para>
        /// <list type="table">
        /// <listheader>
        ///     <term>ValidationMessageInfõvpeB</term>
        ///     <description>i[l</description>
        /// </listheader>
        /// <item>
        ///     <term>Key</term>
        ///     <term>error/error-code</term>
        /// </item>
        /// <item>
        ///     <term>Message</term>
        ///     <description>error/error-message</description>
        /// </item>
        /// <item>
        ///     <term>ErrorPath</term>
        ///     <description>error/error-field</description>
        /// </item>
        /// <item>
        ///     <term>Detail</term>
        ///     <description>error vfǂݍ <see cref="XmlNode"/> CX^XB</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para>
        /// ͒l؃G[̏ꍇA <see cref="ServerException.ErrorType"/> Ɋi[l 
        /// "serverValidateException" łB
        /// </para>
        /// <para><paramref name="errorType"/> ̒l "validateException" ȊOłꍇA<c>error</c> vfɊ܂܂
        /// qvf̒lpāA <see cref="MessageInfo"/> ̃CX^X쐬܂B<c>error</c> vf̎qvfƁA
        /// <see cref="MessageInfo"/> CX^XɊi[̊֘AɂẮAȉ̒ʂƂȂ܂B</para>
        /// <para>
        /// <list type="table">
        /// <listheader>
        ///     <term>MessageInfõvpeB</term>
        ///     <description>i[l</description>
        /// </listheader>
        /// <item>
        ///     <term>Key</term>
        ///     <term>error/error-code</term>
        /// </item>
        /// <item>
        ///     <term>Message</term>
        ///     <description>error/error-message</description>
        /// </item>
        /// <item>
        ///     <term>Detail</term>
        ///     <description>error vfǂݍ <see cref="XmlNode"/> CX^XB</description>
        /// </item>
        /// </list>
        /// </para>
        /// <para><paramref name="errorType"/> ̒l "exception" łƂA<see cref="ServerException.ErrorType"/> 
        /// Ɋi[l "serverException" łB<paramref name="errorType"/> ̒l "validateException" A
        /// "exception" ̂łȂƂA<see cref="ServerException.ErrorType"/> ɂ <paramref name="errorType"/>
        /// ̒l̂܂܊i[܂B</para>
        /// </remarks>
        /// <param name="errorType">T[oŔG[ʂ\B</param>
        /// <param name="xmlDoc">dƂȂ <see cref="XmlDocument"/>B</param>
        /// <returns>
        /// T[oł̃G[ʂi[ <see cref="ServerException"/>B
        /// </returns>
        protected virtual ServerException HandleErrorResult(string errorType, XmlDocument xmlDoc)
        {
            // errorvf擾
            XmlNodeList errorElementList = xmlDoc.DocumentElement.SelectNodes(ERROR_ELEMENT);

            IList<MessageInfo> messageInfoList = new List<MessageInfo>();

            string resultErrorType = null;

            // G[ʂݒ肷
            if (VALIDATE_EXCEPTION.Equals(errorType))
            {            
                resultErrorType = SERVER_VALIDATE_EXCEPTION;
            }
            else
            {
                if (EXCEPTION.Equals(errorType))
                {
                    resultErrorType = SERVER_EXCEPTION;
                }
                else
                {
                    resultErrorType = errorType;
                }
            }

            // G[eݒ肷
            foreach (XmlNode errorElement in errorElementList)
            {
                // error-codevf̒l擾
                string errorCode = null;
                XmlNode currentElement = errorElement.SelectSingleNode(ERROR_CODE);
                if (currentElement != null)
                {
                    errorCode = currentElement.InnerText;
                }

                // error-messagevf̒l擾
                string errorMessage = null;
                currentElement = errorElement.SelectSingleNode(ERROR_MESSAGE);
                if (currentElement != null)
                {
                    errorMessage = currentElement.InnerText;
                }

                // ͒l؃G[̏ꍇ
                if (VALIDATE_EXCEPTION.Equals(errorType))
                {
                    // error-fieldvf̒l擾
                    string errorField = null;
                    currentElement = errorElement.SelectSingleNode(ERROR_FIELD);
                    if (currentElement != null)
                    {
                        errorField = currentElement.InnerText;
                    }

                    ValidationMessageInfo messageInfo = new ValidationMessageInfo(
                        errorCode, errorMessage, errorField);

                    messageInfo.Detail = errorElement;
                    messageInfoList.Add(messageInfo);
                }
                // ͒l؃G[ȊOiVXeG[ƖG[j̏ꍇ
                else
                {
                    MessageInfo messageInfo = new MessageInfo(errorCode, errorMessage);
                    messageInfo.Detail = errorElement;
                    messageInfoList.Add(messageInfo);
                }

            }
            return new ServerException(
                Properties.Resources.E_COMMUNICATION_SERVER_EXCEPTION, resultErrorType, messageInfoList);
        }

        /// <summary>
        /// MTCỸp[Ze[WvZB
        /// </summary>
        /// <param name="receiveSize">MTCYB</param>
        /// <param name="totalSize">vTCYB</param>
        /// <returns>MTCỸp[Ze[WB</returns>
        /// <exception cref="OverflowException">
        /// MTCỸp[Ze[W̌vZŃI[o[t[܂B
        /// </exception>
        protected virtual int CalcReceivePercentage(long receiveSize, long totalSize)
        {
            int percentage = 0;
            checked
            {
                percentage = (int)(receiveSize * 50 / totalSize) + 50;
            }
            return percentage;
        }

        /// <summary>
        /// X|X̓eɃG[Ȃ`FbNB
        /// </summary>
        /// <param name="response">HttpWebResponse</param>
        /// <remarks>
        /// Xe[^XR[h300Ԉȏ̏ꍇAWebExceptionX[B
        /// </remarks>
        /// <exception cref="WebException">
        /// NGXgv͎s܂B
        /// </exception>
        protected virtual void CheckResponseError(HttpWebResponse response)
        {
            if ((int)response.StatusCode >= 300)
            {
                // MHTTPwb_̃Oo
                if (_log.IsTraceEnabled)
                {
                    StringBuilder responseHeader = new StringBuilder();
                    responseHeader.AppendLine(Properties.Resources.T_RESPONSE_RECEIVE_HEADER);
                    foreach (string key in response.Headers.AllKeys)
                    {
                        responseHeader.AppendLine(string.Format(
                        Properties.Resources.T_DICTIONARY_KEY_VALUE,
                        key, response.Headers[key]));
                    }
                    _log.Trace(responseHeader.ToString().Trim());
                }

                WebException exception = new WebException(
                    CreateResponseExceptionString(response), null, WebExceptionStatus.ProtocolError, response);
                throw exception;
            }
        }

        /// <summary>
        /// ObZ[W𐶐B
        /// </summary>
        /// <param name="response">MsX|XIuWFNgB</param>
        /// <returns>ObZ[WB</returns>
        protected virtual string CreateResponseExceptionString(HttpWebResponse response)
        {
            string exceptionString = null;

            int statusCode = (int)response.StatusCode;
            string responseString = ReadResponse(response);

            if (string.IsNullOrEmpty(responseString))
            {
                // MHTTP{fBnull܂͋󕶎̏ꍇ
                exceptionString = string.Format(Properties.Resources.E_COMMUNICATION_INVALID_STATUS_WITH_EMPTY_BODY, statusCode);
            }
            else
            {
                // MHTTP{fB̃Oo
                if (_log.IsTraceEnabled)
                {
                    StringBuilder responseBody = new StringBuilder();
                    responseBody.AppendLine(Properties.Resources.T_RESPONSE_RECEIVE_BODY);
                    responseBody.Append(responseString);
                    _log.Trace(responseBody.ToString());
                }
                exceptionString = string.Format(Properties.Resources.E_COMMUNICATION_INVALID_STATUS, statusCode, responseString);
            }

            return exceptionString;
        }

        /// <summary>
        /// X|X{fBǂݍށB
        /// </summary>
        /// <param name="response">MsX|XIuWFNgB</param>
        /// <returns>X|X{fB̕B</returns>
        protected virtual string ReadResponse(HttpWebResponse response)
        {
            string charset = response.CharacterSet;
            Encoding encoding = GetEncoding(charset);
            Stream stream = response.GetResponseStream();

            using (StreamReader reader = new StreamReader(stream, encoding, true))
            {
                return reader.ReadToEnd();
            }
        }

        /// <summary>
        /// GR[fBO擾B
        /// </summary>
        /// <param name="charset">R[hB</param>
        /// <returns>GR[hB</returns>
        protected virtual Encoding GetEncoding(string charset)
        {
            if (!string.IsNullOrEmpty(charset))
            {
                try
                {
                    return Encoding.GetEncoding(charset);
                }
                catch (ArgumentException)
                {
                    // Ȃ
                }
            }

            return new ASCIIEncoding();
        }

    }
}
