// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.BLogic;
using System.Data;
using TERASOLUNA.Fw.Common.BLogic;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.ClientUT.BLogic.DataSetXmlCommunicateBLogicData;
using TERASOLUNA.Fw.Client.Communication;
using System.Configuration;
using TERASOLUNA.Fw.Client;
namespace TERASOLUNA.Fw.ClientUT.BLogic
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.BLogic.DataSetXmlCommunicateBLogic&lt;T&gt; のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.BLogic.DataSetXmlCommunicateBLogic&lt;T&gt; 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class DataSetXmlCommunicateBLogicTest
    {


        private TestContext testContextInstance;

        /// <summary>
        /// リクエストタイムアウトの構成ファイルの値の取得
        /// </summary>
        private string _reuqestTimeout = null;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
            _reuqestTimeout = ConfigurationManager.AppSettings["RequestTimeout"];
            ConfigurationManager.AppSettings["RequestTimeout"] = "100";
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        [TestCleanup()]
        public void MyTestCleanup()
        {
            ConfigurationManager.AppSettings["RequestTimeout"] = _reuqestTimeout;
        }
        //
        #endregion

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentNullExceptionをスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam が null 参照です。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest01()
        {
            // 準備
            DataSetXmlCommunicateBLogic<DataSet> target = new DataSetXmlCommunicateBLogic<DataSet>();
            BLogicParam blogicParam = null;
            target.Cancelled = false;

            try
            {
                // 実施
                target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("blogicParam", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentExceptionをスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が null 参照です。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest02()
        {
            // 準備
            DataSetXmlCommunicateBLogic<DataSet> target = new DataSetXmlCommunicateBLogic<DataSet>();
            BLogicParam blogicParam = new BLogicParam();
            blogicParam.ParamData = null;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_NULL_PARAMETER", "blogicParam.ParamData"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest03()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);


            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName が未設定です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest04()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_REQUEST_NAME_NOT_FOUND"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest05()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = null;
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_REQUEST_NAME_NOT_FOUND"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }


        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に空文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest06()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = string.Empty;
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_REQUEST_NAME_NOT_FOUND"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に DataSet 型のインスタンスが設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest07()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = new DataSet();
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName.ToString(), communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は設定されていません。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest08()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);


            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に空文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest09()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = string.Empty;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException がスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest10()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "abcdef";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に文字列(-100)が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest11()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "-100";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(Int32.Parse(expectedReuqestTimeout as string), communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に文字列(0)が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest12()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "0";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(Int32.Parse(expectedReuqestTimeout as string), communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に文字列(100)が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders は設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest13()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "100";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(Int32.Parse(expectedReuqestTimeout as string), communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に null 参照が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest14()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            object expectedReuqestHeaders = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 0 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest15()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            object expectedReuqestHeaders = new Dictionary<string, string>();
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException がスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に型 IDictionary&lt:stirng, string&gt: ではない型が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest16()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            object expectedReuqestHeaders = new Object();
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_REFERENCE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders.GetType().FullName, typeof(IDictionary<string, string>).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest17()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = "key01";
            string expectedValue01 = "value01";
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(2, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedValue01, communicator._requestHeaders[expectedKey01]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 3 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest18()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = "key01";
            string expectedKey02 = "key02";
            string expectedKey03 = "key03";
            string expectedValue01 = "value01";
            string expectedValue02 = "value02";
            string expectedValue03 = "value03";
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            expectedReuqestHeaders.Add(expectedKey02, expectedValue02);
            expectedReuqestHeaders.Add(expectedKey03, expectedValue03);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(4, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedValue01, communicator._requestHeaders[expectedKey01]);
            Assert.AreEqual(expectedValue02, communicator._requestHeaders[expectedKey02]);
            Assert.AreEqual(expectedValue03, communicator._requestHeaders[expectedKey03]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には key が空文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest19()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = string.Empty;
            string expectedValue01 = "value01";
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_REQUEST_HEADER_KEY"), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には key が制限付ヘッダ文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest20()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = "Content-Type";
            string expectedValue01 = "value01";
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_RESTRICTED_REQUEST_HEADER_KEY", expectedKey01), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には key が不正な文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest21()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = ";";
            string expectedValue01 = "value01";
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_REQUEST_HEADER_KEY", expectedKey01), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// RequestName はプロパティ RequestName の値が優先されることの確認も行うこと。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には key が RequestName で、正常な value 文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest22()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = CommunicateBLogicBase<DataSet>.REQUEST_NAME;
            string expectedValue01 = "RequestHeaders";
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、value に null 参照が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest23()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = "key01";
            string expectedValue01 = null;
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(2, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedValue01, communicator._requestHeaders[expectedKey01]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、value に空文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest24()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            IDictionary<string, string> expectedReuqestHeaders = new Dictionary<string, string>();
            string expectedKey01 = "key01";
            string expectedValue01 = string.Empty;
            expectedReuqestHeaders.Add(expectedKey01, expectedValue01);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_HEADERS, expectedReuqestHeaders);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(2, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedValue01, communicator._requestHeaders[expectedKey01]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Communicator の処理結果クラス CommunicationResult の プロパティ ResultData は not nullです。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest25()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            DataSet expectedResultData = new DataSet();
            communicator._resultData = expectedResultData;
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.AreSame(expectedResultData, blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Communicator の処理で例外 CommunicationException をスローします。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest26()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub02 target = new DataSetXmlCommunicateBLogicStub02();
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub02 communicator = target._communicator;
            string expectedExceptionMessage = "CommunicationException";
            communicator._exception = new CommunicationException(expectedExceptionMessage);
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            Assert.AreEqual(CommunicateBLogicBase<DataSet>.COMMUNICATION_EXCEPTION, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(1, blogicResult.Errors.Count);
            Assert.AreEqual(CommunicateBLogicBase<DataSet>.COMMUNICATION_EXCEPTION, blogicResult.Errors[0].Key);
            Assert.AreEqual(expectedExceptionMessage, blogicResult.Errors[0].Message);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(CommunicationException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ServerException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Communicator の処理で例外 ServerException (例外種別は serverException) をスローします。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest27()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub02 target = new DataSetXmlCommunicateBLogicStub02();
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub02 communicator = target._communicator;
            Exception expectedException = new ServerException("serverExceptionMessage","serverException");
            communicator._exception = expectedException;
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreSame(expectedException, e);
            }

            // 確認
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Communicator の処理で例外 ServerException (例外種別は serverException) をスローします。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest28()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub02 target = new DataSetXmlCommunicateBLogicStub02();
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub02 communicator = target._communicator;
            string expectedExceptionMessage = "serverExceptionMessage";
            string expectedErrorType = "serverValidateException";
            ServerException exception = new ServerException(expectedExceptionMessage, expectedErrorType);
            MessageInfo messageInfo01 = new MessageInfo("key01", "value01");
            exception.Errors.Add(messageInfo01);
            communicator._exception = exception;
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            Assert.AreEqual(expectedErrorType, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(1, blogicResult.Errors.Count);
            Assert.AreSame(messageInfo01, blogicResult.Errors[0]);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }


        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ServerException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Communicator の処理で例外 ServerException (例外種別は otherException) をスローします。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest29()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub02 target = new DataSetXmlCommunicateBLogicStub02();
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub02 communicator = target._communicator;
            string expectedExceptionMessage = "serverExceptionMessage";
            string expectedErrorType = "otherException";
            ServerException exception = new ServerException(expectedExceptionMessage, expectedErrorType);
            MessageInfo messageInfo01 = new MessageInfo("key01", "value01");
            exception.Errors.Add(messageInfo01);
            communicator._exception = exception;
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            Assert.AreEqual(expectedErrorType, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(1, blogicResult.Errors.Count);
            Assert.AreSame(messageInfo01, blogicResult.Errors[0]);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ServerException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、TerasolunaException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Communicator の処理で例外 TerasolunaException をスローします。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest30()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub02 target = new DataSetXmlCommunicateBLogicStub02();
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub02 communicator = target._communicator;
            Exception expectedException = new TerasolunaException("TerasolunaException");
            communicator._exception = expectedException;
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                Assert.AreSame(expectedException, e);
            }

            // 確認
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(TerasolunaException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// プロパティ Cancelled が true に設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest31()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = true;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            Assert.IsNull(blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Items.Count);
            Assert.AreEqual(0, blogicResult.Errors.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は設定されていません。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout が設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest32()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = null;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100000, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout が設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest33()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = null;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100000, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に空文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout が設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest34()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = string.Empty;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = null;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100000, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout が設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest35()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "abcdef";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = null;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に文字列(100)が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout が設定されていません。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest36()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = 100;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = null;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedReuqestTimeout, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は設定されていません。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に空文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest37()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = string.Empty;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100000, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に空文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest38()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = string.Empty;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100000, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は空文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に空文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest39()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = string.Empty;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = string.Empty;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(100000, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は数値ではない文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に空文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest40()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "abcdef";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = string.Empty;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に文字列(100)が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に空文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest41()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = 100;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = string.Empty;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedReuqestTimeout, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は設定されていません。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest42()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            target.Cancelled = false;

            string expectedAppSettingsRequestTimeout = "abcdef";
            ConfigurationManager.AppSettings["RequestTimeout"] = expectedAppSettingsRequestTimeout;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedAppSettingsRequestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest43()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            string expectedAppSettingsRequestTimeout = "abcdef";
            ConfigurationManager.AppSettings["RequestTimeout"] = expectedAppSettingsRequestTimeout;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedAppSettingsRequestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に空文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest44()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = string.Empty;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            string expectedAppSettingsRequestTimeout = "abcdef";
            ConfigurationManager.AppSettings["RequestTimeout"] = expectedAppSettingsRequestTimeout;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedAppSettingsRequestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest45()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = "abcdef";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            string expectedAppSettingsRequestTimeout = "abcdef";
            ConfigurationManager.AppSettings["RequestTimeout"] = expectedAppSettingsRequestTimeout;

            try
            {
                // 実施
                BLogicResult blogicResult = target.Execute(blogicParam);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_VALUE_TYPE_INVALID_CAST", CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedAppSettingsRequestTimeout, typeof(int).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(ArgumentException), "W_BLOGIC_EXCEPTION"));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に文字列(100)が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に数値ではない文字列が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest46()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = 100;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);
            target.Cancelled = false;

            ConfigurationManager.AppSettings["RequestTimeout"] = "abcdef";

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(expectedReuqestTimeout, communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は設定されていません。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に -100 が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest47()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            target.Cancelled = false;

            string expectedAppSettingsRequestTimeout = "-100";
            ConfigurationManager.AppSettings["RequestTimeout"] = expectedAppSettingsRequestTimeout;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(Int32.Parse(expectedAppSettingsRequestTimeout), communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、ArgumentException をスローすることを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout は設定されていません。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items にキー RequestHeaders に要素数 1 の型 IDictionary&lt:stirng, string&gt: が設定されています。
        /// 要素内には正常な key 文字列で、正常な value 文字列が設定されています。
        /// </item>
        /// <item>
        /// 構成ファイルの AppSettings に RequestTimeout に 0 が設定されています。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest48()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub01 target = new DataSetXmlCommunicateBLogicStub01();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            target.Cancelled = false;

            string expectedAppSettingsRequestTimeout = "0";
            ConfigurationManager.AppSettings["RequestTimeout"] = expectedAppSettingsRequestTimeout;

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            DataSetXmlCommunciateBLogic_DataSetXmlCommunicatorStub01 communicator = target._communicator;
            Assert.AreEqual(BLogicResult.SUCCESS, blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(expectedParamData, communicator._paramData);
            Assert.AreEqual(1, communicator._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, communicator._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);
            Assert.AreEqual(Int32.Parse(expectedAppSettingsRequestTimeout), communicator.RequestTimeout);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        /// Execute (BLogicParam) のテスト
        /// </summary>
        /// <remarks>
        /// 以下の条件のとき、正常に動作することを確認するテスト。
        /// <list type="bullet">
        /// <item>
        /// 引数 blogicParam のプロパティ ParamData が not null です。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestName に正常な値が設定されています。
        /// </item>
        /// <item>
        /// 引数 blogicParam のプロパティ Items のキー RequestTimeout に null 参照が設定されています。
        /// </item>
        /// <item>
        /// 通信処理中にキャンセルが実行される。
        /// </item>
        /// <item>
        /// 通信処理中に例外が発生する。
        /// </item>
        /// </list>
        /// </remarks>
        [TestMethod()]
        public void ExecuteTest49()
        {
            // 準備
            DataSetXmlCommunicateBLogicStub03 target = new DataSetXmlCommunicateBLogicStub03();
            BLogicParam blogicParam = new BLogicParam();
            DataSet expectedParamData = new DataSet();
            blogicParam.ParamData = expectedParamData;
            object expectedRequestName = "DataSetXmlCommunicateBLogic";
            object expectedReuqestTimeout = null;
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_NAME, expectedRequestName);
            blogicParam.Items.Add(CommunicateBLogicBase<DataSet>.REQUEST_TIMEOUT, expectedReuqestTimeout);

            // 実施
            BLogicResult blogicResult = target.Execute(blogicParam);

            // 確認
            Assert.IsNull(blogicResult.ResultString);
            Assert.IsNull(blogicResult.ResultData);
            Assert.AreEqual(0, blogicResult.Items.Count);
            Assert.AreEqual(0, blogicResult.Errors.Count);
            Assert.AreSame(blogicParam, target._communicateBLogicParam);
            Assert.AreEqual(1, target._requestHeaders.Count);
            Assert.AreEqual(expectedRequestName, target._requestHeaders[CommunicateBLogicBase<DataSet>.REQUEST_NAME]);

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicateBLogicBase<DataSet>), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(DataSetXmlCommunicateBLogic<DataSet>), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckWarnUseMessageId(typeof(CommunicateBLogicBase<DataSet>), typeof(TerasolunaException), "W_BLOGIC_EXCEPTION"));
        }
    }
}
