﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Collections.Generic;
using System.Data;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client
{
    /// <summary>
    /// <see cref="Coordinator"/> の実行結果を格納するためのクラスです。
    /// </summary>
    public class ExecutionResult
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ExecutionResult));

        /// <summary>
        /// イベント処理で正常に処理が終了した時、<see cref="ExecutionResult"/> に渡す結果文字列です。
        /// </summary>
        /// <remarks>
        /// <para>
        /// この定数の値は "success" です。
        /// </para>
        /// </remarks>
        public static readonly string SUCCESS = "success";

        /// <summary>
        /// イベント処理の実行結果を表す結果文字列です。
        /// </summary>
        private string _resultString = null;

        /// <summary>
        /// ビジネスロジック出力 <see cref="DataSet"/> です。
        /// </summary>
        private DataSet _blogicResultData = null;

        /// <summary>
        /// エラーメッセージを保持するリストです。
        /// </summary>
        private IList<MessageInfo> _errors = new List<MessageInfo>();


        /// <summary>
        /// <see cref="ExecutionResult"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// デフォルトコンストラクタです。
        /// </remarks>
        public ExecutionResult()
        {
        }

        /// <summary>
        /// <see cref="ExecutionResult"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="resultString">ビジネスロジックの結果文字列。</param>
        public ExecutionResult(string resultString)
            : this(resultString, null)
        {
        }

        /// <summary>
        /// <see cref="ExecutionResult"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="resultString">ビジネスロジックの結果文字列。</param>
        /// <param name="blogicResultData">サーバ処理の結果データを保持する <see cref="DataSet"/></param>
        public ExecutionResult(string resultString, DataSet blogicResultData)
        {
            _resultString = resultString;
            _blogicResultData = blogicResultData;
        }

        /// <summary>
        /// イベント処理の実行結果を表す結果文字列を取得または設定します。
        /// </summary>
        /// <value>
        /// イベント処理の結果文字列。
        /// </value>
        public string ResultString
        {
            get
            {
                return _resultString;
            }
            set
            {
                _resultString = value;
            }
        }

        /// <summary>
        /// ビジネスロジック出力 <see cref="DataSet"/> を取得または設定します。
        /// </summary>
        /// <value>
        /// ビジネスロジック出力 <see cref="DataSet"/> 。
        /// </value>
        public DataSet BLogicResultData
        {
            get
            {
                return _blogicResultData;
            }
            set
            {
                _blogicResultData = value;
            }
        }

        /// <summary>
        /// 実行結果のエラーメッセージリストを取得します。
        /// </summary>
        /// <value>
        /// 実行結果のエラーメッセージリスト。
        /// </value>
        public IList<MessageInfo> Errors
        {
            get
            {
                return _errors;
            }
        }

        /// <summary>
        /// イベント処理の実行結果が "success" かどうかを取得します。
        /// </summary>
        /// <value>
        /// イベント処理の実行結果が "success" であれば true 。
        /// </value>
        public bool Success
        {
            get
            {
                return SUCCESS.Equals(ResultString);
            }
        }

        /// <summary>
        /// 実行結果のエラーメッセージを一括して追加します。
        /// </summary>
        /// <remarks>
        /// 実行結果のエラーメッセージを <paramref name="Errors"/> に追加します。
        /// </remarks>
        /// <param name="messages">追加したいエラーメッセージリスト。</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="messages"/> が <c>null</c> 参照です。
        /// </exception>
        public virtual void AddErrors(IList<MessageInfo> messages)
        {
            if (messages == null)
            {
                ArgumentNullException exception = new ArgumentNullException("messages");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(Properties.Resources.E_NULL_ARGUMENT, "messages"), exception);
                }
                throw exception;
            }

            foreach (MessageInfo message in messages)
            {
                _errors.Add(message);
            }
        }
    }
}
