// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using System.Net;
using TERASOLUNA.Fw.Client.Communication;
using System.Data;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Client;
using System.IO;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Validation;
using System.Xml;
using TERASOLUNA.Fw.ClientUT.Communication.DataSetXmlReceiverData;

namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.Receiver.DataSetXmlReceiver&lt;T&gt; のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.Receiver.DataSetXmlReceiver&lt;T&gt; 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class DataSetXmlReceiverTest
    {

        private TestContext testContextInstance;

        ///<summary>
        /// 正常な通信先を表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://localhost/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NORMAL_URL = "http://localhost/TestServer/Index.aspx";

        ///<summary>
        /// 存在しないサーバを表すURL文字列です。
        ///</summary>
        ///<remarks>
        ///  <para>
        ///   この定数の値は "http://errorserver/TestServer/Index.aspx" です。
        ///  </para>
        ///</remarks>
        private static readonly string TEST_NO_SERVER_URL = "http://errorserver/DummyServer/Index.aspx";

        /// <summary>
        /// 通信相手に通知するリクエスト名のキー値です。
        /// </summary>
        /// <remarks>この定数の値は "RequestName" です。</remarks>
        private static readonly string REQUEST_NAME = "RequestName";

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        ///XmlReadMode のテスト
        ///</summary>
        [TestMethod()]
        public void XmlReadModeTest01()
        {
            //準備
            DataSetXmlReceiver<DataSet> target = new DataSetXmlReceiver<DataSet>();
            XmlReadMode val = XmlReadMode.ReadSchema;

            //実行
            target.ReadMode = val;

            //検証
            Assert.AreEqual(val, target.ReadMode);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SenderBase<DataSet>), LogLevel.INFO));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// requestがnullの場合、ArgumentNullExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest01()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            // 引数
            HttpWebRequest request = null;

            CommunicationResult actual;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("request", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ReceiverBase), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "request"));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// requestのアドレスが存在しないサーバの場合、WebExceptionがスローされることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest02()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            request.Abort();

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// リクエストが正常に終了した場合、電文ヘッダとボディを正常に受信でき、イベントが実行され電文ログが出力されることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest03()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success01");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, target.GetType(), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// reporterがnullの場合、リクエストが正常に終了し、
        /// 電文ヘッダとボディを正常に受信でき、イベントが実行されず、電文ログが出力されることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest04()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success01");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, null);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, target.GetType(), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///BufferSizeが1の場合、正常にイベントが実行されることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest05()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success01");

            // プロパティ
            target.BufferSize = 1;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, target.GetType(), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// XMLのReadModeがReadSchemaの場合、正常に受信できることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest06()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success03");

            // プロパティ
            target.BufferSize = 8192;
            target.ReadMode = XmlReadMode.ReadSchema;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.WriteSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.WriteSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// 電文のボディ部を型なしデータセットにマッピングできることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest07()
        {
            //準備
            DataSetXmlReceiver<DataSet> target = new DataSetXmlReceiver<DataSet>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success02");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", actual.ResultData.Tables["DataTable1"].Rows[0]["Column1"].ToString());

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, target.GetType(), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Typeが空文字列の場合、例外がスローされることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest08()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success04");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_NOT_XML"), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ReceiverBase), typeof(CommunicationException), "E_COMMUNICATION_NOT_XML"));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Typeがtext/xmlでない場合、例外がスローされることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest09()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success05");

            // プロパティ
            target.BufferSize = 8192;

            try
            {
                // 実行
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_NOT_XML"), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ReceiverBase), typeof(CommunicationException), "E_COMMUNICATION_NOT_XML"));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Typeがtext/xml;charset=utf=8の場合、正常に受信できることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest10()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success06");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);
            
            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml;charset=utf-8", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// Content-Typeがtext/xml;charset=sjisの場合、正常に受信できることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest11()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success07");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml;charset=sjis", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error01.xml")]
        public void ReceiveTest12()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error01");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE01", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE01", e.Errors[0].Message);
            }
            
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error01.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：validateException
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        /// error-field要素の数：1個
        /// error-field要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error02.xml")]
        public void ReceiveTest13()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error02");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_VALIDATE_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE02", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE02", e.Errors[0].Message);
                Assert.AreEqual("ERROR_FIELD02", ((ValidationMessageInfo)e.Errors[0]).ErrorPath);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error02.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：otherException
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error03.xml")]
        public void ReceiveTest14()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error03");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual("otherException", e.ErrorType);
                Assert.AreEqual("ERROR_CODE03", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE03", e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error03.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///受信データがXML出ない場合、例外をスローすることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest17()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Download01");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_NOT_XML"), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ReceiverBase), typeof(CommunicationException), "E_COMMUNICATION_NOT_XML"));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///ルートノードが存在しない電文の場合、例外をスローすることの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest18()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success15");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            try
            {
                CommunicationResult actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (CommunicationException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_NOT_XML"), e.Message);
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ReceiverBase), typeof(CommunicationException), "E_COMMUNICATION_NOT_XML"));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：0個
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error04.xml")]
        public void ReceiveTest19()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error04");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(0, e.Errors.Count);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error04.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：1個
        /// error-code要素の数：0個
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error05.xml")]
        public void ReceiveTest20()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error05");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.IsNull(e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE05", e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.ERROR));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));
            
            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error05.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：空文字列
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error06.xml")]
        public void ReceiveTest21()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error06");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual(String.Empty, e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE06", e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.ERROR));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error06.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：0個
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error07.xml")]
        public void ReceiveTest22()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error07");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE07", e.Errors[0].Key);
                Assert.IsNull(e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.ERROR));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error07.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：空文字列
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error08.xml")]
        public void ReceiveTest23()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error08");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE08", e.Errors[0].Key);
                Assert.AreEqual(String.Empty, e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error08.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：validateException
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        /// error-field要素の数：0個
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error09.xml")]
        public void ReceiveTest24()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error09");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_VALIDATE_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE09", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE09", e.Errors[0].Message);
                Assert.IsNull(((ValidationMessageInfo)e.Errors[0]).ErrorPath);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error09.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：validateException
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        /// error-field要素の数：1個
        /// error-field要素の内容：空文字列
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error10.xml")]
        public void ReceiveTest25()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error10");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_VALIDATE_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE10", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE10", e.Errors[0].Message);
                Assert.AreEqual(String.Empty, ((ValidationMessageInfo)e.Errors[0]).ErrorPath);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error10.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：validateException
        /// error要素の数：3個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        /// error-field要素の数：1個
        /// error-field要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error11.xml")]
        public void ReceiveTest26()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error11");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(3, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_VALIDATE_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE11_1", e.Errors[0].Key);
                Assert.AreEqual("ERROR_CODE11_2", e.Errors[1].Key);
                Assert.AreEqual("ERROR_CODE11_3", e.Errors[2].Key);
                Assert.AreEqual("ERROR_MESSAGE11_1", e.Errors[0].Message);
                Assert.AreEqual("ERROR_FIELD11_1", ((ValidationMessageInfo)e.Errors[0]).ErrorPath);
                Assert.AreEqual("ERROR_MESSAGE11_2", e.Errors[1].Message);
                Assert.AreEqual("ERROR_FIELD11_2", ((ValidationMessageInfo)e.Errors[1]).ErrorPath);
                Assert.AreEqual("ERROR_MESSAGE11_3", e.Errors[2].Message);
                Assert.AreEqual("ERROR_FIELD11_3", ((ValidationMessageInfo)e.Errors[2]).ErrorPath);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error11.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        /// エラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：exception
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        /// error-field要素の数：0個
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error12.xml")]
        public void ReceiveTest27()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error12");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE12", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE12", e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.ERROR));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(target.GetType(), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error12.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        /// error以外の要素が含まれているエラー電文を正常に受信できることを確認する。
        /// exceptionヘッダ：validateException
        /// error要素の数：1個
        /// error-code要素の数：1個
        /// error-code要素の内容：正常値
        /// error-message要素の数：1個
        /// error-message要素の内容：正常値
        ///</remarks>
        [TestMethod()]
        [DeploymentItem(@"Communication\DataSetXmlReceiverData\DataSetXmlReceiver_Error13.xml")]
        public void ReceiveTest28()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Error13");

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual = null;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ServerException e)
            {
                Assert.AreEqual(MessageManager.GetMessage("E_COMMUNICATION_SERVER_EXCEPTION"), e.Message);
                Assert.AreEqual(1, e.Errors.Count);
                Assert.AreEqual(ReceiverBase.SERVER_EXCEPTION, e.ErrorType);
                Assert.AreEqual("ERROR_CODE13", e.Errors[0].Key);
                Assert.AreEqual("ERROR_MESSAGE13", e.Errors[0].Message);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), typeof(ServerException), "E_COMMUNICATION_SERVER_EXCEPTION"));

            // 電文ログ確認
            Assert.AreEqual(3, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, target.GetType()));
            StringBuilder errorBody = new StringBuilder();
            errorBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));
            errorBody.Append(new StreamReader("DataSetXmlReceiver_Error13.xml").ReadToEnd());
            Assert.IsTrue(LogUTUtil.CheckTrace(target.GetType(), errorBody.ToString().Trim()));

        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        ///受信データセットと電文のスキーマが違う場合、マッピングが行われないことの確認。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest29()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success08");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // データテーブルが生成されないことの確認
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable2.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable3.Count);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>)));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// データセットに名前空間があるのに、電文に名前空間が付いていない場合、例外がスローされることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest30()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success09");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // データテーブルが生成されないことの確認
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable2.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable3.Count);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);


            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>)));
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// データセットに名前空間と電文の名前空間が一致しない場合、例外がスローされることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest31()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success10");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // データテーブルが生成されないことの確認
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable2.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable3.Count);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);


            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLogHeader(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>)));
        }


        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// 電文がデータセットに正常にマッピングされることの確認。
        /// DataTableの要素数：0
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest32()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success11");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable2.Count);
            Assert.AreEqual(0, ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable3.Count);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// 電文がデータセットに正常にマッピングされることの確認。
        /// DataTableの要素数：3
        /// DataColumnの要素数：1
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest33()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success12");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);
            Assert.AreEqual("テラソルナ02", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable2[0].Column1);
            Assert.AreEqual("テラソルナ03", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable3[0].Column1);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// 電文がデータセットに正常にマッピングされることの確認。
        /// DataTableの要素数：1
        /// DataColumnの要素数：0
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest34()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success13");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.IsTrue(((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].IsColumn1Null());

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// 電文がデータセットに正常にマッピングされることの確認。
        /// DataTableの要素数：1
        /// DataColumnの要素数：3
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest35()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success14");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);
            Assert.AreEqual("テラソルナ02", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column2);
            Assert.AreEqual("テラソルナ03", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column3);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, typeof(DataSetXmlReceiver<DataSetXmlReceiverDataSet01>), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// requestのURL
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest36()
        {
            //準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();
            
            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NO_SERVER_URL);

            // プロパティ
            target.BufferSize = 8192;

            CommunicationResult actual;

            try
            {
                // 実行
                actual = target.Receive(request, reporter);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (WebException)
            {
                // 成功
            }

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(target.GetType(), LogLevel.INFO));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
        }

        /// <summary>
        ///Receive (HttpWebRequest, IProgressChangeReporter) のテスト
        ///</summary>
        ///<remarks>
        /// 電文がデータセットに正常にマッピングされることの確認。
        /// DataTableの要素数：1
        /// DataColumnの要素数：3
        /// DataTableの行（DataRow）の数：3
        ///</remarks>
        [TestMethod()]
        public void ReceiveTest37()
        {
            // 準備
            DataSetXmlReceiver<DataSetXmlReceiverDataSet01> target = new DataSetXmlReceiver<DataSetXmlReceiverDataSet01>();

            // 進行状況イベント
            DataSetXmlReceiver_IProgressChangeReporterStub01 reporter = new DataSetXmlReceiver_IProgressChangeReporterStub01();

            HttpWebRequest request = (HttpWebRequest)HttpWebRequest.Create(TEST_NORMAL_URL);

            // 事前にDataSetXmlSenderを使ってXMLのリクエストを送信しておく
            SendRequest(request, "DataSetXmlReceiver_Success16");

            // プロパティ
            target.BufferSize = 8192;

            // 実行
            CommunicationResult actual = target.Receive(request, reporter);

            // 成功レスポンスが返却されることの確認
            Assert.AreEqual("テラソルナ01_1", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column1);
            Assert.AreEqual("テラソルナ02_1", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column2);
            Assert.AreEqual("テラソルナ03_1", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[0].Column3);
            Assert.AreEqual("テラソルナ01_2", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[1].Column1);
            Assert.AreEqual("テラソルナ02_2", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[1].Column2);
            Assert.AreEqual("テラソルナ03_2", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[1].Column3);
            Assert.AreEqual("テラソルナ01_3", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[2].Column1);
            Assert.AreEqual("テラソルナ02_3", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[2].Column2);
            Assert.AreEqual("テラソルナ03_3", ((DataSetXmlReceiverDataSet01)actual.ResultData).DataTable1[2].Column3);

            // レスポンスヘッダ
            Assert.AreEqual("text/xml", actual.ResponseHeaders["Content-Type"]);
            Assert.AreEqual(GetDataSetXmlStreamLength(actual.ResultData, XmlWriteMode.IgnoreSchema).ToString(), actual.ResponseHeaders["Content-Length"]);

            // イベント実行確認
            Assert.IsTrue(reporter.reportProgressChanged);

            // 電文ログ確認
            Assert.AreEqual(2, LogUTUtil.CountLogs(target.GetType(), LogLevel.TRACE));
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ReceiverBase), LogLevel.INFO));
            Assert.IsTrue(CheckResponseLog(request.GetResponse() as HttpWebResponse, target.GetType(), actual.ResultData, XmlWriteMode.IgnoreSchema));
        }


        /// <summary>
        /// リクエストを送信する。
        /// </summary>
        /// <param name="requestname"></param>
        private void SendRequest(HttpWebRequest request, string requestname)
        {
            DataSetXmlSender sender = new DataSetXmlSender();

            DataSet dataSet = new DataSet();

            // ヘッダ
            System.Collections.Generic.IDictionary<string, string> headerCollection = new Dictionary<string, string>();
            headerCollection.Add(REQUEST_NAME, requestname);

            sender.Send(request, dataSet, headerCollection, null);
        }

        /// <summary>
        /// DataSetから生成されるXMLのサイズ(バイト)を返却する。
        /// </summary>
        /// <returns></returns>
        private int GetDataSetXmlStreamLength(DataSet dataSet, XmlWriteMode mode)
        {
            if(mode.Equals(XmlWriteMode.WriteSchema)){
                dataSet.WriteXml("terasoluna.txt", mode);
                return File.ReadAllBytes("terasoluna.txt").Length;
            }
            else{            
                MemoryStream memoryStream = new MemoryStream();
                dataSet.WriteXml(memoryStream);
                return (int)memoryStream.Length;
            }
        }


        /// <summary>
        /// 電文ログの出力チェックを行い、成功時にtrueを返却するメソッド。
        /// </summary>
        /// <param name="request">リクエスト</param>
        /// <param name="paramData">データセット</param>
        /// <returns>成功の場合true</returns>
        private bool CheckResponseLog(HttpWebResponse response, Type type, DataSet dataSet, XmlWriteMode mode)
        {

            CheckResponseLogHeader(response, type);

            // HTTPボディのチェック
            StringBuilder responseBody = new StringBuilder();
            responseBody.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_BODY"));

            if (XmlWriteMode.WriteSchema.Equals(mode))
            {
                dataSet.WriteXml("terasoluna.txt", XmlWriteMode.WriteSchema);
                using (StreamReader sr = new StreamReader("terasoluna.txt", Encoding.GetEncoding("UTF-8")))
                {
                    responseBody.Append(sr.ReadToEnd());
                }
            }
            else
            {
                responseBody.Append(dataSet.GetXml());
            }

            if (!LogUTUtil.CheckTrace(type, responseBody.ToString()))
            {
                return false;
            }

            return true;
        }

        
        /// <summary>
        /// 電文ログの出力チェックを行い、成功時にtrueを返却するメソッド。
        /// </summary>
        /// <param name="request">リクエスト</param>
        /// <param name="paramData">データセット</param>
        /// <returns>成功の場合true</returns>
        private bool CheckResponseLogHeader(HttpWebResponse response, Type type)
        {

            // HTTPヘッダのチェック
            StringBuilder responseHeader = new StringBuilder();
            responseHeader.AppendLine(MessageManager.GetMessage("T_RESPONSE_RECEIVE_HEADER"));
            foreach (string key in response.Headers.AllKeys)
            {
                responseHeader.AppendLine(string.Format(
                MessageManager.GetMessage("T_DICTIONARY_KEY_VALUE",
                key, response.Headers[key])));
            }

            if (!LogUTUtil.CheckTrace(type, responseHeader.ToString().Trim()))
            {
                return false;
            }

            return true;
        }

    }


}
