// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Communication;
using System.Data;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.ClientUT.Communication
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Communication.CommunicationResult のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Communication.CommunicationResult 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class CommunicationResultTest
    {

        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        ///CommunicationResult (DataSet) のテスト
        ///</summary>
        [TestMethod()]
        public void ConstructorTest()
        {
            // 準備
            DataSet resultData = new DataSet();

            // 実施
            CommunicationResult target = new CommunicationResult(resultData);

            // 確認
            Assert.IsNotNull(target);
            Assert.AreSame(resultData, target.ResultData);
            Assert.AreEqual(0, target.ResponseHeaders.Count);
        }

        /// <summary>
        /// AddResponseHeaders (IDictionary&lt;string,string&gt;) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 responseHeaders が null の場合、ArgumentNullException をスローすることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddResponseHeadersTest01()
        {
            // 準備
            CommunicationResult target = new CommunicationResult();
            IDictionary<string, string> responseHeaders = null;

            try
            {
                // 実施
                target.AddResponseHeaders(responseHeaders);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 確認
                Assert.AreEqual("responseHeaders", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(CommunicationResult), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(CommunicationResult), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "responseHeaders"));
        }

        /// <summary>
        /// AddResponseHeaders (IDictionary&lt;string,string&gt;) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 responseHeaders の要素数が 0 個の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddResponseHeadersTest02()
        {
            // 準備
            CommunicationResult target = new CommunicationResult();
            IDictionary<string, string> ResponseHeaders = new Dictionary<string, string>();

            // 実施
            target.AddResponseHeaders(ResponseHeaders);

            // 確認
            Assert.AreEqual(0, target.ResponseHeaders.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicationResult), LogLevel.INFO));
        }

        /// <summary>
        /// AddResponseHeaders (IDictionary&lt;string,string&gt;) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 responseHeaders の要素数が 1 個の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddResponseHeadersTest03()
        {
            // 準備
            CommunicationResult target = new CommunicationResult();
            IDictionary<string, string> responseHeaders = new Dictionary<string, string>();
            string expectedKey01 = "TERASOLUNA01";
            string expectedValue01 = "terasoluna01";
            responseHeaders.Add(expectedKey01, expectedValue01);

            // 実施
            target.AddResponseHeaders(responseHeaders);

            // 確認
            Assert.AreEqual(1, target.ResponseHeaders.Count);
            Assert.AreEqual(expectedValue01, target.ResponseHeaders[expectedKey01]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicationResult), LogLevel.INFO));
        }

        /// <summary>
        /// AddResponseHeaders (IDictionary&lt;string,string&gt;) のテスト
        /// </summary>
        /// <remarks>
        /// 引数 responseHeaders の要素数が 1 個の場合、正常に動作することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void AddResponseHeadersTest04()
        {
            // 準備
            CommunicationResult target = new CommunicationResult();
            IDictionary<string, string> responseHeaders = new Dictionary<string, string>();
            string expectedKey01 = "TERASOLUNA01";
            string expectedKey02 = "TERASOLUNA02";
            string expectedKey03 = "TERASOLUNA03";
            string expectedValue01 = "terasoluna01";
            string expectedValue02 = "terasoluna02";
            string expectedValue03 = "terasoluna03";
            responseHeaders.Add(expectedKey01, expectedValue01);
            responseHeaders.Add(expectedKey02, expectedValue02);
            responseHeaders.Add(expectedKey03, expectedValue03);

            // 実施
            target.AddResponseHeaders(responseHeaders);

            // 確認
            Assert.AreEqual(3, target.ResponseHeaders.Count);
            Assert.AreEqual(expectedValue01, target.ResponseHeaders[expectedKey01]);
            Assert.AreEqual(expectedValue02, target.ResponseHeaders[expectedKey02]);
            Assert.AreEqual(expectedValue03, target.ResponseHeaders[expectedKey03]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicationResult), LogLevel.INFO));
        }

        /// <summary>
        /// ResultData のテスト
        /// </summary>
        /// <remarks>
        /// プロパティ ResultData の設定・取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ResultDataTest()
        {
            // 準備
            CommunicationResult target = new CommunicationResult();
            DataSet expectedResultData = new DataSet();

            // 実施
            target.ResultData = expectedResultData;
            DataSet actualResultData = target.ResultData;

            // 確認
            Assert.AreEqual(expectedResultData, actualResultData);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicationResult), LogLevel.INFO));
        }

        /// <summary>
        /// ResponseHeaders のテスト
        /// </summary>
        /// <remarks>
        /// プロパティ ResponseHeaders の取得ができることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ResponseHeadersTest()
        {
            // 準備
            CommunicationResult target = new CommunicationResult();

            // 実施
            IDictionary<string, string> actualResponseHeaders = target.ResponseHeaders;

            // 確認
            Assert.AreEqual(0, actualResponseHeaders.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(CommunicationResult), LogLevel.INFO));
        }

    }


}
