// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Web.Configuration.Sql;
using System.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;

namespace TERASOLUNA.Fw.WebUT.Configuration.Sql
{
    /// <summary>
    /// TERASOLUNA.Fw.Common.Configuration.SqlConfiguration のテスト クラスです。すべての
    /// TERASOLUNA.Fw.Common.Configuration.SqlConfiguration 単体テストをここに含めます
    /// </summary>
 
    [TestClass()]
    public class SqlConfigurationTest
    {

        private TestContext testContextInstance;

        /// <summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        /// </summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        /// GetSql (string) のテスト
        /// </summary>
        /// <TestTarget>GetSql (string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に外部設定ファイルに存在するSQL文識別子が指定されていること
        /// </TestCondition>
        /// <TestAssertion>SQLが取得できること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Sql\TestItem_SqlExternal01.config")]
        public void TestGetSql01()
        {
            string name = "1";

            string expected = "SELECT 1 FROM TABLE";
            string actual;

            actual = SqlConfiguration.GetSql(name);

            Assert.AreEqual(expected, actual);
           
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlConfiguration), LogLevel.INFO));
        }

        /// <summary>
        /// GetSql (string) のテスト
        /// </summary>
        /// <TestTarget>GetSql (string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数にnullが指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Sql\TestItem_SqlExternal01.config")]
        public void TestGetSql02()
        {
            string name = null;
            string actual;

            try
            {
                actual = SqlConfiguration.GetSql(name);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("name", e.ParamName);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SqlConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(SqlConfiguration), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "name"));
        }

        /// <summary>
        /// GetSql (string) のテスト
        /// </summary>
        /// <TestTarget>GetSql (string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に空文字列が指定されていること
        /// </TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Sql\TestItem_SqlExternal01.config")]
        public void TestGetSql03()
        {
            string name = "";
            string actual;

            try
            {
                actual = SqlConfiguration.GetSql(name);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 成功
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "name"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SqlConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(SqlConfiguration), typeof(ArgumentException), "E_EMPTY_STRING", "name"));

        }

        /// <summary>
        /// GetSql (string) のテスト
        /// </summary>
        /// <TestTarget>GetSql (string)</TestTarget>
        /// <TestCondition>
        /// 構成ファイル、外部設定ファイルが正常なこと
        /// 引数に外部設定ファイルに存在しないSQL文識別子が指定されていること
        /// </TestCondition>
        /// <TestAssertion>ConfigurationErrorsExceptionがスローされること</TestAssertion>
        [TestMethod()]
        [DeploymentItem(@"Configuration\Sql\TestItem_SqlExternal01.config")]
        public void TestGetSql04()
        {
            string name = "invalid";
            string actual;

            try
            {
                actual = SqlConfiguration.GetSql(name);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException e)
            {
                // 成功
                Assert.AreEqual(MessageManager.GetMessage("E_SQL_STATEMENT_NOT_FOUND", name), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SqlConfiguration), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(SqlConfiguration), typeof(ConfigurationErrorsException), "E_SQL_STATEMENT_NOT_FOUND", name));
        }
    }
}
