// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client;
using System.Reflection;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
namespace TERASOLUNA.Fw.ClientUT
{
    ///<summary>
    /// ExecuteCompletedEventArgs のテスト クラスです。
    ///</summary>

    [TestClass()]
    public class ExecuteCompletedEventArgsTest
    {


        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        ///<summary>
        /// Constructor のテスト
        ///</summary>
        ///<remarks>
        /// executionResultが有効な値であり、かつerrorが発生せず、calcelledがfalseの場合、
        /// 正常にインスタンス化できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            ExecutionResult executionResult = new ExecutionResult();
            Exception error = null;
            bool cancelled = false;

            // 実行
            ExecuteCompletedEventArgs target = new ExecuteCompletedEventArgs(executionResult, error, cancelled);

            // 検証
            Assert.IsNotNull(target);
            Assert.AreSame(executionResult, target.ExecutionResult);
            Assert.IsNull(target.Error);
            Assert.IsFalse(target.Cancelled);

            // ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecuteCompletedEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// ExecutionResult のテスト
        ///</summary>
        ///<remarks>
        /// executionResultが有効な値であり、かつerrorが発生せず、calcelledがfalseの場合、
        /// executionResultが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecutionResultTest01()
        {
            // 準備
            ExecutionResult executionResult = new ExecutionResult();
            Exception error = null;
            bool cancelled = false;

            // 実行
            ExecuteCompletedEventArgs target = new ExecuteCompletedEventArgs(executionResult, error, cancelled);
            ExecutionResult actual = target.ExecutionResult;

            // 検証
            Assert.AreSame(executionResult, actual);

            // ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecuteCompletedEventArgs), LogLevel.INFO));

        }

        ///<summary>
        /// ExecutionResult のテスト
        ///</summary>
        ///<remarks>
        /// executionResultがnullで、かつerrorが発生せず、cancelledがfalseの場合、
        /// executionResultがnullであることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecutionResultTest02()
        {
            // 準備
            ExecutionResult executionResult = null;
            Exception error = null;
            bool cancelled = false;

            // 実行
            ExecuteCompletedEventArgs target = new ExecuteCompletedEventArgs(executionResult, error, cancelled);
            ExecutionResult actual = target.ExecutionResult;

            // 検証
            Assert.IsNull(actual);

            // ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecuteCompletedEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// ExecutionResult のテスト
        ///</summary>
        ///<remarks>
        /// executionResultがnullで、かつerrorが発生した場合に
        /// TargetInvocationExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecutionResultTest03()
        {
            // 準備
            ExecutionResult executionResult = null;
            Exception error = new Exception();
            bool cancelled = false;

            ExecuteCompletedEventArgs target = new ExecuteCompletedEventArgs(executionResult, error, cancelled);

            // 実行
            try
            {
                ExecutionResult actual = target.ExecutionResult;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                //成功
                Assert.IsInstanceOfType(e.InnerException, error.GetType());
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecuteCompletedEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// ExecutionResult のテスト
        ///</summary>
        ///<remarks>
        /// cancelledがtrueの場合にInvalidOperationExceptionが
        /// スローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecutionResultTest04()
        {
            // 引数
            ExecutionResult executionResult = new ExecutionResult();
            Exception error = null;
            bool cancelled = true;

            ExecuteCompletedEventArgs target = new ExecuteCompletedEventArgs(executionResult, error, cancelled);

            // 実行
            try
            {
                ExecutionResult actual = target.ExecutionResult;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (InvalidOperationException e)
            {
                // 成功
                Assert.IsNull(e.InnerException);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecuteCompletedEventArgs), LogLevel.INFO));
        }

        ///<summary>
        /// ExecutionResult のテスト
        ///</summary>
        ///<remarks>
        /// cancelledがtrueの場合且つにerrorが発生している場合
        /// TargetInvocationExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ExecutionResultTest05()
        {
            // 引数
            ExecutionResult executionResult = new ExecutionResult();
            Exception error = new Exception();
            bool cancelled = true;

            ExecuteCompletedEventArgs target = new ExecuteCompletedEventArgs(executionResult, error, cancelled);

            // 実行
            try
            {
                ExecutionResult actual = target.ExecutionResult;
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TargetInvocationException e)
            {
                // 成功
                Assert.IsInstanceOfType(e.InnerException, error.GetType());
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ExecuteCompletedEventArgs), LogLevel.INFO));
        }
    }
}
