package jp.sourceforge.talisman.hermes.maven;

/*
 * $Id: RepositoryTest.java 198 2009-05-30 14:45:26Z tama3 $
 */

import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 * 
 * @author Haruaki Tamada
 * @version $Revision: 198 $
 */
public class RepositoryTest{
    private Repository repository;

    @Before
    public void setUp(){
        repository = new Repository(
            "local", "local repository for testing", getClass().getResource("/repository/")
        );
    }

    @Test
    public void testRepositoryItSelf() throws Exception{
        Assert.assertEquals(repository.getId(), "local");
        Assert.assertEquals(repository.getName(), "local repository for testing");
        Assert.assertTrue(
            repository.isSameLocation(new File(
                System.getProperty("user.dir") + "/target/test-classes/repository/"
            ).toURI().toURL())
        );
    }

    @Test
    public void testEquals() throws Exception{
        Repository repo2 = new Repository(
            "local", "local repository for testing", getClass().getResource("/repository/")
        ); 
        Assert.assertFalse(repository.equals(null));
        Assert.assertTrue(repository.equals(repo2));
        Assert.assertEquals(repository.hashCode(), repo2.hashCode());
    }

    @Test
    public void testBasic() throws IOException{
        Assert.assertTrue(repository.contains("jp.sourceforge", "stigmata"));
        Assert.assertTrue(repository.contains("jp.sourceforge", "stigmata", "2.1.0"));
        Assert.assertTrue(repository.contains("jp.sourceforge", "stigmata", "2.0.0"));
        Assert.assertTrue(repository.contains("jp.sourceforge", "stigmata", "1.0.0"));

        Assert.assertFalse(repository.contains("jp.sourceforge", "NOTtalisman"));
        Assert.assertFalse(repository.contains("jp.sourceforge", "stigmata", "1.1.0"));
        Assert.assertFalse(repository.contains("jp.sourceforge", "stigmata", "3.0.0"));

        Assert.assertTrue(repository.contains(new Artifact("jp.sourceforge", "stigmata")));
        Assert.assertTrue(repository.contains("jp.sourceforge", "stigmata", null));
    }

    @Test
    public void testFetcher() throws IOException{
        Artifact artifact = new Artifact("jp.sourceforge", "stigmata", "2.0.0");
        ItemFetcher fetcher = repository.getFetcher();
        Assert.assertNotNull(fetcher.fetch(artifact));
    }

    @Test(expected=RepositoryItemNotFoundException.class)
    public void testFetcherRepositoryItemNotFound() throws IOException{
        Artifact artifact = new Artifact("jp.sourceforge", "NOTstigmata", "2.0.0");
        ItemFetcher fetcher = repository.getFetcher();
        fetcher.fetch(artifact);
    }

    @Test
    public void testFindArtifact() throws Exception{
        Artifact artifact1 = new Artifact("jp.sourceforge", "stigmata");
        Artifact artifact2 = repository.findArtifact(artifact1);
        Artifact artifact3 = repository.findArtifact(artifact2);

        Assert.assertNotSame(artifact1, artifact2);
        Assert.assertFalse(artifact1.equals(artifact2)); // because artifact1 do not have version.
        Assert.assertSame(artifact2, artifact3);
        Assert.assertEquals(artifact2, artifact3);

        Assert.assertEquals(artifact2.getGroupId(), "jp.sourceforge");
        Assert.assertEquals(artifact2.getArtifactId(), "stigmata");
        Assert.assertEquals(artifact2.getVersion(), "2.1.0");

        Assert.assertEquals(artifact2.getPom().getDependencyCount(), 8);

        ProjectObjectModel pom = artifact2.getPom();
        Assert.assertTrue(pom.hasLicense());
        Assert.assertEquals(pom.getLicenseCount(), 1);
        License license = pom.getLicense();
        Assert.assertEquals(license.getName(), "Apache License 2.0");
        Assert.assertEquals(license.getUrl(), "http://www.apache.org/licenses/LICENSE-2.0.txt");

        Iterator<Artifact> artifactIterator = artifact2.dependencies();
        Artifact a1 = artifactIterator.next();
        Assert.assertEquals(a1.getGroupId(), "asm");
        Assert.assertEquals(a1.getArtifactId(), "asm-all");
        Assert.assertEquals(a1.getVersion(), "3.1");
        Assert.assertEquals(a1.getScope(), DependencyScope.COMPILE);

        Artifact a2 = artifactIterator.next();
        Assert.assertEquals(a2.getGroupId(), "commons-dbutils");
        Assert.assertEquals(a2.getArtifactId(), "commons-dbutils");
        Assert.assertEquals(a2.getVersion(), "1.1");
        Assert.assertEquals(a2.getScope(), DependencyScope.COMPILE);

        Artifact a3 = artifactIterator.next();
        Assert.assertEquals(a3.getGroupId(), "commons-beanutils");
        Assert.assertEquals(a3.getArtifactId(), "commons-beanutils");
        Assert.assertEquals(a3.getVersion(), "1.7.0");
        Assert.assertEquals(a3.getScope(), DependencyScope.COMPILE);

        Artifact a4 = artifactIterator.next();
        Assert.assertEquals(a4.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a4.getArtifactId(), "mds");
        Assert.assertEquals(a4.getVersion(), "1.0.0");
        Assert.assertEquals(a4.getScope(), DependencyScope.COMPILE);

        Artifact a5 = artifactIterator.next();
        Assert.assertEquals(a5.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a5.getArtifactId(), "xmlcli");
        Assert.assertEquals(a5.getVersion(), "1.2.2");
        Assert.assertEquals(a5.getScope(), DependencyScope.COMPILE);

        Artifact a6 = artifactIterator.next();
        Assert.assertEquals(a6.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(a6.getArtifactId(), "i18n");
        Assert.assertEquals(a6.getVersion(), "1.0.1");
        Assert.assertEquals(a6.getScope(), DependencyScope.COMPILE);

        Artifact a7 = artifactIterator.next();
        Assert.assertEquals(a7.getGroupId(), "jp.sourceforge.stigmata");
        Assert.assertEquals(a7.getArtifactId(), "digger");
        Assert.assertEquals(a7.getVersion(), "1.2.0");
        Assert.assertEquals(a7.getScope(), DependencyScope.COMPILE);

        Artifact a8 = artifactIterator.next();
        Assert.assertEquals(a8.getGroupId(), "junit");
        Assert.assertEquals(a8.getArtifactId(), "junit");
        Assert.assertEquals(a8.getVersion(), "4.3");
        Assert.assertEquals(a8.getScope(), DependencyScope.TEST);

        Assert.assertFalse(artifactIterator.hasNext());
    }

    @Test(expected=RepositoryItemNotFoundException.class)
    public void testFindArtifactNotInRepository() throws Exception{
        repository.findArtifact(new Artifact("org.apache.bcel", "bcel", "5.2"));
    }

    @Test(expected=RepositoryItemNotFoundException.class)
    public void testFindArtifactNotInRepository2() throws Exception{
        repository.findArtifact(new Artifact("org.apache.bcel", "bcel"));
    }

    @Test
    public void testParentPom() throws Exception{
        Artifact artifact = repository.findArtifact(new Artifact("jp.sourceforge.talisman", "csvio", "1.0.1"));
        ProjectObjectModel pom1 = artifact.getPom();
        ProjectObjectModel pom2 = pom1.getParent();

        Assert.assertEquals(pom1.getGroupId(), "jp.sourceforge.talisman");
        Assert.assertEquals(pom1.getArtifactId(), "csvio");
        Assert.assertEquals(pom1.getVersion(), "1.0.1");

        Assert.assertEquals(pom2.getGroupId(), "jp.sourceforge");
        Assert.assertEquals(pom2.getArtifactId(), "talisman");
        Assert.assertEquals(pom2.getVersion(), "1.0");

        Assert.assertEquals(pom1.getLicenseCount(), 1);
        Assert.assertEquals(pom2.getLicenseCount(), 1);

        License license1 = pom1.getLicense();
        License license2 = pom2.getLicense();
        Assert.assertEquals(license1, license2);
        Assert.assertEquals(license1.getName(), "Apache License 2.0");
        Assert.assertEquals(license1.getUrl(), "http://apache.org/licenses/LICENSE-2.0.txt");

        Assert.assertNull(pom2.getParent());
    }
}
