package jp.sourceforge.talisman.hermes.maven;

/*
 * $Id: Repository.java 203 2009-06-01 05:44:11Z tama3 $
 */

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;

import jp.sourceforge.talisman.hermes.RepositoryScope;

/**
 * This class represents a maven2 repository.
 * 
 * @author Haruaki Tamada
 * @version $Revision: 203 $
 */
public class Repository implements Serializable{
    private static final long serialVersionUID = 3922032760947434478L;
    private static final Repository LOCAL_REPOSITORY;

    static{
        try{
            LOCAL_REPOSITORY = new Repository(
                "local", "Maven Local Repository",
                new File(System.getProperty("user.home"), ".m2/repository").toURI().toURL()
            );
        } catch(MalformedURLException e){
            throw new InternalError();
        }
    }

    private transient ItemFetcher fetcher;
    private String id;
    private String name;
    private URL location;

    /**
     * Basic constructor.
     */
    public Repository(String id, String name, URL location){
        this(location);
        this.id = id;
        this.name = name;
    }

    /**
     * Constructor for no id and no name.
     */
    public Repository(URL location){
        this.location = location;
    }

    /**
     * returns local maven2 repository.
     * System.getProperty("user.home") + ".m2/repository"
     */
    public static final Repository getLocalRepository(){
        return LOCAL_REPOSITORY;
    }

    /**
     * returns true when this repository contains given artifact.
     */
    public boolean contains(Artifact artifact) throws IOException{
        if(!artifact.hasVersion()){
            return contains(artifact.getGroupId(), artifact.getArtifactId());
        }
        URL url = new URL(location + artifact.getRelativePath() + "/" + artifact.getFileName());

        try{
            return url.getContent() != null;
        } catch(IOException e){
            return false;
        }
    }

    /**
     * returns true when this repository contains an artifact which
     * has given groupId and given artifactId.
     */
    public boolean contains(String groupId, String artifactId) throws IOException{
        URL url = new URL(location + groupId.replace('.', '/') + "/" + artifactId);
        try{
            return url.getContent() != null;
        } catch(IOException e){
            return false;
        }
    }

    /**
     * returns true when this repository contains an artifact which
     * has given groupId, given artifactId, and given version.
     */
    public boolean contains(String groupId, String artifactId, String version) throws IOException{
        boolean flag;
        if(version == null){
            flag = contains(groupId, artifactId);
        }
        else{
            flag = contains(new Artifact(groupId, artifactId, version));
        }
        return flag;
    }

    public boolean equals(Object o){
        if(o instanceof Repository){
            Repository r = (Repository)o;
            boolean flag = getLocation().equals(r.getLocation());
            if(getId() != null){
                flag = flag & getId().equals(r.getId());
            }
            else{
                flag = flag & r.getId() == null;
            }

            if(getName() != null){
                flag = flag & getName().equals(r.getName());
            }
            else{
                flag = flag & r.getName() == null;
            }

            return flag;
        }
        return false;
    }

    /**
     * find given artifact in this repository.
     * @return found artifact.
     * @throws IOException I/O Error.
     * @throws RepositoryItemNotFoundException This repository do not have given artifact. 
     */
    public Artifact findArtifact(Artifact origArtifact) throws IOException{
        if(origArtifact.hasVersion() && origArtifact.getPom().isAlreadyRead()){
            return origArtifact;
        }
        Artifact artifact = origArtifact;
        if(!origArtifact.hasVersion()){
            artifact = findLatestVersion(artifact);
        }
        if(artifact.hasVersion()){
            parsePom(artifact.getPom());

            return artifact;
        }
        throw new RepositoryItemNotFoundException(String.format(
            "%s.%s is not found.", origArtifact.getGroupId(), origArtifact.getArtifactId()
        ));
    }

    /**
     * find latest version of given artifact.
     */
    public Artifact findLatestVersion(Artifact artifact) throws IOException{
        try{
            String latestVersion = getLatestVersion(artifact);
            return new Artifact(artifact.getGroupId(), artifact.getArtifactId(), latestVersion);
        } catch(RepositoryItemNotFoundException e){
            return artifact;
        }
    }

    /**
     * returns ItemFetcher for downloading items from this repository.
     */
    public synchronized ItemFetcher getFetcher(){
        if(fetcher == null){
            fetcher = new ItemFetcher(this);
        }
        return fetcher;
    }

    /**
     * returns repository id.
     */
    public String getId(){
        return id;
    }

    /**
     * returns repository location.
     */
    public URL getLocation(){
        return location;
    }

    /**
     * returns repository name.
     */
    public String getName(){
        return name;
    }

    public int hashCode(){
        final int prime = 31;
        int result = 1;
        result = prime * result + ((getId() == null)? 0: getId().hashCode());
        result = prime * result + ((getName() == null)? 0: getName().hashCode());
        result = prime * result + ((getLocation() == null)? 0: getLocation().hashCode());
        return result;
    }

    public boolean isAllowed(RepositoryScope scope){
        return scope.isScope(getLocation());
    }

    public boolean isSameLocation(URL location){
        return getLocation().sameFile(location);
    }

    private String getLatestVersion(Artifact artifact) throws IOException{
        Metadata metadata = new Metadata(artifact.getGroupId(), artifact.getArtifactId());
        InputStream in = getFetcher().fetch(metadata);
        new MetadataParser(metadata).parse(in);
        return metadata.getLatestVersion();
    }

    private void parsePom(ProjectObjectModel pom) throws IOException, RepositoryIOException{
        InputStream in = getFetcher().fetch(pom);
        new ProjectObjectModelParser(pom).parse(in);
        pom.setRepository(this);
        if(pom.getParent() != null){
            parsePom(pom.getParent());
        }
        pom.setAlreadyRead(true);
    }
}
