/*
 * Copyright 2002-2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * 
 * "Portions Copyrighted [2006] [Masahito Henmi]"
 * 
 * 
 */
package strawberry.mockup;

import java.io.*;
import java.lang.reflect.Array;
import java.security.Principal;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.logging.*;
/**
 * Mock implementation of the HttpServletRequest interface.
 *
 * <p>Used for testing the web framework; also useful
 * for testing application controllers.
 *
 * @author Rod Johnson
 * @author Juergen Hoeller
 * @since 1.0.2
 */
public class MockHttpServletRequest implements HttpServletRequest {

  // jdk1.4 logging
  protected final static Logger log = Logger.getLogger("strawberry.ctx.request");

  protected final static Logger headLog = Logger.getLogger("strawberry.ctx.header");

  
  public static final String DEFAULT_PROTOCOL = "http";

  public static final String DEFAULT_SERVER_ADDR = "127.0.0.1";

  public static final String DEFAULT_SERVER_NAME = "localhost";

  public static final int DEFAULT_SERVER_PORT = 80;

  public static final String DEFAULT_REMOTE_ADDR = "127.0.0.1";

  public static final String DEFAULT_REMOTE_HOST = "localhost";


  //---------------------------------------------------------------------
  // ServletRequest properties
  //---------------------------------------------------------------------

  private final Hashtable attributes = new Hashtable();

  private String characterEncoding;

  private byte[] content;

  private String contentType;

  private final Map  parameters = new Hashtable();

  private String protocol = DEFAULT_PROTOCOL;

  private String scheme = DEFAULT_PROTOCOL;

  private String serverName = DEFAULT_SERVER_NAME;

  private int serverPort = DEFAULT_SERVER_PORT;

  private String remoteAddr = DEFAULT_REMOTE_ADDR;

  private String remoteHost = DEFAULT_REMOTE_HOST;

  /** List of locales in descending order */
  private final Vector locales = new Vector();

  private boolean secure = false;

  private ServletContext servletContext = null;

  private int remotePort = DEFAULT_SERVER_PORT;

  private String localName = DEFAULT_SERVER_NAME;

  private String localAddr = DEFAULT_SERVER_ADDR;

  private int localPort = DEFAULT_SERVER_PORT;


  //---------------------------------------------------------------------
  // HttpServletRequest properties
  //---------------------------------------------------------------------

  private String authType;

  private Cookie[] cookies;

  private final Hashtable headers = new Hashtable();

  private String method = "GET";

  private String pathInfo;

  private String contextPath = "";

  private String queryString;

  private String remoteUser;

  private final Set  userRoles = new HashSet();

  private Principal userPrincipal;

  private String requestURI = "";

  private String servletPath = "";

  private HttpSession session;

  private boolean requestedSessionIdValid = true;

  private boolean requestedSessionIdFromCookie = true;

  private boolean requestedSessionIdFromURL = false;


  //---------------------------------------------------------------------
  // Constructors
  //---------------------------------------------------------------------

  /**
   * Create a new MockHttpServletRequest.
   * @param servletContext the ServletContext that the request runs in
   */
  public MockHttpServletRequest(ServletContext servletContext) {
    this.locales.add(Locale.ENGLISH);
    this.servletContext = servletContext;
    log.finer("RXgN^P");
  }

  /**
   * Create a new MockHttpServletRequest.
   * @param servletContext the ServletContext that the request runs in
   * @param method the request method
   * @param requestURI the request URI
   * @see #setMethod
   * @see #setRequestURI
   */
  public MockHttpServletRequest(ServletContext servletContext, String method, String requestURI) {
    this.servletContext = servletContext;
    this.method = method;
    this.requestURI = requestURI;
    this.locales.add(Locale.ENGLISH);
    log.finer("RXgN^R");
  }

  /**
   * Create a new MockHttpServletRequest with a MockServletContext.
   * @see MockServletContext
   */
  public MockHttpServletRequest() {
    this.locales.add(Locale.ENGLISH);
    log.finer("RXgN^O");
  }


  //---------------------------------------------------------------------
  // ServletRequest interface
  //---------------------------------------------------------------------

  public Object getAttribute(String name) {
    log.finer("getAttribute ####");
    return this.attributes.get(name);
  }

  public Enumeration getAttributeNames() {
    log.finer("getAttributeNames ####");
    return this.attributes.keys();
  }

  public String getCharacterEncoding() {
    log.finer("getCharacterEncoding ####");
    return characterEncoding;
  }

  public void setCharacterEncoding(String characterEncoding) {
    log.finer("setCharacterEncoding ####");
    this.characterEncoding = characterEncoding;
  }

  public void setContent(byte[] content) {
    log.finer("setContent ####");
    this.content = content;
  }

  public int getContentLength() {
    log.finer("getContentLength ####");
    return (this.content != null ? content.length : -1);
  }

  public void setContentType(String contentType) {
    log.finer("setContentType ####");
    this.contentType = contentType;
  }

  public String getContentType() {
    log.finer("getContentType ####");
    return contentType;
  }

  public ServletInputStream getInputStream() {
    log.finer("getInputStream ####");
    return null;
  }

  /**
   * Add a single value for an HTTP parameter.
   * <p>If there are already one or more values registered for the given
   * parameter name, the given value will be added to the end of the list.
   */
  public void addParameter(String name, String value) {
    addParameter(name, new String[] {value});
  }

  /**
   * Add an array of values for an HTTP parameter.
   * <p>If there are already one or more values registered for the given
   * parameter name, the given values will be added to the end of the list.
   */
  public void addParameter(String name, String[] values) {
    String[] oldArr = (String[]) this.parameters.get(name);
    if (oldArr != null) {
      String[] newArr = new String[oldArr.length + values.length];
      System.arraycopy(oldArr, 0, newArr, 0, oldArr.length);
      System.arraycopy(values, 0, newArr, oldArr.length, values.length);
      this.parameters.put(name, newArr);
    }
    else {
      this.parameters.put(name, values);
    }
  }

  public String getParameter(String name) {
    String[] arr = (String[]) this.parameters.get(name);
    String ans = (arr != null && arr.length > 0 ? arr[0] : null);

    log.fine("257) getParameter(" + name + ")  ==> " + ans);
    return ans;
  }

  public Enumeration getParameterNames() {
    log.finer("getParameterNames ####");
    return Collections.enumeration(this.parameters.keySet());
  }

  public String[] getParameterValues(String name) {
    log.finer("getParameterValues ####");
    return (String[]) this.parameters.get(name);
  }

  public Map getParameterMap() {
    log.finer("getParameterMap ####");
    return Collections.unmodifiableMap(this.parameters);
  }

  public void setProtocol(String protocol) {
    log.finer("setProtocol ####");
    this.protocol = protocol;
  }

  public String getProtocol() {
    log.finer("setProtocol ####");
    return protocol;
  }

  public void setScheme(String scheme) {
    this.scheme = scheme;
  }

  public String getScheme() {
    log.finer("getScheme ####");
    return scheme;
  }

  public void setServerName(String serverName) {
    this.serverName = serverName;
  }

  public String getServerName() {
    log.finer("getServerName ####");
    return serverName;
  }

  public void setServerPort(int serverPort) {
    this.serverPort = serverPort;
  }

  public int getServerPort() {
    log.finer("getServerPort ####");
    return serverPort;
  }

  public BufferedReader getReader() throws UnsupportedEncodingException {
    if (this.content != null) {
      InputStream sourceStream = new ByteArrayInputStream(this.content);
      Reader sourceReader = (this.characterEncoding != null) ?
          new InputStreamReader(sourceStream, this.characterEncoding) : new InputStreamReader(sourceStream);
      return new BufferedReader(sourceReader);
    } else {
      return null;
    }
  }

  public void setRemoteAddr(String remoteAddr) {
    this.remoteAddr = remoteAddr;
  }

  public String getRemoteAddr() {
    log.finer("getRemoteAddr ####");
    return remoteAddr;
  }

  public void setRemoteHost(String remoteHost) {
    this.remoteHost = remoteHost;
  }

  public String getRemoteHost() {
    log.finer("getRemoteHost ####");
    return remoteHost;
  }

  public void setAttribute(String name, Object value) {
    log.finer("setAttribute ####");
    if (value != null) {
      this.attributes.put(name, value);
    } else {
      this.attributes.remove(name);
    }
  }

  public void removeAttribute(String name) {
    log.finer("removeAttribute ####");
    this.attributes.remove(name);
  }

  /**
   * Add a new preferred locale, before any existing locales.
   */
  public void addPreferredLocale(Locale locale) {
    this.locales.add(0, locale);
  }

  public Locale getLocale() {
    log.finer("getLocale ####");
    return (Locale) this.locales.get(0);
  }

  public Enumeration getLocales() {
    log.finer("getLocales ####");
    return this.locales.elements();
  }

  public void setSecure(boolean secure) {
    this.secure = secure;
  }

  public boolean isSecure() {
    return secure;
  }
  
  RequestDispatcher requestDispatcher = null;

  public RequestDispatcher getRequestDispatcher(String path) {
    log.finer("getRequestDispatcher ####");
    return requestDispatcher;
  }

  public String getRealPath(String path) {
    log.finer("getRealPath ####");
    return this.servletContext.getRealPath(path);
  }

  public void setRemotePort(int remotePort) {
    this.remotePort = remotePort;
  }

  public int getRemotePort() {
    log.finer("getRemotePort ####");
    return remotePort;
  }

  public void setLocalName(String localName) {
    this.localName = localName;
  }

  public String getLocalName() {
    log.finer("getLocalName ####");
    return localName;
  }

  public void setLocalAddr(String localAddr) {
    this.localAddr = localAddr;
  }

  public String getLocalAddr() {
    log.finer("getLocalAddr ####");
    return localAddr;
  }

  public void setLocalPort(int localPort) {
    this.localPort = localPort;
  }

  public int getLocalPort() {
    return localPort;
  }

  //---------------------------------------------------------------------
  // HttpServletRequest interface
  //---------------------------------------------------------------------
  public void setAuthType(String authType) {
    this.authType = authType;
  }

  public String getAuthType() {
    log.finer("getAuthType ####");
    return authType;
  }

  public void setCookies(Cookie[] cookies) {
    this.cookies = cookies;
  }

  public Cookie[] getCookies() {
    log.finer("getCookies ####");
    return cookies;
  }

  /**
   * Add a header entry for the given name.
   * <p>If there was no entry for that header name before,
   * the value will be used as-is. In case of an existing entry,
   * a String array will be created, adding the given value (more
   * specifically, its toString representation) as further element.
   * <p>Multiple values can only be stored as list of Strings,
   * following the Servlet spec (see <code>getHeaders</code> accessor).
   * As alternative to repeated <code>addHeader</code> calls for
   * individual elements, you can use a single call with an entire
   * array or Collection of values as parameter.
   * @see #getHeaderNames
   * @see #getHeader
   * @see #getHeaders
   * @see #getDateHeader
   * @see #getIntHeader
   */
  public void addHeader(String name, Object value) {
    if(name==null) throw new NullPointerException("name must not be null");
    if(value==null) throw new NullPointerException( "value must not be null");
    Object oldValue = this.headers.get(name);
    if (oldValue instanceof List) {
      List list = (List) oldValue;
      addHeaderValue(list, value);
    }
    else if (oldValue != null) {
      List list = new LinkedList();
      list.add(oldValue);
      addHeaderValue(list, value);
      this.headers.put(name, list);
    }
    else if (value instanceof Collection || value.getClass().isArray()) {
      List list = new LinkedList();
      addHeaderValue(list, value);
      this.headers.put(name, list);
    }
    else {
      headLog.info("490) " + name + ", " + value);
      this.headers.put(name, value);
    }
  }

  private void addHeaderValue(List list, Object value) {
    if (value instanceof Collection) {
      Collection valueColl = (Collection) value;
      for (Iterator it = valueColl.iterator(); it.hasNext();) {
        Object element = it.next();
        //l("Value collection must not contain null elements");
        list.add(element.toString());
      }
    }
    else if (value.getClass().isArray()) {
      int length = Array.getLength(value);
      for (int i = 0; i < length; i++) {
        Object element = Array.get(value, i);
        //Assert.notNull("Value collection must not contain null elements");
        list.add(element.toString());
      }
    }
    else {
      list.add(value);
    }
  }

  public long getDateHeader(String name) {
    log.finer("getDateHeader ####");
    if (name == null) throw new NullPointerException( "name must not be null");
    Object value = this.headers.get(name);
    if (value instanceof Date) {
      return ((Date) value).getTime();
    }
    else if (value instanceof Number) {
      return ((Number) value).longValue();
    }
    else if (value != null) {
      throw new IllegalArgumentException(
          "Value for header '" + name + "' is neither a Date nor a Number: " + value);
    }
    else {
      return -1L;
    }
  }

  public String getHeader(String name) {
    log.finer("getHeader ####");
    if(name==null) throw new NullPointerException( "name must not be null");
    Object value = this.headers.get(name);

    String cls = "null";
    if (value != null) cls = value.getClass().getName();
    headLog.fine("540) " + value + ", " + cls);
                 
    if (value instanceof List) {
  //    return StringUtils.collectionToCommaDelimitedString((List) value);
      throw new RuntimeException();
    }
    else if (value != null) {
      return value.toString();
    }
    else {
      return null;
    }
  }

  public Enumeration getHeaders(String name) {
    if (name==null) throw new NullPointerException( "name must not be null");
    Object value = this.headers.get(name);
    if (value instanceof List) {
      return Collections.enumeration((List) value);
    }
    else if (value != null) {
      Vector vector = new Vector(1);
      vector.add(value.toString());
      return vector.elements();
    }
    else {
      return Collections.enumeration(Collections.EMPTY_SET);
    }
  }

  public Enumeration getHeaderNames() {
    return this.headers.keys();
  }

  public int getIntHeader(String name) {
    if (name== null) throw new NullPointerException( "name must not be null");
    Object value = this.headers.get(name);
    if (value instanceof Number) {
      return ((Number) value).intValue();
    }
    else if (value != null) {
      throw new NumberFormatException("Value for header '" + name + "' is not a Number: " + value);
    }
    else {
      return -1;
    }
  }

  public void setMethod(String method) {
    this.method = method;
  }

  public String getMethod() {
    log.finer("getMethod return s ==> " + method);
    return method;
  }

  public void setPathInfo(String pathInfo) {
    this.pathInfo = pathInfo;
  }

  public String getPathInfo() {
    log.fine("getPathInfo return s ==> " + pathInfo);
    return pathInfo;
  }

  public String getPathTranslated() {
    return (this.pathInfo != null ? getRealPath(this.pathInfo) : null);
  }

  public void setContextPath(String contextPath) {
    this.contextPath = contextPath;
  }

  public String getContextPath() {
    log.finer("getContextPath return s ==> " + contextPath);
    return contextPath;
  }

  public void setQueryString(String queryString) {
    this.queryString = queryString;
  }

  public String getQueryString() {
    log.finer("getQueryString return s ==> " + queryString);
    return queryString;
  }

  public void setRemoteUser(String remoteUser) {
    this.remoteUser = remoteUser;
  }

  public String getRemoteUser() {
    log.finer("getRemoteUser return s ==> " + remoteUser);
    return remoteUser;
  }

  /**
   * @deprecated in favor of addUserRole
   * @see #addUserRole
   */
  public void addRole(String role) {
    addUserRole(role);
  }

  public void addUserRole(String role) {
    this.userRoles.add(role);
  }

  public boolean isUserInRole(String role) {
    return this.userRoles.contains(role);
  }

  public void setUserPrincipal(Principal userPrincipal) {
    this.userPrincipal = userPrincipal;
  }

  public Principal getUserPrincipal() {
    return userPrincipal;
  }

  public String getRequestedSessionId() {
    HttpSession session = this.getSession();
    return (session != null ? session.getId() : null);
  }

  public void setRequestURI(String requestURI) {
    this.requestURI = requestURI;
  }

  public String getRequestURI() {
    
    log.finer("getRequestURI returns => " + requestURI);
    return requestURI;
  }

  public StringBuffer getRequestURL() {
    StringBuffer url = new StringBuffer(this.scheme);
    url.append("://").append(this.serverName).append(':').append(this.serverPort);
    url.append(getRequestURI());
    return url;
  }

  public void setServletPath(String servletPath) {
    this.servletPath = servletPath;
  }

  public String getServletPath() {
    log.fine("getServletPath returns => " + servletPath);
    return servletPath;
  }

  public void setSession(HttpSession session) {
    this.session = session;
  }

  public HttpSession getSession(boolean create) {

    HttpSession ses = _getSession(create);
    log.finer("getSession create(" + create + ") => " + ses);
    return ses;
  }

  private HttpSession _getSession(boolean create) {

    if (session == null && create) {
      session = new MockHttpSession(); // ServletContextKv
    }
    return session;
  }
  
  public HttpSession getSession() {

    HttpSession ses = _getSession(true);
    log.finer("getSession create() => " + ses);
    return ses;
  }

  public void setRequestedSessionIdValid(boolean requestedSessionIdValid) {
    this.requestedSessionIdValid = requestedSessionIdValid;
  }

  public boolean isRequestedSessionIdValid() {
    return this.requestedSessionIdValid;
  }

  public void setRequestedSessionIdFromCookie(boolean requestedSessionIdFromCookie) {
    this.requestedSessionIdFromCookie = requestedSessionIdFromCookie;
  }

  public boolean isRequestedSessionIdFromCookie() {
    return this.requestedSessionIdFromCookie;
  }

  public void setRequestedSessionIdFromURL(boolean requestedSessionIdFromURL) {
    this.requestedSessionIdFromURL = requestedSessionIdFromURL;
  }

  public boolean isRequestedSessionIdFromURL() {
    return this.requestedSessionIdFromURL;
  }

  public boolean isRequestedSessionIdFromUrl() {
    return isRequestedSessionIdFromURL();
  }
}
