/*
 * Copyright 2002-2005 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * 
 * "Portions Copyrighted [2006] [Masahito Henmi]"
 * 
 * 
 */
package strawberry.mockup;

import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.util.logging.*;

/**
 * Mock implementation of the HttpSession interface.
 *
 * <p>Used for testing the web framework; also useful
 * for testing application controllers.
 *
 * @author Rod Johnson
 * @author Juergen Hoeller
 * @since 1.0.2
 */
public class MockHttpSession implements HttpSession {

  // jdk1.4 logging
  protected final static Logger log = Logger.getLogger("strawberry.ctx.session");
  
  public static final String SESSION_COOKIE_NAME = "JSESSION";

  private static int nextId = 1;

  private final String id = Integer.toString(nextId++);

  private final long creationTime = System.currentTimeMillis();

  private int maxInactiveInterval;

  private long lastAccessedTime = System.currentTimeMillis();

  private ServletContext servletContext;

  private final Hashtable attributes = new Hashtable();

  private boolean invalid = false;

  private boolean isNew = true;

  /**
   * Create a new MockHttpSession with a MockServletContext.
   * @see MockServletContext
   */
  public MockHttpSession() {
    //this(new MockServletContext()); 2007.6.14
  }


  public long getCreationTime() {
    log.finer("getCreationTime");
    return creationTime;
  }

  public String getId() {
    return id;
  }

  public void access() {
    log.finer("access");
    this.lastAccessedTime = System.currentTimeMillis();
    this.isNew = false;
  }

  public long getLastAccessedTime() {
    log.finer("getLastAccessedTime");
    return lastAccessedTime;
  }

  public ServletContext getServletContext() {
    if (servletContext == null) throw new NullPointerException();
    return servletContext;
  }

  public void setServletContext(ServletContext sc) {
    servletContext = sc;
  }

  public void setMaxInactiveInterval(int interval) {
    log.finer("setMaxInactiveInterval");
    maxInactiveInterval = interval;
  }

  public int getMaxInactiveInterval() {
    log.finer("getMaxInactiveInterval");
    return maxInactiveInterval;
  }

  /**
   * deprecated p~
   */
  public HttpSessionContext getSessionContext() {
    throw new UnsupportedOperationException("getSessionContext");
  }

  public Object getAttribute(String name) {
    log.finer("getAttribute");
    return this.attributes.get(name);
  }

  public Object getValue(String name) {
    log.finer("getValue");
    return getAttribute(name);
  }

  public Enumeration getAttributeNames() {
    log.finer("getAttributeNames");
    return this.attributes.keys();
  }

  public String[] getValueNames() {
    log.finer("getValueNames");
    return (String[]) this.attributes.keySet().toArray(new String[this.attributes.size()]);
  }

  public void setAttribute(String name, Object value) {
    log.finer("setAttribute");
    if (value != null) {
      this.attributes.put(name, value);
    }
    else {
      this.attributes.remove(name);
    }
  }

  public void putValue(String name, Object value) {
    log.finer("putValue");
    setAttribute(name, value);
  }

  public void removeAttribute(String name) {
    log.finer("removeAttribute");
    this.attributes.remove(name);
  }

  public void removeValue(String name) {
    log.finer("removeValue");
    removeAttribute(name);
  }

  public void invalidate() {
    log.finer("invalidate");
    this.invalid = true;
    this.attributes.clear();
  }

  public boolean isInvalid() {
    log.finer("isInvalid");
    return invalid;
  }

  public void setNew(boolean value) {
    log.finer("setNew");
    isNew = value;
  }

  public boolean isNew() {
    log.finer("isNew");
    return isNew;
  }

}
