/*  -*- Mode: java4; -*-
 * Copyright 2002-2005 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * 
 * "Portions Copyrighted [2007] [Masahito Henmi]"
 * 
 * 
 */
package strawberry.mockup;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.*;
import java.util.logging.*;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;
/**
 * Mock implementation of the HttpServletResponse interface.
 *
 * <p>Used for testing the web framework; also useful
 * for testing application controllers.
 *
 * @author Rod Johnson
 * @author Juergen Hoeller
 * @since 1.0.2
 */
public class MockHttpServletResponse implements HttpServletResponse {

    // jdk1.4 logging
    protected final static Logger log = Logger.getLogger("strawberry.ctx.response");

    public static final int DEFAULT_SERVER_PORT = 80;


	//---------------------------------------------------------------------
	// ServletResponse properties
	//---------------------------------------------------------------------

	private String characterEncoding = "MS932"; //WebUtils.DEFAULT_CHARACTER_ENCODING;

	private final ByteArrayOutputStream content = new ByteArrayOutputStream();

	private final ServletOutputStream outputStream = null;//new DelegatingServletOutputStream(this.content);

	private PrintWriter writer;

	private int contentLength = 0;

	private String contentType;

	private int bufferSize = 4096;

	private boolean committed;

	private Locale locale = Locale.getDefault();


	//---------------------------------------------------------------------
	// HttpServletResponse properties
	//---------------------------------------------------------------------

	private final List cookies = new ArrayList();

	private final Map headers = new HashMap();

	private int status = HttpServletResponse.SC_OK;

	private String errorMessage;

	private String redirectedUrl;

	private String forwardedUrl;

	private String includedUrl;


	//---------------------------------------------------------------------
	// ServletResponse interface
	//---------------------------------------------------------------------

        /** implements of HttpServletResponse */
	public void setCharacterEncoding(String characterEncoding) {
                log.fine("-------------------- setCharacterEncoding ");
		this.characterEncoding = characterEncoding;
	}

        /** implements of HttpServletResponse */
	public String getCharacterEncoding() {
                log.fine("-------------------- getCharacterEncoding ");
		return characterEncoding;
	}

        /** implements of HttpServletResponse */
	public ServletOutputStream getOutputStream() {
                log.fine("-------------------- getOutputStream ");
		return this.outputStream;
	}

        /** implements of HttpServletResponse */
	public PrintWriter getWriter() throws UnsupportedEncodingException {
                log.fine("-------------------- getWriter ");
		if (this.writer == null) {
			Writer targetWriter = (this.characterEncoding != null ?
					new OutputStreamWriter(this.content, this.characterEncoding) : new OutputStreamWriter(this.content));
			this.writer = new PrintWriter(targetWriter);
		}
		return writer;
	}

	public byte[] getContentAsByteArray() {
		flushBuffer();
		return this.content.toByteArray();
	}

	public String getContentAsString() throws UnsupportedEncodingException {
		flushBuffer();
		return (this.characterEncoding != null) ?
				this.content.toString(this.characterEncoding) : this.content.toString();
	}

        /** implements of HttpServletResponse */
	public void setContentLength(int contentLength) {
                log.fine("-------------------- setContentLength ");
		this.contentLength = contentLength;
	}

	public int getContentLength() {
		return contentLength;
	}

        /** implements of HttpServletResponse */
	public void setContentType(String contentType) {
                log.fine("-------------------- setContentType '" + contentType + "'");
		this.contentType = contentType;
	}

        /** implements of HttpServletResponse */
	public String getContentType() {
                log.fine("-------------------- getContentType ");
		return contentType;
	}

        /** implements of HttpServletResponse */
	public void setBufferSize(int bufferSize) {
                log.fine("-------------------- setBufferSize ");
		this.bufferSize = bufferSize;
	}

        /** implements of HttpServletResponse */
	public int getBufferSize() {
                log.fine("-------------------- getBufferSize ");
		return bufferSize;
	}

        /** implements of HttpServletResponse */
	public void flushBuffer() {
                log.fine("-------------------- flushBuffer ");
		if (this.writer != null) {
			this.writer.flush();
		}
		if (this.outputStream != null) {
			try {
				this.outputStream.flush();
			}
			catch (IOException ex) {
				throw new IllegalStateException("Could not flush OutputStream: " + ex.getMessage());
			}
		}
		this.committed = true;
	}

        /** implements of HttpServletResponse */
	public void resetBuffer() {
                log.fine("-------------------- resetBuffer ");
		if (this.committed) {
			throw new IllegalStateException("Cannot reset buffer - response is already committed");
		}
		this.content.reset();
	}

	public void setCommitted(boolean committed) {
		this.committed = committed;
	}

        /** implements of HttpServletResponse */
	public boolean isCommitted() {
                log.fine("-------------------- isCommitted ");
		return committed;
	}

        /** implements of HttpServletResponse */
	public void reset() {
                log.fine("-------------------- reset ");
		resetBuffer();
		this.characterEncoding = null;
		this.contentLength = 0;
		this.contentType = null;
		this.locale = null;
		this.cookies.clear();
		this.headers.clear();
		this.status = HttpServletResponse.SC_OK;
		this.errorMessage = null;
	}

        /** implements of HttpServletResponse */
	public void setLocale(Locale locale) {
                log.fine("-------------------- setLocale ");
		this.locale = locale;
	}

        /** implements of HttpServletResponse */
	public Locale getLocale() {
                log.fine("-------------------- getLocale ");
		return locale;
	}


	//---------------------------------------------------------------------
	// HttpServletResponse interface
	//---------------------------------------------------------------------

        /** implements of HttpServletResponse */
	public void addCookie(Cookie cookie) {
                log.fine("-------------------- addCookie ");
		this.cookies.add(cookie);
	}

	public Cookie[] getCookies() {
		return (Cookie[]) this.cookies.toArray(new Cookie[this.cookies.size()]);
	}

	public Cookie getCookie(String name) {
		for (Iterator it = this.cookies.iterator(); it.hasNext();) {
			Cookie cookie = (Cookie) it.next();
			if (name.equals(cookie.getName())) {
				return cookie;
			}
		}
		return null;
	}

        /** implements of HttpServletResponse */
	public boolean containsHeader(String name) {
                log.fine("-------------------- containsHeader ");
		return this.headers.containsKey(name);
	}

	public Object getHeader(String name) {
		return this.headers.get(name);
	}

	public List getHeaders(String name) {
		Object value = this.headers.get(name);
		if (value instanceof List) {
			return (List) value;
		}
		else if (value != null) {
			return Collections.singletonList(value);
		}
		else {
			return Collections.EMPTY_LIST;
		}
	}

	public Set getHeaderNames() {
		return this.headers.keySet();
	}

        /** implements of HttpServletResponse */
	public String encodeURL(String url) {
                log.fine("-------------------- encodeURL ");
		return url;
	}

        /** implements of HttpServletResponse */
	public String encodeRedirectURL(String url) {
                log.fine("-------------------- encodeRedirectURL ");
		return url;
	}

        /** deprecation of HttpServletResponse */
	public String encodeUrl(String url) {
                log.fine("-------------------- encodeUrl ");
		return url;
	}

        /** deprecation of HttpServletResponse */
	public String encodeRedirectUrl(String url) {
                log.fine("-------------------- encodeRedirectUrl ");
		return url;
	}

        /** implements of HttpServletResponse */
	public void sendError(int status, String errorMessage) throws IOException {
                log.fine("-------------------- sendError ");
		if (this.committed) {
			throw new IllegalStateException("Cannot set error status - response is already committed");
		}
		this.status = status;
		this.errorMessage = errorMessage;
		this.committed = true;
	}

        /** implements of HttpServletResponse */
	public void sendError(int status) throws IOException {
                log.fine("-------------------- sendError ");
		if (this.committed) {
			throw new IllegalStateException("Cannot set error status - response is already committed");
		}
		this.status = status;
		this.committed = true;
	}

    
        /** implements of HttpServletResponse */
	public void sendRedirect(String url) throws IOException {
                log.fine("-------------------- sendRedirect ");
		if (this.committed) {
			throw new IllegalStateException("Cannot send redirect - response is already committed");
		}
		this.redirectedUrl = url;
		this.committed = true;
	}

	public String getRedirectedUrl() {
		return redirectedUrl;
	}

        /** implements of HttpServletResponse */
	public void setDateHeader(String name, long value) {
                log.fine("-------------------- setDateHeader ");
		this.headers.put(name, new Long(value));
	}

        /** implements of HttpServletResponse */
	public void addDateHeader(String name, long value) {
                log.fine("-------------------- addDateHeader ");
		doAddHeader(name, new Long(value));
	}

        /** implements of HttpServletResponse */
	public void setHeader(String name, String value) {
                log.fine("-------------------- setHeader ");
		this.headers.put(name, value);
	}

        /** implements of HttpServletResponse */
	public void addHeader(String name, String value) {
                log.fine("-------------------- addHeader ");
		doAddHeader(name, value);
	}

        /** implements of HttpServletResponse */
	public void setIntHeader(String name, int value) {
                log.fine("-------------------- setIntHeader ");
		this.headers.put(name, new Integer(value));
	}

        /** implements of HttpServletResponse */
	public void addIntHeader(String name, int value) {
                log.fine("-------------------- addIntHeader ");
		doAddHeader(name, new Integer(value));
	}

	private void doAddHeader(String name, Object value) {
		if (name==null) throw new NullPointerException( "name must not be null");
		if(value==null) throw new NullPointerException( "value must not be null");
		Object oldValue = this.headers.get(name);
		if (oldValue instanceof List) {
			List list = (List) oldValue;
			list.add(value);
		}
		else if (oldValue != null) {
			List list = new LinkedList();
			list.add(oldValue);
			list.add(value);
			this.headers.put(name, list);
		}
		else {
			this.headers.put(name, value);
		}
	}

        /** implements of HttpServletResponse */
	public void setStatus(int status) {
                log.fine("-------------------- setStatus ");
		this.status = status;
	}

        /** deprecation of HttpServletResponse */
	public void setStatus(int status, String errorMessage) {
                log.fine("-------------------- setStatus ");
		this.status = status;
		this.errorMessage = errorMessage;
	}

	public int getStatus() {
		return status;
	}

	public String getErrorMessage() {
		return errorMessage;
	}


	/*---------------------------------------------------------------------
	// Methods for MockRequestDispatcher
	//---------------------------------------------------------------------

	public void setForwardedUrl(String forwardedUrl) {
		this.forwardedUrl = forwardedUrl;
	}

	public String getForwardedUrl() {
		return forwardedUrl;
	}

	public void setIncludedUrl(String includedUrl) {
		this.includedUrl = includedUrl;
	}

	public String getIncludedUrl() {
		return includedUrl;
	}
*/
}
