package org.herac.tuxguitar.gui.mixer;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.herac.tuxguitar.gui.TablatureEditor;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.actions.composition.ChangeInfoAction;
import org.herac.tuxguitar.gui.actions.track.ChangeTrackPropertiesAction;
import org.herac.tuxguitar.gui.actions.track.GoToTrackAction;
import org.herac.tuxguitar.gui.tab.MeasureCoords;
import org.herac.tuxguitar.gui.tab.SongTrackCoords;
import org.herac.tuxguitar.song.models.TrackColor;

public class SongMixer extends Composite{
	private TablatureEditor editor;
	
	private Table table;
	private Composite trackMeasures;
	private Button trackMeasuresHeader;
	
	public SongMixer(Composite parent,TablatureEditor editor) {
        super(parent,SWT.BORDER | SWT.V_SCROLL);
        this.setLayout(new FillLayout());
        this.editor = editor;
        this.init();
        this.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent arg0) {
				updateTable();
				updateScrolls();
				
				trackMeasures.redraw();
			}
		});
        final Point origin = new Point(0, 0);
        final ScrollBar vBar = getVerticalBar();
        vBar.addListener(SWT.Selection, new Listener() {
          public void handleEvent(Event e) {
            int vSelection = vBar.getSelection();
            int destY = -vSelection - origin.y;
            Rectangle rect = trackMeasures.getBounds();
            getShell().scroll(0, destY, 0, 0, rect.width, rect.height, false);
            origin.y = -vSelection;            
            table.getVerticalBar().setSelection(vSelection);
            trackMeasures.redraw();
          }
        });    
    }
	
	private void init(){
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0); 
		data.right = new FormAttachment(100, 0); 
		data.top = new FormAttachment(0, 0);
		data.bottom = new FormAttachment(100, 0);
		setLayoutData(data);
		
		SashForm sashForm = new SashForm(this, SWT.HORIZONTAL);
	    sashForm.setLayout(new FormLayout());
	    data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0,0);
		data.bottom = new FormAttachment(100, 0);
		setLayoutData(data);
		
		initTable(sashForm);
		initTrackMeasures(sashForm);
	}
	
	private void initTable(Composite parent){
		this.table = new Table(parent,SWT.NONE);
		this.table.getHorizontalBar().setVisible(true);
		this.table.getVerticalBar().setVisible(false);
		//this.table.setRedraw(false);
		this.table.setLinesVisible(true);
		this.table.setHeaderVisible(true);
	    for (int i = 0; i < 4; i++) {
	    	new TableColumn(this.table, SWT.NONE);
	    }
	    this.loadProperties();
	    
		this.updateTable();
		/*
		this.table.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				
				table.setData(table.getSelection()[0].getData());
				TuxGuitar.instance().getAction(GoToTrackAction.NAME).doAction(e);
			}
		});
		*/
	    
		this.table.addMouseListener(new MouseAdapter() {		
			public void mouseDown(MouseEvent e) {
				int index = getSelectedTrackIndex(e.y);
				if(index >= 0){
					TableItem item = table.getItem(index);
					table.setData(item.getData());
					TuxGuitar.instance().getAction(GoToTrackAction.NAME).doAction(e);
				}
			}
		
			public void mouseDoubleClick(MouseEvent e) {    
				int index = getSelectedTrackIndex(e.y);
				if(index >= 0){
					TuxGuitar.instance().getAction(ChangeTrackPropertiesAction.NAME).doAction(e);
				}
			}

		});
		
		this.table.getColumn(0).pack();
		this.table.getColumn(1).setWidth(200);
		this.table.getColumn(2).pack();
	}
	
	private void initTrackMeasures(Composite parent){
		this.trackMeasures = new Composite(parent,SWT.H_SCROLL);
		this.trackMeasuresHeader = new Button(this.trackMeasures,SWT.PUSH);
		this.trackMeasuresHeader.addMouseListener(new MouseAdapter() {
			public void mouseDoubleClick(MouseEvent e) {
				TuxGuitar.instance().getAction(ChangeInfoAction.NAME).doAction(e);
			}
		});
		
        final Point origin = new Point(0, 0);
        final ScrollBar hBar = this.trackMeasures.getHorizontalBar();
        hBar.addListener(SWT.Selection, new Listener() {
          public void handleEvent(Event e) {
            int hSelection = hBar.getSelection();
            int destX = -hSelection - origin.x;
            Rectangle rect = trackMeasures.getClientArea();
            trackMeasures.getShell().scroll(destX, 0, 0, 0, rect.width, rect.height, false);
            origin.x = -hSelection;           
            trackMeasures.redraw();
          }
        });

        
		this.trackMeasures.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				int hSelection = hBar.getSelection();
				int vSelection = table.getVerticalBar().getSelection();
				paintMeasures(e.gc,-hSelection,-vSelection);
			}
		
		});
		
		this.trackMeasures.addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				int trackIndex = getSelectedTrackIndex(e.y);
					if(trackIndex >= 0){
						int hSelection = hBar.getSelection();
						int itemSpan = table.getItemHeight();
						SongTrackCoords track = (SongTrackCoords)editor.getTablature().getSongCoords().getTrackCoords().get(trackIndex);
						List measures = track.getMeasuresCoords();
						int measureLength = measures.size();
						int trackX = - hSelection;
						for(int j = 0;j < measureLength;j++){
							if(e.x >= trackX && e.x < (trackX + itemSpan)){
								MeasureCoords measure = (MeasureCoords)measures.get(j);
								editor.getTablature().getCaret().moveTo(track,measure,measure.getFirstComponent());
								editor.getTablature().getSongManager().getPlayer().setTickPosition(measure.getMeasure().getStartWidthRepetitions());
								table.setSelection(trackIndex);
								TuxGuitar.instance().redraw();
								break;
							}
							trackX += itemSpan;
						}
						
					}
			}
		});
	}
	/*
	private void updateTable(){
		int selectionIndex = this.table.getSelectionIndex();
		
		this.table.removeAll();
	    
		List tracks = this.editor.getTablature().getSongCoords().getTrackCoords();
		for(int i = 0;i < tracks.size();i++){
			final SongTrackCoords track = (SongTrackCoords)tracks.get(i);
			String number = Integer.toString(i + 1);
			String name = track.getTrack().getName();
			String instrument = editor.getSongManager().getInstrumentName(track.getTrack().getInstrument());
	    	TableItem item = new TableItem(this.table, SWT.NONE);
	    	item.setText(new String[]{number,name,instrument});
	    	item.setData(track);
	    }

		
		this.table.setSelection(selectionIndex);
	}
*/
	private void updateTable(){	    
		List tracks = this.editor.getTablature().getSongCoords().getTrackCoords();
		int length = tracks.size();
		
		int itemCount = this.table.getItemCount();
		//agrego los item que falten
		for(int i = itemCount;i < length;i++){
			new TableItem(this.table, SWT.NONE);
		}
		//borro los item que sobren
		for(int i = itemCount;i > length;i--){
			this.table.remove(i - 1);
		}
		
		for(int i = 0;i < length;i++){
			final SongTrackCoords track = (SongTrackCoords)tracks.get(i);
			String number = Integer.toString(i + 1);
			String name = track.getTrack().getName();
			String instrument = editor.getSongManager().getInstrumentName(track.getTrack().getInstrument());
			TableItem item = this.table.getItem(i);
	    	item.setText(new String[]{number,name,instrument});
	    	item.setData(track);
	    }
;
	}
	
	
	
	private void paintMeasures(GC gc,int fromX,int fromY){
		int width = this.trackMeasures.getBounds().width;
		int headerHeight = table.getHeaderHeight();
		int itemSpan = table.getItemHeight();
		
		int x = fromX;
		int y = fromY + table.getHeaderHeight();

		//cabecera
		this.trackMeasuresHeader.setBounds(0,0,width,headerHeight);		
		this.trackMeasuresHeader.setText(this.editor.getSongManager().getSong().getName());
		
		boolean playing = this.editor.getSongManager().getPlayer().isRunning();
		List tracks = this.editor.getTablature().getSongCoords().getTrackCoords();
		int trackLength = tracks.size();
		for(int i = 0;i < trackLength;i++){
			SongTrackCoords track = (SongTrackCoords)tracks.get(i);
		
			gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_GRAY));
			gc.fillRectangle(0,y,width,itemSpan - 1);

			TrackColor trackColor = track.getTrack().getColor();
			Color color = new Color(getDisplay(),trackColor.getR(),trackColor.getG(),trackColor.getB());
			gc.setBackground(color);

			List measures = track.getMeasuresCoords();
			int measureLength = measures.size();
			for(int j = 0;j < measureLength;j++){
				MeasureCoords measure = (MeasureCoords)measures.get(j);
				gc.fillRectangle(x,y,itemSpan - 1,itemSpan - 1);
				
				if((playing && measure.isPlaying()) || (!playing && measure.hasCaret())){
					gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_BLACK));
					gc.fillRectangle(x + 4,y + 4,itemSpan - 9,itemSpan - 9);
					gc.setBackground(color);
				}
				
				x += itemSpan;
			}
			x = fromX;
			y += itemSpan;

			color.dispose();
		}
		gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_BLACK));
	}

	
	private int getSelectedTrackIndex(int y){
		int vSelection = table.getVerticalBar().getSelection();
		int headerHeight = table.getHeaderHeight();
		int itemSpan = table.getItemHeight();
		int trackIndex = -1;
		if(y > headerHeight){
			List tracks = editor.getTablature().getSongCoords().getTrackCoords();
			int trackLength = tracks.size();
			int trackY = headerHeight - vSelection;
			for(int i = 0;i < trackLength;i++){
				if(y >= trackY && y < (trackY + itemSpan)){
					trackIndex = i;
					break;
				}
				trackY += itemSpan;
			}
		}
		return trackIndex;
	}
	
    public void updateScrolls(){
		int height = table.getHeaderHeight() + (table.getItemCount() * table.getItemHeight());
		if(trackMeasures.getHorizontalBar().isVisible() || table.getHorizontalBar().isVisible()){
			height += 20;
		}
		Rectangle client = getClientArea();        
        ScrollBar vBar = getVerticalBar();
		vBar.setMaximum(height);
		vBar.setThumb(Math.min(height ,client.height));
		
		int width = (editor.getTablature().getCaret().getSongTrackCoords().getMeasuresCoords().size() * table.getItemHeight());
		client = trackMeasures.getClientArea();  
		ScrollBar hBar = trackMeasures.getHorizontalBar();
		hBar.setMaximum(width);
		hBar.setThumb(Math.min(width ,client.width));
		
    }
    
    
    public void loadProperties(){
        this.table.getColumn(0).setText(TuxGuitar.getProperty("track.number"));
	    this.table.getColumn(1).setText(TuxGuitar.getProperty("track.name"));
	    this.table.getColumn(2).setText(TuxGuitar.getProperty("track.instrument"));
    }
}
