/* -*- c-file-style: "gnu" -*- */

#include <senna.h>

#include <gcutter.h>
#include <glib/gstdio.h>

#include "sen-assertions.h"

void test_simple_exec(void);
void test_customized_tag(void);
void test_multi_conditions(void);
void test_invalid_result_index(void);
void test_html_mapping(void);

static sen_snip *snip;
static const gchar default_open_tag[] = "[[";
static const gchar default_close_tag[] = "]]";
static const gchar text[] =
  "Senna is an embeddable fulltext search engine, which you can use in\n"
  "conjunction with various scripting languages and databases. Senna is\n"
  "an inverted index based engine, & combines the best of n-gram\n"
  "indexing and word indexing to achieve fast, precise searches. While\n"
  "senna codebase is rather compact it is scalable enough to handle large\n"
  "amounts of data and queries.";
static gchar *result;

void
setup(void)
{
  snip = NULL;
  result = NULL;
}

void
teardown(void)
{
  if (snip) {
    sen_snip_close(snip);
  }
  if (result) {
    g_free(result);
  }
}

#define open_snip_full(encoding, flags, width, max_results,             \
                       default_open_tag, default_open_tag_len,          \
                       default_close_tag, default_close_tag_len,        \
                       mapping) do                                      \
{                                                                       \
  snip = sen_snip_open(encoding, flags, width, max_results,             \
                       default_open_tag, default_open_tag_len,          \
                       default_close_tag, default_close_tag_len,        \
                       mapping);                                        \
  cut_assert(snip);                                                     \
} while (0)

#define open_snip_with_mapping(mapping)                                 \
  open_snip_full(sen_enc_default, 0, 100, 10,                           \
                 default_open_tag, strlen(default_open_tag),            \
                 default_close_tag, strlen(default_close_tag),          \
                 mapping)

#define open_snip()                                                     \
  open_snip_with_mapping(NULL)

#define open_html_mapping_snip()                                        \
  open_snip_with_mapping((sen_snip_mapping *)-1)

void
test_simple_exec(void)
{
  unsigned int n_results;
  unsigned int max_tagged_len;
  unsigned int result_len;
  const gchar keyword[] = "Senna";

  open_snip();
  cut_assert_sen(sen_snip_add_cond(snip, keyword, strlen(keyword),
                                   NULL, 0, NULL, 0));

  cut_assert_sen(sen_snip_exec(snip, text, strlen(text),
                               &n_results, &max_tagged_len));
  cut_assert_equal_uint(2, n_results);
  cut_assert_equal_uint(105, max_tagged_len);
  result = g_new(gchar, max_tagged_len);

  cut_assert_sen(sen_snip_get_result(snip, 0, result, &result_len));
  cut_assert_equal_string("[[Senna]] is an embeddable fulltext search engine, "
                          "which you can use in\n"
                          "conjunction with various scripti",
                          result);
  cut_assert_equal_uint(104, result_len);

  cut_assert_sen(sen_snip_get_result(snip, 1, result, &result_len));
  cut_assert_equal_string("ng languages and databases. [[Senna]] is\n"
                          "an inverted index based engine, & combines "
                          "the best of n-gram\ni",
                          result);
  cut_assert_equal_uint(104, result_len);
}

void
test_customized_tag(void)
{
  const gchar open_tag[] = "((*";
  const gchar close_tag[] = "*))";
  unsigned int n_results;
  unsigned int max_tagged_len;
  unsigned int result_len;
  const gchar keyword[] = "engine";

  open_snip();
  cut_assert_sen(sen_snip_add_cond(snip, keyword, strlen(keyword),
                                   open_tag, strlen(open_tag),
                                   close_tag, strlen(close_tag)));

  cut_assert_sen(sen_snip_exec(snip, text, strlen(text),
                               &n_results, &max_tagged_len));
  cut_assert_equal_uint(2, n_results);
  cut_assert_equal_uint(107, max_tagged_len);
  result = g_new(gchar, max_tagged_len);

  cut_assert_sen(sen_snip_get_result(snip, 0, result, &result_len));
  cut_assert_equal_string("Senna is an embeddable fulltext search "
                          "((*engine*)), which you can use in\n"
                          "conjunction with various scripti",
                          result);
  cut_assert_equal_uint(106, result_len);

  cut_assert_sen(sen_snip_get_result(snip, 1, result, &result_len));
  cut_assert_equal_string("nd databases. Senna is\n"
                          "an inverted index based ((*engine*)), "
                          "& combines the best of n-gram\n"
                          "indexing and wo",
                          result);
  cut_assert_equal_uint(106, result_len);
}

void
test_multi_conditions(void)
{
  const gchar open_tag[] = "((*";
  const gchar close_tag[] = "*))";
  unsigned int n_results;
  unsigned int max_tagged_len;
  unsigned int result_len;
  const gchar keyword1[] = "fulltext";
  const gchar keyword2[] = "senna";

  open_snip();
  cut_assert_sen(sen_snip_add_cond(snip, keyword1, strlen(keyword1),
                                   open_tag, strlen(open_tag),
                                   close_tag, strlen(close_tag)));
  cut_assert_sen(sen_snip_add_cond(snip, keyword2, strlen(keyword2),
                                   NULL, 0, NULL, 0));

  cut_assert_sen(sen_snip_exec(snip, text, strlen(text),
                               &n_results, &max_tagged_len));
  cut_assert_equal_uint(2, n_results);
  cut_assert_equal_uint(107, max_tagged_len);
  result = g_new(gchar, max_tagged_len);

  cut_assert_sen(sen_snip_get_result(snip, 0, result, &result_len));
  cut_assert_equal_string("Senna is an embeddable ((*fulltext*)) search "
                          "engine, which you can use in\n"
                          "conjunction with various scripti",
                          result);
  cut_assert_equal_uint(106, result_len);

  cut_assert_sen(sen_snip_get_result(snip, 1, result, &result_len));
  cut_assert_equal_string("dexing to achieve fast, precise searches. While\n"
                          "[[senna]] codebase is rather compact it is "
                          "scalable enou",
                          result);
  cut_assert_equal_uint(104, result_len);
}

void
test_invalid_result_index(void)
{
  unsigned int n_results;
  unsigned int max_tagged_len;
  unsigned int result_len;
  const gchar keyword[] = "index";

  open_snip();
  cut_assert_sen(sen_snip_add_cond(snip, keyword, strlen(keyword),
                                   NULL, 0, NULL, 0));

  cut_assert_sen(sen_snip_exec(snip, text, strlen(text),
                               &n_results, &max_tagged_len));
  cut_assert_equal_uint(1, n_results);
  cut_assert_equal_uint(113, max_tagged_len);
  result = g_new(gchar, max_tagged_len);

  cut_assert_sen(sen_snip_get_result(snip, 0, result, &result_len));
  cut_assert_equal_sen_rc(sen_invalid_argument,
                          sen_snip_get_result(snip, 1, result, &result_len));
}

void
test_html_mapping(void)
{
  const gchar open_tag[] = "<<";
  unsigned int n_results;
  unsigned int max_tagged_len;
  unsigned int result_len;
  const gchar keyword[] = "indexing";

  open_html_mapping_snip();
  cut_assert_sen(sen_snip_add_cond(snip, keyword, strlen(keyword),
                                   open_tag, strlen(open_tag), NULL, 0));

  cut_assert_sen(sen_snip_exec(snip, text, strlen(text),
                               &n_results, &max_tagged_len));
  cut_assert_equal_uint(1, n_results);
  cut_assert_equal_uint(113, max_tagged_len);
  result = g_new(gchar, max_tagged_len);

  cut_assert_sen(sen_snip_get_result(snip, 0, result, &result_len));
  cut_assert_equal_string("ngine, &amp; combines the best of n-gram\n"
                          "<<indexing]] and word <<indexing]] to achieve fast, "
                          "precise searches. W",
                          result);
  cut_assert_equal_uint(112, result_len);
}
