/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include <stdio.h>
#include <string.h>
#include "sym.h"
#include "inv.h"
#include "str.h"
#include "set.h"
#include "lex.h"
#include "cache.h"
#include "store.h"
#include "ctx.h"
#include "com.h"

/* sen_index */

inline static int
build_flags(int flags)
{
  if (flags & SEN_INDEX_ENABLE_SUFFIX_SEARCH) {
    return flags | SEN_SYM_WITH_SIS;
  } else if (flags & SEN_INDEX_DISABLE_SUFFIX_SEARCH) {
    return flags & ~SEN_SYM_WITH_SIS;
  } else {   /* default */
    switch (flags & SEN_INDEX_TOKENIZER_MASK) {
    case SEN_INDEX_MORPH_ANALYSE :
      return flags | SEN_SYM_WITH_SIS;
    case SEN_INDEX_NGRAM :
      return flags & ~SEN_SYM_WITH_SIS;
    case SEN_INDEX_DELIMITED :
      return flags & ~SEN_SYM_WITH_SIS;
    default :
      return flags & ~SEN_SYM_WITH_SIS;
    }
  }
}

void
sen_index_expire(void)
{
  sen_inv_expire();
}

sen_index *
sen_index_create(const char *path, int key_size,
                 int flags, int initial_n_segments, sen_encoding encoding)
{
  sen_index *i;
  char buffer[PATH_MAX];
  if (!path) { SEN_LOG(sen_log_warning, "sen_index_create: invalid argument"); return NULL; }
  if (initial_n_segments == 0) { initial_n_segments = SENNA_DEFAULT_INITIAL_N_SEGMENTS; }
  if (encoding == sen_enc_default) { encoding = sen_gctx.encoding; }
  if (strlen(path) > PATH_MAX - 4) {
    SEN_LOG(sen_log_warning, "sen_index_create: too long index path (%s)", path);
    return NULL;
  }
  if (!(i = SEN_GMALLOC(sizeof(sen_index)))) { return NULL; }
  SEN_LOG(sen_log_notice, "creating '%s' encoding=%s initial_n_segments=%d",
          path, sen_enctostr(encoding), initial_n_segments);
  strcpy(buffer, path);
  strcat(buffer, ".SEN");
  i->foreign_flags = 0;
  if ((i->keys = sen_sym_create(buffer, key_size, (flags & 0x70000), sen_enc_none))) {
    strcpy(buffer, path);
    strcat(buffer, ".SEN.l");
    if ((i->lexicon = sen_sym_create(buffer, 0, build_flags(flags), encoding))) {
      strcpy(buffer, path);
      strcat(buffer, ".SEN.i");
      if ((i->inv = sen_inv_create(buffer, i->lexicon, initial_n_segments))) {
        if ((flags & SEN_INDEX_WITH_VGRAM)) {
          strcpy(buffer, path);
          strcat(buffer, ".SEN.v");
          i->vgram= sen_vgram_create(buffer);
        } else {
          i->vgram = NULL;
        }
        if (!(flags & SEN_INDEX_WITH_VGRAM) || i->vgram) {
          SEN_LOG(sen_log_notice, "index created (%s) flags=%x", path, i->lexicon->flags);
          return i;
        }
        sen_inv_close(i->inv);
      }
      sen_sym_close(i->lexicon);
    }
    sen_sym_close(i->keys);
  }
  SEN_GFREE(i);
  return NULL;
}

sen_index *
sen_index_open(const char *path)
{
  sen_index *i;
  char buffer[PATH_MAX];
  if (!path) { SEN_LOG(sen_log_warning, "sen_index_open: invalid argument"); return NULL; }
  if (strlen(path) > PATH_MAX - 4) {
    SEN_LOG(sen_log_warning, "sen_index_open: too long index path (%s)", path);
    return NULL;
  }
  if (!(i = SEN_GMALLOC(sizeof(sen_index)))) { return NULL; }
  strcpy(buffer, path);
  strcat(buffer, ".SEN");
  i->foreign_flags = 0;
  if ((i->keys = sen_sym_open(buffer))) {
    strcpy(buffer, path);
    strcat(buffer, ".SEN.l");
    if ((i->lexicon = sen_sym_open(buffer))) {
      strcpy(buffer, path);
      strcat(buffer, ".SEN.i");
      if ((i->inv = sen_inv_open(buffer, i->lexicon))) {
        if ((i->lexicon->flags & SEN_INDEX_WITH_VGRAM)) {
          strcpy(buffer, path);
          strcat(buffer, ".SEN.v");
          i->vgram = sen_vgram_open(buffer);
        } else {
          i->vgram = NULL;
        }
        if (!(i->lexicon->flags & SEN_INDEX_WITH_VGRAM) || i->vgram) {
          SEN_LOG(sen_log_notice, "index opened (%p:%s) flags=%x", i, path, i->lexicon->flags);
          return i;
        }
        sen_inv_close(i->inv);
      }
      sen_sym_close(i->lexicon);
    }
    sen_sym_close(i->keys);
  }
  SEN_GFREE(i);
  return NULL;
}

#define FOREIGN_KEY     1
#define FOREIGN_LEXICON 2

sen_index *
sen_index_create_with_keys(const char *path, sen_sym *keys,
                           int flags, int initial_n_segments, sen_encoding encoding)
{
  sen_index *i;
  char buffer[PATH_MAX];
  if (!path || !keys) {
    SEN_LOG(sen_log_warning, "sen_index_create_with_keys: invalid argument");
    return NULL;
  }
  if (initial_n_segments == 0) { initial_n_segments = SENNA_DEFAULT_INITIAL_N_SEGMENTS; }
  if (encoding == sen_enc_default) { encoding = sen_gctx.encoding; }
  if (strlen(path) > PATH_MAX - 4) {
    SEN_LOG(sen_log_warning, "too long index path (%s)", path);
    return NULL;
  }
  if (!(i = SEN_GMALLOC(sizeof(sen_index)))) { return NULL; }
  SEN_LOG(sen_log_notice, "creating '%s' encoding=%s initial_n_segments=%d",
          path, sen_enctostr(encoding), initial_n_segments);
  i->keys = keys;
  i->foreign_flags = FOREIGN_KEY;
  strcpy(buffer, path);
  strcat(buffer, ".SEN.l");
  if ((i->lexicon = sen_sym_create(buffer, 0, build_flags(flags), encoding))) {
    strcpy(buffer, path);
    strcat(buffer, ".SEN.i");
    if ((i->inv = sen_inv_create(buffer, i->lexicon, initial_n_segments))) {
      if ((flags & SEN_INDEX_WITH_VGRAM)) {
        strcpy(buffer, path);
        strcat(buffer, ".SEN.v");
        i->vgram= sen_vgram_create(buffer);
      } else {
        i->vgram = NULL;
      }
      if (!(flags & SEN_INDEX_WITH_VGRAM) || i->vgram) {
        SEN_LOG(sen_log_notice, "index created (%s) flags=%x", path, i->lexicon->flags);
        return i;
      }
      sen_inv_close(i->inv);
    }
    sen_sym_close(i->lexicon);
  }
  SEN_GFREE(i);
  return NULL;
}

sen_index *
sen_index_open_with_keys(const char *path, sen_sym *keys)
{
  sen_index *i;
  char buffer[PATH_MAX];
  if (!path || !keys) {
    SEN_LOG(sen_log_warning, "sen_index_open_with_keys: invalid argument");
    return NULL;
  }
  if (strlen(path) > PATH_MAX - 4) {
    SEN_LOG(sen_log_warning, "too long index path (%s)", path);
    return NULL;
  }
  if (!(i = SEN_GMALLOC(sizeof(sen_index)))) { return NULL; }
  i->keys = keys;
  i->foreign_flags = FOREIGN_KEY;
  strcpy(buffer, path);
  strcat(buffer, ".SEN.l");
  if ((i->lexicon = sen_sym_open(buffer))) {
    strcpy(buffer, path);
    strcat(buffer, ".SEN.i");
    if ((i->inv = sen_inv_open(buffer, i->lexicon))) {
      if ((i->lexicon->flags & SEN_INDEX_WITH_VGRAM)) {
        strcpy(buffer, path);
        strcat(buffer, ".SEN.v");
        i->vgram = sen_vgram_open(buffer);
      } else {
        i->vgram = NULL;
      }
      if(!(i->lexicon->flags & SEN_INDEX_WITH_VGRAM) || i->vgram) {
        SEN_LOG(sen_log_notice, "index opened (%p:%s) flags=%x", i, path, i->lexicon->flags);
        return i;
      }
      sen_inv_close(i->inv);
    }
    sen_sym_close(i->lexicon);
  }
  SEN_GFREE(i);
  return NULL;
}

sen_index *
sen_index_create_with_keys_lexicon(const char *path, sen_sym *keys, sen_sym *lexicon,
                                   int initial_n_segments)
{
  sen_index *i;
  if (!keys || !path || !lexicon) {
    SEN_LOG(sen_log_warning, "sen_index_create_with_keys_lexicon: invalid argument");
    return NULL;
  }
  if (initial_n_segments == 0) { initial_n_segments = SENNA_DEFAULT_INITIAL_N_SEGMENTS; }
  if (!(i = SEN_GMALLOC(sizeof(sen_index)))) { return NULL; }
  SEN_LOG(sen_log_notice, "creating '%s' encoding=%s initial_n_segments=%d",
          path, sen_enctostr(lexicon->encoding), initial_n_segments);
  i->keys = keys;
  i->lexicon = lexicon;
  i->foreign_flags = FOREIGN_KEY|FOREIGN_LEXICON;
  i->vgram = NULL;
  if ((i->inv = sen_inv_create(path, i->lexicon, initial_n_segments))) {
    SEN_LOG(sen_log_notice, "index created (%s) flags=%x", path, i->lexicon->flags);
    return i;
  }
  SEN_GFREE(i);
  return NULL;
}

sen_index *
sen_index_open_with_keys_lexicon(const char *path, sen_sym *keys, sen_sym *lexicon)
{
  sen_index *i;
  if (!keys || !path || !lexicon) {
    SEN_LOG(sen_log_warning, "sen_index_open_with_keys_lexicon: invalid argument");
    return NULL;
  }
  if (!(i = SEN_GMALLOC(sizeof(sen_index)))) { return NULL; }
  i->keys = keys;
  i->lexicon = lexicon;
  i->foreign_flags = FOREIGN_KEY|FOREIGN_LEXICON;
  i->vgram = NULL;
  if ((i->inv = sen_inv_open(path, i->lexicon))) {
    SEN_LOG(sen_log_notice, "index opened (%p:%s) flags=%x", i, path, i->lexicon->flags);
    return i;
  }
  SEN_GFREE(i);
  return NULL;
}

sen_rc
sen_index_close(sen_index *i)
{
  if (!i) { return sen_invalid_argument; }
  if (!(i->foreign_flags & FOREIGN_KEY)) { sen_sym_close(i->keys); }
  if (!(i->foreign_flags & FOREIGN_LEXICON)) { sen_sym_close(i->lexicon); }
  sen_inv_close(i->inv);
  if (i->vgram) { sen_vgram_close(i->vgram); }
  SEN_GFREE(i);
  return sen_success;
}

sen_rc
sen_index_remove(const char *path)
{
  sen_rc rc;
  char buffer[PATH_MAX];
  if (!path || strlen(path) > PATH_MAX - 8) { return sen_invalid_argument; }
  snprintf(buffer, PATH_MAX, "%s.SEN", path);
  if ((rc = sen_sym_remove(buffer))) { goto exit; }
  snprintf(buffer, PATH_MAX, "%s.SEN.i", path);
  if ((rc = sen_inv_remove(buffer))) { goto exit; }
  snprintf(buffer, PATH_MAX, "%s.SEN.l", path);
  if ((rc = sen_sym_remove(buffer))) { goto exit; }
  snprintf(buffer, PATH_MAX, "%s.SEN.v", path);
  sen_io_remove(buffer); // sen_vgram_remove
exit :
  return rc;
}

sen_rc
sen_index_rename(const char *old_name, const char *new_name)
{
  char old_buffer[PATH_MAX];
  char new_buffer[PATH_MAX];
  if (!old_name || strlen(old_name) > PATH_MAX - 8) { return sen_invalid_argument; }
  if (!new_name || strlen(new_name) > PATH_MAX - 8) { return sen_invalid_argument; }
  snprintf(old_buffer, PATH_MAX, "%s.SEN", old_name);
  snprintf(new_buffer, PATH_MAX, "%s.SEN", new_name);
  sen_io_rename(old_buffer, new_buffer);
  snprintf(old_buffer, PATH_MAX, "%s.SEN.i", old_name);
  snprintf(new_buffer, PATH_MAX, "%s.SEN.i", new_name);
  sen_io_rename(old_buffer, new_buffer);
  snprintf(old_buffer, PATH_MAX, "%s.SEN.i.c", old_name);
  snprintf(new_buffer, PATH_MAX, "%s.SEN.i.c", new_name);
  sen_io_rename(old_buffer, new_buffer);
  snprintf(old_buffer, PATH_MAX, "%s.SEN.l", old_name);
  snprintf(new_buffer, PATH_MAX, "%s.SEN.l", new_name);
  sen_io_rename(old_buffer, new_buffer);
  snprintf(old_buffer, PATH_MAX, "%s.SEN.v", old_name);
  snprintf(new_buffer, PATH_MAX, "%s.SEN.v", new_name);
  sen_io_rename(old_buffer, new_buffer);
  return sen_success;
}

sen_rc
sen_index_info(sen_index *i, int *key_size, int *flags,
               int *initial_n_segments, sen_encoding *encoding,
               unsigned *nrecords_keys, unsigned *file_size_keys,
               unsigned *nrecords_lexicon, unsigned *file_size_lexicon,
               unsigned long long *inv_seg_size, unsigned long long *inv_chunk_size)
{
  sen_rc rc = sen_success;

  if (!i) { return sen_invalid_argument; }
  if (key_size) { *key_size = i->keys->key_size; }
  if (flags) { *flags = i->lexicon->flags & ~SEN_SYM_WITH_SIS; }
  if (initial_n_segments) { *initial_n_segments = sen_inv_initial_n_segments(i->inv); }
  if (encoding) { *encoding = i->lexicon->encoding; }
  if (nrecords_keys || file_size_keys) {
    if ((rc = sen_sym_info(i->keys, NULL, NULL, NULL, nrecords_keys, file_size_keys))) { return rc; }
  }
  if (nrecords_lexicon || file_size_lexicon) {
    if ((rc = sen_sym_info(i->lexicon, NULL, NULL, NULL, nrecords_lexicon, file_size_lexicon))) { return rc; }
  }
  if (inv_seg_size || inv_chunk_size) {
    uint64_t seg_size, chunk_size;

    rc = sen_inv_info(i->inv, &seg_size, &chunk_size);

    if (inv_seg_size) {
      *inv_seg_size = seg_size;
    }

    if (inv_chunk_size) {
      *inv_chunk_size = chunk_size;
    }

    if (rc != sen_success) {
      return rc;
    }
  }
  return sen_success;
}

sen_rc
sen_index_lock(sen_index *i, int timeout)
{
  if (!i) { return sen_invalid_argument; }
  return sen_sym_lock(i->keys, timeout);
}

sen_rc
sen_index_unlock(sen_index *i)
{
  if (!i) { return sen_invalid_argument; }
  return sen_sym_unlock(i->keys);
}

sen_rc
sen_index_clear_lock(sen_index *i)
{
  if (!i) { return sen_invalid_argument; }
  return sen_sym_clear_lock(i->keys);
}

int
sen_index_path(sen_index *i, char *pathbuf, int bufsize)
{
  const char *invpath;
  int pathsize;
  if (!i) {
    SEN_LOG(sen_log_warning, "sen_index_path: invalid argument");
    return sen_invalid_argument;
  }
  invpath = sen_io_path(i->lexicon->io);
  pathsize = strlen(invpath) - 5;
  if (bufsize >= pathsize && pathbuf) {
    memcpy(pathbuf, invpath, pathsize - 1);
    pathbuf[pathsize - 1] = '\0';
  }
  return pathsize;
}

sen_rc
sen_index_upd(sen_index *i, const void *key,
              const char *oldvalue, unsigned int oldvalue_len,
              const char *newvalue, unsigned int newvalue_len)
{
  sen_id rid;
  sen_rc rc = sen_invalid_argument;
  if (!i || !key) {
    SEN_LOG(sen_log_warning, "sen_index_upd: invalid argument");
    return sen_invalid_argument;
  }
  if ((rc = sen_index_lock(i, -1))) {
    SEN_LOG(sen_log_crit, "sen_index_upd: index lock failed");
    return rc;
  }
  if (oldvalue && *oldvalue) {
    if (!(rid = sen_sym_at(i->keys, key))) {
      SEN_LOG(sen_log_error, "del : (%x) (invalid key)", key);
      goto exit;
    }
  } else if (newvalue && *newvalue) {
    if (!(rid = sen_sym_get(i->keys, key))) { goto exit; }
  } else {
    goto exit;
  }
  rc = sen_inv_upd(i->inv, rid, i->vgram, oldvalue, oldvalue_len, newvalue, newvalue_len);
exit :
  sen_index_unlock(i);
  return rc;
}

#define DELETE_FLAG 1

sen_rc
sen_index_del(sen_index *i, const void *key)
{
  sen_id rid;
  if (!i || !key) { SEN_LOG(sen_log_warning, "sen_index_del: invalid argument"); return sen_invalid_argument; }
  rid = sen_sym_at(i->keys, key);
  if (!rid) { return sen_invalid_argument; }
  return sen_sym_pocket_set(i->keys, rid, DELETE_FLAG);
}

#define INITIAL_VALUE_SIZE 1024

sen_values *
sen_values_open(void)
{
  sen_ctx *ctx = &sen_gctx; /* todo : replace it with the local ctx */
  sen_values *v = SEN_MALLOC(sizeof(sen_values));
  if (v) {
    v->n_values = 0;
    v->values = NULL;
  }
  return v;
}

sen_rc
sen_values_close(sen_values *v)
{
  sen_ctx *ctx = &sen_gctx; /* todo : replace it with the local ctx */
  if (!v) { return sen_invalid_argument; }
  if (v->values) { SEN_FREE(v->values); }
  SEN_FREE(v);
  return sen_success;
}

sen_rc
sen_values_add(sen_values *v, const char *str, unsigned int str_len, unsigned int weight)
{
  sen_ctx *ctx = &sen_gctx; /* todo : replace it with the local ctx */
  sen_value *vp;
  if (!v || !str) { SEN_LOG(sen_log_warning, "sen_values_add: invalid argument"); return sen_invalid_argument; }
  if (!(v->n_values & (INITIAL_VALUE_SIZE - 1))) {
    vp = SEN_REALLOC(v->values, sizeof(sen_value) * (v->n_values + INITIAL_VALUE_SIZE));
    SEN_LOG(sen_log_debug, "expanded values to %d,%p", v->n_values + INITIAL_VALUE_SIZE, vp);
    if (!vp) { return sen_memory_exhausted; }
    v->values = vp;
  }
  vp = &v->values[v->n_values];
  vp->str = str;
  vp->str_len = str_len;
  vp->weight = weight;
  v->n_values++;
  return sen_success;
}

sen_rc
sen_index_update(sen_index *i, const void *key, unsigned int section,
                 sen_values *oldvalues, sen_values *newvalues)
{
  sen_id rid;
  sen_rc rc = sen_invalid_argument;
  if (!i || !key) {
    SEN_LOG(sen_log_warning, "sen_index_update: invalid argument");
    return rc;
  }
  if ((rc = sen_index_lock(i, -1))) {
    SEN_LOG(sen_log_crit, "sen_index_update: index lock failed");
    return rc;
  }
  if (newvalues) {
    if (!(rid = sen_sym_get(i->keys, key))) { goto exit; }
  } else {
    if (!(rid = sen_sym_at(i->keys, key))) { goto exit; }
  }
  rc = sen_inv_update(i->inv, rid, i->vgram, section, oldvalues, newvalues);
exit :
  sen_index_unlock(i);
  return rc;
}

/* select */

sen_rc
sen_index_similar_search(sen_index *i, const char *string,
                         unsigned int string_len, sen_records *r,
                         sen_sel_operator op, sen_select_optarg *optarg)
{
  sen_rc rc;
  if (!i || !string || !r || !optarg) { return sen_invalid_argument; }
  r->keys = i->keys;
  optarg->max_size = sen_sym_size(i->keys) * sizeof(int);
  rc = sen_inv_similar_search(i->inv, string, string_len, r, op, optarg);
  sen_records_cursor_clear(r);
  return rc;
}

#define TERM_EXTRACT_EACH_POST 0
#define TERM_EXTRACT_EACH_TERM 1

sen_rc
sen_index_term_extract(sen_index *i, const char *string,
                       unsigned int string_len, sen_records *r,
                       sen_sel_operator op, sen_select_optarg *optarg)
{
  sen_rc rc;
  r->keys = i->keys;
  rc = sen_inv_term_extract(i->inv, string, string_len, r, op, optarg);
  sen_records_cursor_clear(r);
  if (!rc && optarg->max_interval == TERM_EXTRACT_EACH_POST) {
    sen_sort_optarg opt;
    opt.mode = sen_sort_ascending;
    opt.compar = NULL;
    opt.compar_arg = (void *)(intptr_t)r->key_size;
    sen_records_sort(r, 10000, &opt); /* todo : why 10000? */
  }
  return rc;
}

sen_rc
sen_index_select(sen_index *i, const char *string, unsigned int string_len,
                 sen_records *r, sen_sel_operator op, sen_select_optarg *optarg)
{
  sen_rc rc;
  r->keys = i->keys;
  if (optarg) { optarg->max_size = sen_sym_size(i->keys) * sizeof(int); }
  rc = sen_inv_select(i->inv, string, string_len, r, op, optarg);
  sen_records_cursor_clear(r);
  return rc;
}

sen_records *
sen_index_sel(sen_index *i, const char *string, unsigned int string_len)
{
  sen_records *r = sen_inv_sel(i->inv, string, string_len);
  if (r) { r->keys = i->keys; }
  return r;
}

/* sen_records_heap class */

struct _sen_records_heap {
  int n_entries;
  int n_bins;
  sen_records **bins;
  int limit;
  int curr;
  int dir;
  int (*compar)(sen_records *, sen_recordh *, sen_records *, sen_recordh *, void *);
  void *compar_arg;
};

inline static int
records_heap_cmp(sen_records_heap *h, sen_records *r1, sen_records *r2)
{
  sen_recordh *rh1 = (sen_recordh *)sen_records_curr_rec(r1);
  sen_recordh *rh2 = (sen_recordh *)sen_records_curr_rec(r2);
  if (!h->compar) {
    int off1, off2;
    if (h->compar_arg == (void *)-1) {
      off1 = (r1->key_size) / sizeof(int32_t);
      off2 = (r2->key_size) / sizeof(int32_t);
    } else {
      off1 = off2 = (int)(intptr_t)h->compar_arg;
    }
    return (((int32_t *)(rh2))[off2] - ((int32_t *)(rh1))[off1]) * h->dir > 0;
  }
  return h->compar(r1, rh1, r2, rh2, h->compar_arg) * h->dir > 0;
}

sen_records_heap *
sen_records_heap_open(int size, int limit, sen_sort_optarg *optarg)
{
  sen_ctx *ctx = &sen_gctx; /* todo : replace it with the local ctx */
  sen_records_heap *h = SEN_MALLOC(sizeof(sen_records_heap));
  if (!h) { return NULL; }
  h->bins = SEN_MALLOC(sizeof(sen_records *) * size);
  if (!h->bins) {
    SEN_FREE(h);
    return NULL;
  }
  h->n_entries = 0;
  h->n_bins = size;
  h->limit = limit;
  h->curr = 0;
  if (optarg) {
    h->dir = (optarg->mode == sen_sort_ascending) ? 1 : -1;
    h->compar = optarg->compar;
    h->compar_arg = optarg->compar_arg;
  } else {
    h->dir = -1;
    h->compar = NULL;
    h->compar_arg = (void *) -1;
  }
  return h;
}

sen_rc
sen_records_heap_add(sen_records_heap *h, sen_records *r)
{
  sen_ctx *ctx = &sen_gctx; /* todo : replace it with the local ctx */
  if (h->n_entries >= h->n_bins) {
    int size = h->n_bins * 2;
    sen_records **bins = SEN_REALLOC(h->bins, sizeof(sen_records *) * size);
    // sen_log("expanded sen_records_heap to %d,%p", size, bins);
    if (!bins) { return sen_memory_exhausted; }
    h->n_bins = size;
    h->bins = bins;
  }
  if (!sen_records_next(r, NULL, 0, NULL)) {
    sen_records_close(r);
    return sen_internal_error;
  }
  {
    int n, n2;
    sen_records *r2;
    n = h->n_entries++;
    while (n) {
      n2 = (n - 1) >> 1;
      r2 = h->bins[n2];
      if (records_heap_cmp(h, r, r2)) { break; }
      h->bins[n] = r2;
      n = n2;
    }
    h->bins[n] = r;
  }
  return sen_success;
}

int
sen_records_heap_next(sen_records_heap *h)
{
  if (!h || !h->n_entries) { return 0; }
  {
    sen_records *r = h->bins[0];
    if (!sen_records_next(r, NULL, 0, NULL)) {
      sen_records_close(r);
      r = h->bins[0] = h->bins[--h->n_entries];
    }
    {
      int n = 0, m = h->n_entries;
      if (m > 1) {
        for (;;) {
          int n1 = n * 2 + 1;
          int n2 = n1 + 1;
          sen_records *r1 = n1 < m ? h->bins[n1] : NULL;
          sen_records *r2 = n2 < m ? h->bins[n2] : NULL;
          if (r1 && records_heap_cmp(h, r, r1)) {
            if (r2 && records_heap_cmp(h, r, r2) && records_heap_cmp(h, r1, r2)) {
              h->bins[n] = r2;
              n = n2;
            } else {
              h->bins[n] = r1;
              n = n1;
            }
          } else {
            if (r2 && records_heap_cmp(h, r, r2)) {
              h->bins[n] = r2;
              n = n2;
            } else {
              h->bins[n] = r;
              break;
            }
          }
        }
      }
      h->curr++;
      return m;
    }
  }
}

sen_records *
sen_records_heap_head(sen_records_heap *h)
{
  return h->n_entries ? h->bins[0] : NULL;
}

sen_rc
sen_records_heap_close(sen_records_heap *h)
{
  sen_ctx *ctx = &sen_gctx; /* todo : replace it with the local ctx */
  int i;
  if (!h) { return sen_invalid_argument; }
  for (i = h->n_entries; i--;) { sen_records_close(h->bins[i]); }
  SEN_FREE(h->bins);
  SEN_FREE(h);
  return sen_success;
}

/* todo : config_path will be disappeared */
sen_rc
sen_info(char **version,
         char **configure_options,
         char **config_path,
         sen_encoding *default_encoding,
         unsigned int *initial_n_segments,
         unsigned int *partial_match_threshold)
{
  if (version) {
    *version = PACKAGE_VERSION;
  }
  if (configure_options) {
    *configure_options = CONFIGURE_OPTIONS;
  }
  if (default_encoding) {
    *default_encoding = sen_gctx.encoding;
  }
  if (initial_n_segments) {
    *initial_n_segments = SENNA_DEFAULT_INITIAL_N_SEGMENTS;
  }
  if (partial_match_threshold) {
    *partial_match_threshold = SENNA_DEFAULT_QUERY_ESCALATION_THRESHOLD;
  }
  return sen_success;
}
