/* Copyright(C) 2006-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "senna_in.h"
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "sym.h"
#include "ql.h"

static sen_obj *_native_method_records(sen_ctx *c, sen_obj *args, sen_ql_co *co);
static sen_obj *_native_method_object(sen_ctx *c, sen_obj *args, sen_ql_co *co);
static sen_obj *_native_method_void(sen_ctx *c, sen_obj *args, sen_ql_co *co);

inline static void
rec_obj_bind(sen_obj *obj, sen_records *rec, sen_id cls)
{
  obj->type = sen_ql_records;
  obj->flags = SEN_OBJ_NATIVE|SEN_OBJ_ALLOCATED;
  obj->class = cls;
  obj->u.r.records = rec;
  obj->u.o.func = _native_method_records;
}

inline static void
obj_obj_bind(sen_obj *obj, sen_id cls, sen_id self)
{
  obj->type = sen_ql_object;
  obj->class = cls;
  obj->u.o.self = self;
  obj->flags = SEN_OBJ_NATIVE;
  obj->u.o.func = _native_method_object;
}

sen_obj *
sen_ql_mk_obj(sen_ctx *c, sen_id cls, sen_id self)
{
  sen_obj *o = sen_obj_new(c);
  if (o) { obj_obj_bind(o, cls, self); }
  return o;
}

inline static sen_obj *
slot_value_obj(sen_ctx *c, sen_db_store *slot, sen_id id, const sen_obj *args, sen_obj *res)
{
  sen_id *ip;
  sen_obj *car;
  POP(car, args); // todo : delete when called with (())
  ip = NILP(car) ? sen_ra_at(slot->u.o.ra, id) : sen_ra_get(slot->u.o.ra, id);
  if (!ip) { return NIL; }
  if (!NILP(car)) {
    switch (car->type) {
    case sen_ql_object :
      if (car->class != slot->u.o.class) { return NIL; }
      *ip = car->u.o.self;
      break;
    case sen_ql_bulk :
      {
        sen_db_store *cls;
        if (!(cls = sen_db_store_by_id(slot->db, slot->u.o.class)) ||
            !(*ip = sen_sym_get(cls->u.c.keys, car->u.b.value))) {
          return NIL;
        }
      }
      break;
    default :
      return NIL;
      break;
    }
    // todo : trigger
  }
  if (!*ip) { return NIL; }
  if (!res) { res = sen_obj_new(c); }
  obj_obj_bind(res, slot->u.o.class, *ip);
  return res;
}

inline static sen_obj *
slot_value_ra(sen_ctx *c, sen_db_store *slot, sen_id id, const sen_obj *args, sen_obj *res)
{
  void *vp;
  sen_obj *car;
  POP(car, args); // todo : delete when called with (())
  vp = NILP(car) ? sen_ra_at(slot->u.f.ra, id) : sen_ra_get(slot->u.f.ra, id);
  if (!vp) { return NIL; }
  if (!NILP(car)) {
    switch (car->type) {
    case sen_ql_bulk :
      if (sizeof(int32_t) == slot->u.f.ra->header->element_size) {
        int32_t i = sen_atoi(car->u.b.value,
                             (char *)car->u.b.value + car->u.b.size, NULL);
        memcpy(vp, &i, sizeof(int32_t));
      } else {
        if (car->u.b.size != slot->u.f.ra->header->element_size) { return NIL; }
        memcpy(vp, car->u.b.value, car->u.b.size);
      }
      break;
    case sen_ql_int :
      // todo : use slot->u.f.class instead of slot->u.f.ra->header->element_size
      if (sizeof(int32_t) != slot->u.f.ra->header->element_size) { return NIL; }
      memcpy(vp, &car->u.i.i, sizeof(int32_t));
      break;
    default :
      return NIL;
    }
  // todo : trigger
  }
  if (!res) { res = sen_obj_new(c); }
  if (slot->u.f.ra->header->element_size == sizeof(int32_t)) {
    res->type = sen_ql_int;
    memcpy(&res->u.i.i, vp, sizeof(int32_t));
  } else {
    res->type = sen_ql_bulk;
    res->u.b.size = slot->u.f.ra->header->element_size;
    res->u.b.value = vp;
  }
  return res;
}

inline static sen_obj *
slot_value_ja(sen_ctx *c, sen_db_store *slot, sen_id id, const sen_obj *args, sen_obj *res)
{
  void *vp;
  uint32_t value_len;
  sen_obj *car;
  POP(car, args); // todo : delete when called with (())
  vp = (void *)sen_ja_ref(slot->u.v.ja, id, &value_len);
  // todo : unref
  if (NILP(car)) {
    if (!vp) { return NIL; }
    if (!res) { res = sen_obj_new(c); }
    res->type = sen_ql_bulk;
    res->u.b.size = value_len;
    res->u.b.value = vp;
    return res;
  } else {
    sen_db_trigger *t;
    // todo : support append and so on..
    if (!BULKP(car)) { return NIL; }
    if (value_len == car->u.b.size && !memcmp(vp, car->u.b.value, value_len)) {
      return car;
    }
    for (t = slot->triggers; t; t = t->next) {
      if (t->type == sen_db_before_update_trigger) {
        sen_db_store *index = t->target;
        if (sen_index_upd(index->u.i.index, _sen_sym_key(index->u.i.index->keys, id),
                          vp, value_len, car->u.b.value, car->u.b.size)) {
          SEN_LOG(sen_log_error, "sen_index_upd failed. id=%d key=(%s) id'=%d", id, _sen_sym_key(index->u.i.index->keys, id), sen_sym_at(index->u.i.index->keys, _sen_sym_key(index->u.i.index->keys, id)));
        }
      }
    }
    return sen_ja_put(slot->u.v.ja, id, car->u.b.value, car->u.b.size, 0) ? NIL : car;
  }
}

inline static sen_obj *
slot_value(sen_ctx *c, sen_db_store *slot, sen_id obj, sen_obj *args, sen_obj *res)
{
  switch (slot->type) {
  case sen_db_obj_slot :
    return slot_value_obj(c, slot, obj, args, res);
    break;
  case sen_db_ra_slot :
    return slot_value_ra(c, slot, obj, args, res);
    break;
  case sen_db_ja_slot :
    return slot_value_ja(c, slot, obj, args, res);
    break;
  case sen_db_idx_slot :
    {
      sen_records *rec;
      const char *key = _sen_sym_key(slot->u.i.index->lexicon, obj);
      if (!key) { return NIL; }
      if (!(rec = sen_index_sel(slot->u.i.index, key, strlen(key)))) {
        return NIL;
      }
      if (!res) { res = sen_obj_new(c); }
      rec_obj_bind(res, rec, slot->u.i.class);
      return res;
    }
    break;
  default :
    return NIL;
    break;
  }
}

inline static sen_obj *
int2strobj(sen_ctx *c, int32_t i)
{
  char buf[32], *rest;
  if (sen_str_itoa(i, buf, buf + 32, &rest)) { return NULL; }
  return sen_ql_mk_string(c, buf, rest - buf);
}

inline static char *
str_value(sen_ctx *c, sen_obj *o)
{
  if (o->flags & SEN_OBJ_SYMBOL) {
    char *r = SEN_SET_STRKEY_BY_VAL(o);
    return *r == ':' ? r + 1 : r;
  } else if (o->type == sen_ql_bulk) {
    return o->u.b.value;
  } else if (o->type == sen_ql_int) {
    sen_obj *p = int2strobj(c, o->u.i.i);
    return p ? p->u.b.value : NULL;
  }
  return NULL;
}

inline static sen_obj *
obj2oid(sen_ctx *c, sen_obj *obj)
{
  char buf[32];
  sen_rbuf bogus_buf = { /*.head = */buf, /*.curr = */buf, /*.tail = */buf + 32 };
  if (obj->type != sen_ql_object) { return NIL; }
  sen_obj_inspect(c, obj, &bogus_buf, SEN_OBJ_INSPECT_ESC);
  return sen_ql_mk_string(c, buf, SEN_RBUF_VSIZE(&bogus_buf));
}

static sen_obj *
_native_method_object(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_db_store *slot;
  sen_obj *obj, *car, *res = c->code;
  POP(car, args);
  if (!(obj = c->code) || !(msg = str_value(c, car))) { return res; }
  if (*msg == ':') {
    switch (msg[1]) {
    case 'k' : /* :key */
    case 'K' :
      {
        const char *key;
        sen_db_store *cls = sen_db_store_by_id(c->db, obj->class);
        if (!cls) { return F; }
        switch (cls->type) {
        case sen_db_class :
          if (!(key = _sen_sym_key(cls->u.c.keys, obj->u.o.self))) { return F; }
          res = sen_ql_mk_string(c, key, strlen(key));
          break;
        case sen_db_rel1 :
          res = sen_obj_new(c);
          res->type = sen_ql_int;
          res->u.i.i = obj->u.o.self;
          break;
        default :
          res = F;
          break;
        }
      }
      break;
    case 'i' : /* :id */
    case 'I' :
      res = obj2oid(c, obj);
      break;
    }
  } else {
    if (!(slot = sen_db_class_slot(c->db, obj->class, msg))) { return F; }
    res = slot_value(c, slot, obj->u.o.self, args, NULL);
  }
  return res;
}

sen_obj *
sen_ql_class_at(sen_ctx *c, sen_db_store *cls, const void *key, int flags, sen_obj *res)
{
  sen_id id = flags ? sen_sym_get(cls->u.c.keys, key) : sen_sym_at(cls->u.c.keys, key);
  if (id) {
    if (!res) { res = sen_obj_new(c); }
    obj_obj_bind(res, cls->id, id);
    return res;
  } else {
    return NIL;
  }
}

static sen_obj *
_native_method_void(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  if (!c->code) { return F; }
  return c->code;
}

static int
compar_ra(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  void *va, *vb;
  sen_id *pa, *pb;
  sen_ra *raa = (sen_ra *)ra->userdata, *rab = (sen_ra *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = sen_ra_at(raa, *pa);
  vb = sen_ra_at(rab, *pb);
  if (va) {
    if (vb) {
      if (raa->header->element_size == sizeof(int)) {
        // todo : support uint
        return *((int *)va) - *((int *)vb);
      } else {
        return memcmp(va, vb, raa->header->element_size);
      }
    } else {
      return 1;
    }
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_ja(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  int r;
  const void *va, *vb;
  uint32_t la, lb;
  sen_id *pa, *pb;
  sen_ja *jaa = (sen_ja *)ra->userdata, *jab = (sen_ja *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = sen_ja_ref(jaa, *pa, &la);
  vb = sen_ja_ref(jab, *pb, &lb);
  if (va) {
    if (vb) {
      if (la > lb) {
        if ((r = memcmp(va, vb, lb))) {
          return r;
        } else {
          return 1;
        }
      } else {
        if ((r = memcmp(va, vb, la))) {
          return r;
        } else {
          return la == lb ? 0 : -1;
        }
      }
    } else {
      return 1;
    }
  } else {
    return vb ? -1 : 0;
  }
}

static int
compar_key(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  const char *va, *vb;
  sen_id *pa, *pb;
  sen_sym *ka = ra->userdata, *kb = rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = _sen_sym_key(ka, *pa);
  vb = _sen_sym_key(kb, *pb);
  // todo : if (key_size)..
  if (va) {
    return vb ? strcmp(va, vb) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

// from index.c
typedef struct {
  int score;
  sen_id subid;
} subrec;

typedef struct {
  int score;
  int n_subrecs;
  subrec subrecs[1];
} recinfo;

/* todo : must be unified with scm.c:cons() */
inline static sen_obj *
cons(sen_ctx *c, sen_obj *a, sen_obj *b)
{
  sen_obj *o = sen_obj_new(c);
  o->type = sen_ql_list;
  o->flags = SEN_OBJ_REFERER;
  o->u.l.car = a;
  o->u.l.cdr = b;
  return o;
}

static sen_obj sen_db_pslot_key = {
  /*.type = */sen_db_pslot,
  /*.flags = */SEN_OBJ_NATIVE,
  /*.nrefs = */0,
  /*.class = */0,
  {
    {
      /*.u.o.self = */SEN_DB_PSLOT_FLAG,
      /*.u.o.func = */NULL
    }
  }
};

static sen_obj sen_db_pslot_id = {
  /*.type = */sen_db_pslot,
  /*.flags = */SEN_OBJ_NATIVE,
  /*.nrefs = */0,
  /*.class = */0,
  {
    {
      /*.u.o.self = */SEN_DB_PSLOT_FLAG|SEN_DB_PSLOT_ID,
      /*.u.o.func = */NULL
    }
  }
};
static sen_obj sen_db_pslot_score = {
  /*.type = */sen_db_pslot,
  /*.flags = */SEN_OBJ_NATIVE,
  /*.nrefs = */0,
  /*.class = */0,
  {
    {
      /*.u.o.self = */SEN_DB_PSLOT_FLAG|SEN_DB_PSLOT_SCORE,
      /*.u.o.func = */NULL
    }
  }
};
static sen_obj sen_db_pslot_nsubrecs = {
  /*.type = */sen_db_pslot,
  /*.flags = */SEN_OBJ_NATIVE,
  /*.nrefs = */0,
  /*.class = */0,
  {
    {
      /*.u.o.self = */SEN_DB_PSLOT_FLAG|SEN_DB_PSLOT_NSUBRECS,
      /*.u.o.func = */NULL
    }
  }
};

inline static sen_obj *
class_slot(sen_ctx *c, sen_id base, char *msg, sen_records *records, int *recpslotp)
{
  *recpslotp = 0;
  if (*msg == ':') {
    switch (msg[1]) {
    case 'i' :
    case 'I' :
      return &sen_db_pslot_id;
    case 'K' : /* :key */
    case 'k' :
      return &sen_db_pslot_key;
    case 'S' : /* :score */
    case 's' :
      if (records) {
        *recpslotp = 1;
        return &sen_db_pslot_score;
      }
      return F;
    case 'N' : /* :nsubrecs */
    case 'n' :
      if (records) {
        *recpslotp = 1;
        return &sen_db_pslot_nsubrecs;
      }
      return F;
    default :
      return F;
    }
  } else {
    sen_db_store *slot;
    char buf[SEN_SYM_MAX_KEY_SIZE];
    if (sen_db_class_slotpath(c->db, base, msg, buf)) { return F; }
    if (!(slot = sen_db_store_open(c->db, buf))) { return F; }
    return sen_ql_mk_symbol(c, buf);
  }
}

static sen_obj *
slotexp_prepare(sen_ctx *c, sen_id base, sen_obj *expr, sen_records *records)
{
  char *str;
  int recpslotp;
  sen_obj *e, *slot, *r = NULL;
  if (!LISTP(expr)) { goto exit; }
  e = CAR(expr);
  if (LISTP(e)) {
    for (r = NIL; LISTP(CAR(e)); e = CAR(e)) {
      if (LISTP(CDR(e))) { r = cons(c, CDR(e), r); }
    }
    if (CAR(e) == NIL) { e = CDR(e); }
    if (!(str = str_value(c, CAR(e)))) { goto exit; }
    if (*str == '\0') {
      if (!records) { goto exit; }
      base = records->subrec_id;
      slot = sen_ql_mk_symbol(c, _sen_sym_key(c->db->keys, base));
      if (!CLASSP(slot)) { goto exit; }
      r = cons(c, cons(c, slot, CDR(e)), r);
    } else {
      if ((slot = class_slot(c, base, str, records, &recpslotp)) == F) { goto exit; }
      if (recpslotp) { r = slot; goto exit; }
      r = cons(c, cons(c, slot, CDR(e)), r);
      base = slot->class;
    }
    for (e = CDR(r); LISTP(e); e = CDR(e)) {
      if (!(str = str_value(c, CAAR(e))) ||
          (slot = class_slot(c, base, str, records, &recpslotp)) == F) { goto exit; }
      if (recpslotp) { r = slot; goto exit; }
      e->u.l.car = cons(c, slot, CDAR(e));
      base = slot->class;
    }
  } else {
    if (!(str = str_value(c, e))) { goto exit; }
    r = class_slot(c, base, str, records, &recpslotp);
  }
exit :
  return r;
}

inline static sen_obj *
pslot_value(sen_ctx *c, sen_id slot, sen_obj *value, sen_obj *args, recinfo *ri)
{
  uint8_t pslot_type = slot & SEN_DB_PSLOT_MASK;
  switch (pslot_type) {
  case SEN_DB_PSLOT_ID :
    value = obj2oid(c, value);
    break;
  case SEN_DB_PSLOT_SCORE :
    value->type = sen_ql_int;
    value->u.i.i = ri->score;
    break;
  case SEN_DB_PSLOT_NSUBRECS :
    value->type = sen_ql_int;
    value->u.i.i = ri->n_subrecs;
    break;
  }
  return value;
}

#define SET_SLOT_VALUE(c,slot,value,args,ri) do {\
  if (slot->u.o.self & SEN_DB_PSLOT_FLAG) {\
    value = pslot_value(c, slot->u.o.self, value, args, ri);\
  } else {\
    sen_db_store *dbs = sen_db_store_by_id(c->db, slot->u.o.self);\
    value = slot_value(c, dbs, value->u.o.self, args, value);\
  }\
} while(0)

static sen_obj *
slotexp_exec(sen_ctx *c, sen_obj *expr, sen_obj *value, recinfo *ri)
{
  sen_obj *t, *car;
  if (LISTP(expr)) {
    POP(t, expr);
    car = CAR(t);
    if (CLASSP(car)) {
      int i = 0;
      if (INTP(CADR(t))) { i = CADR(t)->u.i.i; }
      obj_obj_bind(value, car->u.o.self, ri->subrecs[i].subid);
    } else {
      SET_SLOT_VALUE(c, car, value, CDR(t), ri);
    }
  } else if (SLOTP(expr)) {
    SET_SLOT_VALUE(c, expr, value, NIL, ri);
  }
  while (value != NIL && LISTP(expr)) {
    POP(t, expr);
    if (!LISTP(t)) { break; }
    car = CAR(t);
    SET_SLOT_VALUE(c, car, value, CDR(t), ri);
  }
  return value;
}

static int
compar_expr(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  sen_obj oa, ob, *va, *vb;
  sen_id *pa, *pb;
  recinfo *ria, *rib;
  sen_ctx *c = (sen_ctx *) arg;
  sen_obj *exa = (sen_obj *)ra->userdata, *exb = (sen_obj *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, (void **)&ria);
  oa.u.o.self = *pa;
  sen_set_element_info(rb->records, b, (void **)&pb, (void **)&rib);
  ob.u.o.self = *pb;

  va = slotexp_exec(c, exa, &oa, ria);
  vb = slotexp_exec(c, exb, &ob, rib);

  if (va == NIL) { return (vb == NIL) ? 0 : -1; }
  if (vb == NIL) { return 1; }

  if (va->type != vb->type) {
    SEN_LOG(sen_log_error, "obj type unmatch in compar_expr");
    return 0;
  }

  switch (va->type) {
  case sen_ql_object :
    {
      sen_db_store *ca, *cb;
      if (!(ca = sen_db_store_by_id(c->db, va->class)) ||
           (cb = sen_db_store_by_id(c->db, vb->class))) {
         SEN_LOG(sen_log_error, "clas open failed in compar_expr");
         return 0;
      }
      return strcmp(_sen_sym_key(ca->u.c.keys, va->u.o.self),
                    _sen_sym_key(cb->u.c.keys, vb->u.o.self));
    }
    break;
  case sen_ql_bulk :
    {
      int r;
      uint32_t la = va->u.b.size, lb = vb->u.b.size;
      if (la > lb) {
        if ((r = memcmp(va->u.b.value, vb->u.b.value, lb))) {
          return r;
        } else {
          return 1;
        }
      } else {
        if ((r = memcmp(va->u.b.value, vb->u.b.value, la))) {
          return r;
        } else {
          return la == lb ? 0 : -1;
        }
      }
    }
    break;
  case sen_ql_int :
    return va->u.i.i - vb->u.i.i;
    break;
  default :
    SEN_LOG(sen_log_error, "invalid value in compar_expr");
    break;
  }
  return 0;
}

static int
compar_obj(sen_records *ra, const sen_recordh *a, sen_records *rb, const sen_recordh *b, void *arg)
{
  const char *va, *vb;
  sen_id *pa, *pb, *oa, *ob;
  sen_sym *key = (sen_sym *)arg;
  // todo : target class may not be identical
  sen_ra *raa = (sen_ra *)ra->userdata, *rab = (sen_ra *)rb->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  sen_set_element_info(rb->records, b, (void **)&pb, NULL);
  va = (oa = sen_ra_at(raa, *pa)) ? _sen_sym_key(key, *oa) : NULL;
  vb = (ob = sen_ra_at(rab, *pb)) ? _sen_sym_key(key, *ob) : NULL;
  // todo : if (key_size)..
  if (va) {
    return vb ? strcmp(va, vb) : 1;
  } else {
    return vb ? -1 : 0;
  }
}

static int
group_obj(sen_records *ra, const sen_recordh *a, void *gkey, void *arg)
{
  sen_id *pa, *oa;
  sen_ra *raa = (sen_ra *)ra->userdata;
  sen_set_element_info(ra->records, a, (void **)&pa, NULL);
  if (!(oa = sen_ra_at(raa, *pa))) { return 1; }
  memcpy(gkey, oa, sizeof(sen_id));
  return 0;
}

static sen_obj *
_native_method_records(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_obj *car, *res = c->code;
  if (!c->code) { return F; }
  POP(car, args);
  if (!(msg = str_value(c, car))) { return res; }
  switch (*msg) {
  case '\0' : /* get instance by key */
    {
      char *name;
      sen_db_store *cls;
      POP(car, args);
      if (!(name = str_value(c, car))) { return F; }
      if (c->code->class) {
        cls = sen_db_store_by_id(c->db, c->code->class);
        res = sen_ql_class_at(c, cls, name, 0, NULL);
        if (!NILP(res) &&
            !sen_set_at(c->code->u.r.records->records, &res->u.o.self, NULL)) {
          res = NIL;
        }
      } else {
        res = sen_ql_at(c, name);
        if (!(res->flags & SEN_OBJ_NATIVE) ||
            !sen_set_at(c->code->u.r.records->records, &res->u.o.self, NULL)) {
          res = NIL;
        }
      }
    }
    break;
  case ':' :
    switch (msg[1]) {
    case 'd' : /* :difference */
    case 'D' :
      {
        sen_records *r = c->code->u.r.records;
        res = c->code;
        if (LISTP(args)) {
          POP(car, args);
          if (RECORDSP(car)) {
            sen_records_difference(r, car->u.r.records);
          }
        }
      }
      break;
    case 'g' : /* :group */
    case 'G' :
      {
        char *str;
        int limit = 0;
        sen_db_store *slot;
        sen_group_optarg arg;
        sen_obj *rec = c->code;
        POP(car, args);
        if (!(str = str_value(c, car))) { break; }
        if (!(slot = sen_db_class_slot(c->db, rec->class, str))) { break; }
        if (slot->type != sen_db_obj_slot) { break; } // todo : support others
        rec->u.r.records->userdata = slot->u.o.ra;
        arg.mode = sen_sort_descending;
        arg.func = group_obj;
        arg.func_arg = NULL;
        arg.key_size = sizeof(sen_id);
        POP(car, args);
        if (!sen_obj2int(car)) { limit = car->u.i.i; }
        POP(car, args);
        if ((str = str_value(c, car)) && (*str == 'a')) {
          arg.mode = sen_sort_ascending;
        }
        if (!sen_records_group(rec->u.r.records, limit, &arg)) {
          sen_db_store *cls = sen_db_store_by_id(c->db, slot->u.o.class);
          rec->u.r.records->subrec_id = rec->class;
          rec->class = slot->u.o.class;
          rec->u.r.records->keys = cls->u.c.keys;
          res = rec;
        }
      }
      break;
    case 'i' : /* :intersect */
    case 'I' :
      {
        sen_records *r = c->code->u.r.records;
        res = c->code;
        while (LISTP(args)) {
          POP(car, args);
          if (!RECORDSP(car)) { continue; }
          sen_records_intersect(r, car->u.r.records);
          car->type = sen_ql_void;
          car->u.o.func = _native_method_void;
          car->flags &= ~SEN_OBJ_ALLOCATED;
        }
      }
      break;
    case 'n' : /* :nrecs */
    case 'N' :
      res = sen_obj_new(c);
      res->type = sen_ql_int;
      res->u.i.i = sen_records_nhits(c->code->u.r.records);
      break;
    case 's' :
    case 'S' :
      {
        switch (msg[2]) {
        case 'o' : /* :sort */
        case 'O' :
          {
            int limit = 10;
            const char *str;
            sen_sort_optarg arg;
            sen_obj *rec = c->code;
            arg.compar = NULL;
            arg.compar_arg = (void *)(intptr_t)rec->u.r.records->record_size;
            arg.mode = sen_sort_descending;
            if ((str = str_value(c, CAR(args)))) {
              if (*str == ':') {
                switch (str[1]) {
                case 's' : /* :score */
                  break;
                case 'k' : /* :key */
                  if (rec->class) {
                    sen_db_store *cls = sen_db_store_by_id(c->db, rec->class);
                    if (cls) {
                      rec->u.r.records->userdata = cls->u.c.keys;
                      arg.compar = compar_key;
                    }
                  } else {
                    rec->u.r.records->userdata = c->db->keys;
                    arg.compar = compar_key;
                  }
                  break;
                case 'n' :
                  arg.compar_arg =
                    (void *)(intptr_t)(rec->u.r.records->record_size + sizeof(int));
                  break;
                }
              } else {
                sen_db_store *slot = sen_db_class_slot(c->db, rec->class, str);
                if (slot) {
                  switch (slot->type) {
                  case sen_db_ra_slot :
                    rec->u.r.records->userdata = slot->u.f.ra;
                    arg.compar = compar_ra;
                    break;
                  case sen_db_ja_slot :
                    rec->u.r.records->userdata = slot->u.v.ja;
                    arg.compar = compar_ja;
                    break;
                  case sen_db_obj_slot :
                    {
                      sen_db_store *cls = sen_db_store_by_id(c->db, slot->u.o.class);
                      if (cls) {
                        rec->u.r.records->userdata = slot->u.o.ra;
                        arg.compar = compar_obj;
                        arg.compar_arg = cls->u.c.keys;
                      }
                    }
                    break;
                  default :
                    break;
                  }
                }
              }
            } else {
              sen_obj *se = slotexp_prepare(c, rec->class, args, rec->u.r.records);
              if (!se) {
                res = F;
                break;
              }
              rec->u.r.records->userdata = se;
              arg.compar = compar_expr;
              arg.compar_arg = c;
            }
            POP(car, args);
            POP(car, args);
            if (!sen_obj2int(car)) { limit = car->u.i.i; }
            POP(car, args);
            if ((str = str_value(c, car)) && *str == 'a') {
              arg.mode = sen_sort_ascending;
            }
            if (!sen_records_sort(rec->u.r.records, limit, &arg)) { res = rec; }
          }
          break;
        case 'u' : /* :subtract */
        case 'U' :
          {
            sen_records *r = c->code->u.r.records;
            res = c->code;
            while (LISTP(args)) {
              POP(car, args);
              if (!RECORDSP(car)) { continue; }
              sen_records_subtract(r, car->u.r.records);
              car->type = sen_ql_void;
              car->u.o.func = _native_method_void;
              car->flags &= ~SEN_OBJ_ALLOCATED;
            }
          }
          break;
        default :
          {
            /* ambiguous message. todo : return error */
            res = F;
          }
        }
      }
      break;
    case 'u' : /* :union */
    case 'U' :
      {
        sen_records *r = c->code->u.r.records;
        res = c->code;
        while (LISTP(args)) {
          POP(car, args);
          if (!RECORDSP(car)) { continue; }
          sen_records_union(r, car->u.r.records);
          car->type = sen_ql_void;
          car->u.o.func = _native_method_void;
          car->flags &= ~SEN_OBJ_ALLOCATED;
        }
      }
      break;
    case '+' : /* :+ (iterator next) */
      {
        sen_id *rid;
        sen_records *r = c->code->u.r.records;
        if (c->code->class) {
          POP(res, args);
          if (res->type == sen_ql_object && res->class == c->code->class) {
            if (sen_records_next(r, NULL, 0, NULL)) {
              sen_set_element_info(r->records, r->curr_rec, (void **)&rid, NULL);
              res->u.o.self = *rid;
            } else {
              res->type = sen_ql_void;
              res->u.o.func = _native_method_void;
            }
          }
        } else {
          if (sen_records_next(r, NULL, 0, NULL)) {
            sen_set_element_info(r->records, r->curr_rec, (void **)&rid, NULL);
            res = sen_ql_mk_symbol(c, _sen_sym_key(c->db->keys, *rid));
          } else {
            res = NIL;
          }
        }
      }
      break;
    case '\0' : /* : (iterator begin) */
      {
        sen_id *rid;
        sen_records *r = c->code->u.r.records;
        sen_records_rewind(r);
        if (sen_records_next(r, NULL, 0, NULL)) {
          sen_set_element_info(r->records, r->curr_rec, (void **)&rid, NULL);
          if (c->code->class) {
            res = sen_obj_new(c);
            obj_obj_bind(res, c->code->class, *rid);
          } else {
            res = sen_ql_mk_symbol(c, _sen_sym_key(c->db->keys, *rid));
          }
        } else {
          res = NIL;
        }
      }
      break;
    }
    break;
  default : /* invalid message */
    res = F;
    break;
  }
  return res;
}

inline static sen_obj *
rec_obj_new(sen_ctx *c, sen_db_store *cls, sen_rec_unit record_unit,
            sen_rec_unit subrec_unit, unsigned int max_n_subrecs)
{
  sen_records *r;
  sen_obj *res;
  if (!(r = sen_records_open(record_unit, subrec_unit, max_n_subrecs))) { return NULL; }
  r->keys = cls->u.c.keys;
  res = sen_obj_new(c);
  rec_obj_bind(res, r, cls->id);
  return res;
}

inline static sen_obj *
query_obj_new(sen_ctx *c, const char *str, unsigned int str_len,
              sen_sel_operator default_op, int max_exprs, sen_encoding encoding)
{
  sen_query *q;
  sen_obj *res;
  if (!(q = sen_query_open(str, str_len, default_op, max_exprs, encoding))) {
    return NULL;
  }
  res = sen_obj_new(c);
  res->type = sen_ql_query;
  res->flags = SEN_OBJ_ALLOCATED;
  res->u.q.query = q;
  return res;
}

typedef struct {
  sen_db_store *slot;
  sen_obj *expr;
  sen_obj *value;
  sen_obj *res;
  sen_obj buf;
} match_spec;

inline static sen_obj*
match_prepare(sen_ctx *c, match_spec *spec, sen_id base, sen_obj *expr)
{
  sen_obj *car, *res;
  char *op, *name;
  if (!LISTP(expr)) { return NIL; }
  POP(car, expr);
  if (!(op = str_value(c, car))) { return NIL; }
  if (*op != 'e') { return NIL; } // todo : support other operators
  if ((name = str_value(c, CAR(expr)))) {
    if (!(spec->slot = sen_db_class_slot(c->db, base, name))) { return NIL; }
    spec->expr = NULL;
  } else {
    sen_obj *e;
    if (!(spec->expr = slotexp_prepare(c, base, expr, NULL))) { return NIL; }
    for (e = spec->expr; CDR(e) != NIL; e = CDR(e));
    spec->slot = sen_db_store_by_id(c->db, CAAR(e)->u.o.self);
  }
  POP(car, expr);
  POP(car, expr);
  switch (spec->slot->type) {
  case sen_db_obj_slot :
    switch (car->type) {
    case sen_ql_object :
      if (car->class != spec->slot->u.o.class) { return NIL; }
      spec->value = car;
      break;
    case sen_ql_bulk :
      {
        sen_id si;
        sen_db_store *sc;
        if (!(sc = sen_db_store_by_id(spec->slot->db, spec->slot->u.o.class)) ||
            !(si = sen_sym_at(sc->u.c.keys, car->u.b.value))) {
          return NIL;
        }
        spec->value = &spec->buf;
        obj_obj_bind(spec->value, spec->slot->u.o.class, si);
      }
      break;
    default :
      return NIL;
      break;
    }
    break;
  case sen_db_ra_slot :
    if (spec->slot->u.f.ra->header->element_size == sizeof(int32_t)) {
      if (sen_obj2int(car)) { return NIL; }
      spec->value = car;
      break;
    }
    /* fall through */
  case sen_db_ja_slot :
    if (!BULKP(car)) { return NIL; }
    spec->value = car;
    break;
  case sen_db_idx_slot :
    return NIL;
    break;
  default :
    return NIL;
    break;
  }
  POP(res, expr);
  if (RECORDSP(res)) {
    /* todo : support operator */
  } else {
    sen_db_store *cls;
    cls = sen_db_store_by_id(c->db, base);
    if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
      return NIL;
    }
  }
  spec->res = res;
  return res;
}

/* todo : integrate scm.c:eqv() */
inline static int
eqv(sen_obj *a, sen_obj *b)
{
  if (a->type != b->type) { return 0; }
  switch (a->type) {
  case sen_ql_object :
    return (a->class == b->class && a->u.o.self == b->u.o.self);
    break;
  case sen_ql_bulk :
    return (a->u.b.size == b->u.b.size &&
            !memcmp(a->u.b.value, b->u.b.value, a->u.b.size));
    break;
  case sen_ql_int :
    return (a->u.i.i == b->u.i.i);
    break;
  default :
    /* todo : support other types */
    return 0;
    break;
  }
}

inline static int
match_exec(sen_ctx *c, match_spec *spec, sen_id id)
{
  sen_obj buf, *value;
  if (spec->expr) {
    buf.u.o.self = id;
    value = slotexp_exec(c, spec->expr, &buf, NULL);
  } else {
    value = slot_value(c, spec->slot, id, NIL, &buf);
  }
  if (NILP(value) || !eqv(spec->value, value)) {
    return 0;
  }
  sen_set_get(spec->res->u.r.records->records, &id, NULL);
  return 1;
}


struct _ins_stat {
  sen_obj *slots;
  int nslots;
  int nrecs;
};

// todo : refine
#define MAXSLOTS 0x100

static sen_obj *
_native_method_class(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_id base;
  int load = 0;
  sen_obj *car, *res = c->code;
  if (!c->code) { return F; }
  base = c->code->u.o.self;
  SEN_QL_CO_BEGIN(co);
  POP(car, args);
  if (!(msg = str_value(c, car))) { return res; }
  switch (*msg) {
  case '\0' : /* get instance by key */
    {
      char *name;
      sen_db_store *cls;
      POP(car, args);
      if (!(name = str_value(c, car))) { return F; }
      cls = sen_db_store_by_id(c->db, base);
      res = sen_ql_class_at(c, cls, name, 0, NULL);
    }
    break;
  case ':' :
    switch (msg[1]) {
    case 'c' : /* :common-prefix-search */
    case 'C' :
      {
        char *name;
        sen_id id;
        sen_db_store *cls = sen_db_store_by_id(c->db, base);
        POP(car, args);
        if (!(name = str_value(c, car))) { return F; }
        if ((id = sen_sym_common_prefix_search(cls->u.c.keys, name))) {
          if ((res = sen_obj_new(c))) {
            obj_obj_bind(res, base, id);
          }
        } else {
          res = NIL;
        }
      }
      break;
    case 'd' :
    case 'D' :
      switch (msg[2]) {
      case 'e' :
      case 'E' :
        switch (msg[3]) {
        case 'f' : /* :def */
        case 'F' :
          {
            char *name;
            sen_id target = 0;
            sen_db_store *slot;
            sen_db_store_spec spec;
            POP(car, args);
            if (!(name = str_value(c, car))) { return F; }
            if (sen_db_class_slot(c->db, base, name)) { return T; /* already exists */ }
            POP(car, args);
            spec.u.s.class = car->u.o.self;
            spec.u.s.size = 0;
            spec.u.s.collection_type = 0;
            switch (car->type) {
            case sen_db_raw_class :
              {
                sen_db_store *cls = sen_db_store_by_id(c->db, spec.u.s.class);
                if (!cls) { return F; }
                spec.type = (cls->u.bc.element_size > 8) ? sen_db_ja_slot : sen_db_ra_slot;
                spec.u.s.size = cls->u.bc.element_size;
              }
              break;
            case sen_db_class :
              spec.type = sen_db_obj_slot;
              break;
            case sen_db_obj_slot :
            case sen_db_ra_slot :
            case sen_db_ja_slot :
              spec.type = sen_db_idx_slot;
              break;
            case sen_ql_void :
              /* keyword might be assigned */
              break;
            default :
              return F;
            }
            while (LISTP(args)) {
              POP(car, args);
              if (LISTP(car)) { /* view definition */
                char *opt = str_value(c, CADR(car));
                if (opt && !strcmp(opt, ":match")) { /* fulltext index */
                  spec.type = sen_db_idx_slot;
                  car = CAR(car);
                  if (LISTP(car)) {
                    char *slotname;
                    sen_db_store *ts;
                    if (CAR(car)->type != sen_db_class) { return F; }
                    spec.u.s.class = CAR(car)->u.o.self;
                    if (!(slotname = str_value(c, CADR(car))) ||
                        !(ts = sen_db_class_slot(c->db, spec.u.s.class, slotname))) {
                      return F;
                    }
                    target = ts->id;
                  } else {
                    sen_db_store *cls = sen_db_slot_class_by_id(c->db, car->u.o.self);
                    if (!cls) { return F; }
                    spec.u.s.class = cls->id;
                    target = car->u.o.self;
                  }
                }
              }
            }
            {
              char buf[SEN_SYM_MAX_KEY_SIZE];
              if (sen_db_class_slotpath(c->db, base, name, buf)) { return F; }
              if (!(slot = sen_db_store_create(c->db, buf, &spec))) { return F; }
              if (spec.type == sen_db_idx_slot && target) {
                sen_db_store_rel_spec rs;
                rs.type = sen_db_index_target;
                rs.target = target;
                sen_db_store_add_trigger(slot, &rs);
              }
              res = sen_ql_mk_symbol(c, buf);
              sen_ql_bind_symbol(slot, res);
            }
          }
          break;
        case 'l' : /* :delete */
        case 'L' :
          // todo : delete
          break;
        default :
          res = F;
        }
        break;
      default :
        res = F;
      }
      break;
    case 'l' : /* :load */
    case 'L' :
      load = 1;
      break;
    case 'n' :
    case 'N' :
      {
        sen_db_store *cls;
        switch (msg[2]) {
        case 'e' : /* :new */
        case 'E' :
          {
            char *name;
            POP(car, args);
            if (!(name = str_value(c, car))) { return F; }
            cls = sen_db_store_by_id(c->db, base);
            res = sen_ql_class_at(c, cls, name, 1, NULL);
            if (!NILP(res)) {
              sen_obj cons, dummy;
              sen_db_store *slot;
              cons.type = sen_ql_list;
              cons.flags = SEN_OBJ_REFERER;
              cons.u.l.cdr = NIL;
              while (LISTP(args)) {
                POP(car, args);
                if (!(msg = str_value(c, car))) { break; }
                POP(car, args);
                if (NILP(car)) { break; }
                if (!(slot = sen_db_class_slot(c->db, base, msg))) { break; }
                cons.u.l.car = car;
                slot_value(c, slot, res->u.o.self, &cons, &dummy);
              }
            }
          }
          break;
        case 'r' : /* :nrecs */
        case 'R' :
          {
            cls = sen_db_store_by_id(c->db, base);
            res = sen_obj_new(c);
            res->type = sen_ql_int;
            res->u.i.i = sen_sym_size(cls->u.c.keys);
          }
          break;
        default :
          {
            /* ambiguous message. todo : return error */
            res = F;
          }
        }
      }
      break;
    case 'p' : /* :prefix-search */
    case 'P' :
      {
        char *name;
        sen_db_store *cls = sen_db_store_by_id(c->db, base);
        POP(car, args);
        if (!(name = str_value(c, car))) { return F; }
        if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
          return NIL;
        }
        sen_sym_prefix_search_with_set(cls->u.c.keys, name, res->u.r.records->records);
      }
      break;
    case 's' :
    case 'S' :
      switch (msg[2]) {
      case 'c' : /* :scan-select */
      case 'C' :
        {
          match_spec spec;
          sen_db_store *cls = sen_db_store_by_id(c->db, base);
          sen_id id = SEN_SYM_NIL, maxid = sen_sym_curr_id(cls->u.c.keys);
          res = match_prepare(c, &spec, base, CAR(args));
          if (NILP(res)) { break; }
          while (++id <= maxid) { match_exec(c, &spec, id); }
        }
        break;
      case 'u' : /* :suffix-search */
      case 'U' :
        {
          char *name;
          sen_db_store *cls = sen_db_store_by_id(c->db, base);
          POP(car, args);
          if (!(name = str_value(c, car))) { return F; }
          if (!(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
            return NIL;
          }
          sen_sym_suffix_search_with_set(cls->u.c.keys, name, res->u.r.records->records);
        }
        break;
      case 'l' : /* :slots */
      case 'L' :
        {
          char *name;
          char buf[SEN_SYM_MAX_KEY_SIZE];
          POP(car, args);
          if (!(name = str_value(c, car))) { name = ""; }
          if (sen_db_class_slotpath(c->db, base, name, buf)) { return NIL; }
          {
            sen_records *r;
            if (!(r = sen_records_open(sen_rec_document, sen_rec_none, 0))) {
              return NIL;
            }
            r->keys = c->db->keys;
            res = sen_obj_new(c);
            rec_obj_bind(res, r, 0);
          }
          sen_sym_prefix_search_with_set(c->db->keys, buf, res->u.r.records->records);
        }
        break;
      }
      break;
    case 'u' : /* :undef */
    case 'U' :
      // todo : undef slot
      break;
    case '+' : /* :+ (iterator next) */
      {
        sen_db_store *cls;
        cls = sen_db_store_by_id(c->db, base);
        POP(res, args);
        if (res->type == sen_ql_object && res->class == cls->id) {
          res->u.o.self = sen_sym_next(cls->u.c.keys, res->u.o.self);
          if (res->u.o.self == SEN_SYM_NIL) {
            res->type = sen_ql_void;
            res->u.o.func = _native_method_void;
          }
        }
      }
      break;
    case '\0' : /* : (iterator begin) */
      {
        sen_id id;
        sen_db_store *cls;
        cls = sen_db_store_by_id(c->db, base);
        id = sen_sym_next(cls->u.c.keys, SEN_SYM_NIL);
        if (id == SEN_SYM_NIL) {
          res = NIL;
        } else {
          res = sen_obj_new(c);
          obj_obj_bind(res, cls->id, id);
        }
      }
      break;
    }
    break;
  default : /* :slotname */
    {
      int recpslotp;
      res = class_slot(c, base, msg, NULL, &recpslotp);
    }
    break;
  }
  if (load) {
    int i, recpslotp;
    sen_obj *s;
    struct _ins_stat *stat;
    for (s = args, i = 0; LISTP(s); s = CDR(s), i++) {
      car = CAR(s);
      if (!(msg = str_value(c, car))) { return F; }
      if ((s->u.l.car = class_slot(c, base, msg, NULL, &recpslotp)) == F) { return F; }
    }
    if (!(s = sen_obj_alloc(c, sizeof(struct _ins_stat)))) { /* todo */ }
    stat = (struct _ins_stat *)s->u.b.value; // todo : not GC safe
    stat->slots = args;
    stat->nslots = i + 1;
    stat->nrecs = 0;
    do {
      SEN_QL_CO_WAIT(co, stat);
      if (BULKP(args) && args->u.b.size) {
        char *tokbuf[MAXSLOTS];
        sen_db_store *cls, *slot;
        sen_obj val, obj, cons, dummy;
        cons.type = sen_ql_list;
        cons.flags = SEN_OBJ_REFERER;
        cons.u.l.car = &val;
        cons.u.l.cdr = NIL;
        val.type = sen_ql_bulk;
        if (sen_str_tok(args->u.b.value, args->u.b.size, '\t', tokbuf, MAXSLOTS, NULL) == stat->nslots) {
          sen_obj *o;
          cls = sen_db_store_by_id(c->db, base);
          *tokbuf[0] = '\0';
          o = sen_ql_class_at(c, cls, args->u.b.value, 1, &obj);
          if (NILP(o)) { continue; }
          for (s = stat->slots, i = 1; i < stat->nslots; s = CDR(s), i++) {
            val.u.b.value = tokbuf[i - 1] + 1;
            val.u.b.size = tokbuf[i] - val.u.b.value;
            if (!(slot = sen_db_store_by_id(c->db, CAR(s)->u.o.self))) { /* todo */ }
            slot_value(c, slot, obj.u.o.self, &cons, &dummy); // todo : refine cons
          }
          stat->nrecs++;
        }
      } else {
        co->mode |= SEN_QL_TAIL;
      }
    } while (!(co->mode & (SEN_QL_HEAD|SEN_QL_TAIL)));
    res = sen_obj_new(c);
    res->type = sen_ql_int;
    res->u.i.i = stat->nrecs;
  }
  SEN_QL_CO_END(co);
  return res;
}

static sen_obj *
_native_method_rel1(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_id base;
  sen_obj *args0 = args, *car, *res = c->code;
  if (!c->code) { return F; }
  base = c->code->u.o.self;
  POP(car, args);
  if (!(msg = str_value(c, car))) { return res; }
  switch (*msg) {
  case '\0' : /* get instance by key */
    {
      sen_id id;
      uint8_t *v;
      sen_db_store *cls;
      POP(car, args);
      cls = sen_db_store_by_id(c->db, base);
      if (cls->u.f.class) {
        sen_db_store *tcls = sen_db_store_by_id(c->db, cls->u.f.class);
        char *name;
        POP(car, args);
        if (!(name = str_value(c, car))) { return F; }
        res = sen_ql_class_at(c, tcls, name, 0, NULL);
        if (!NILP(res)) {
          id = res->u.o.self;
          if (!(v = (uint8_t *)sen_ra_at(cls->u.f.ra, id)) || !*v) { return F; }
        }
      } else {
        if (car->type == sen_ql_bulk) {
          id = sen_atoi(car->u.b.value, (char *)car->u.b.value + car->u.b.size, NULL);
        } else if (car->type == sen_ql_int) {
          id = car->u.i.i;
        } else { return F; }
        if (!(v = (uint8_t *)sen_ra_at(cls->u.f.ra, id))) { return F; }
        if (!(*v & 1)) { return F; }
        res = sen_ql_mk_obj(c, base, id);
      }
      return res;
    }
    break;
  case ':' :
    switch (msg[1]) {
    case 'n' :
    case 'N' :
      {
        sen_db_store *cls;
        switch (msg[2]) {
        case 'e' : /* :new */
        case 'E' :
          {
            sen_id id;
            uint8_t *v;
            cls = sen_db_store_by_id(c->db, base);
            if (cls->u.f.class) {
              sen_db_store *tcls = sen_db_store_by_id(c->db, cls->u.f.class);
              char *name;
              POP(car, args);
              if (!(name = str_value(c, car))) { return F; }
              res = sen_ql_class_at(c, tcls, name, 0, NULL);
              if (!NILP(res)) {
                id = res->u.o.self;
                if (!(v = (uint8_t *)sen_ra_get(cls->u.f.ra, id))) { return F; }
                if (!*v) {
                  cls->u.f.ra->header->nrecords += 1;
                  *v |= 1;
                }
              }
            } else {
              id = cls->u.f.ra->header->curr_max + 1;
              if (!(v = (uint8_t *)sen_ra_get(cls->u.f.ra, id))) { return F; }
              cls->u.f.ra->header->nrecords += 1;
              *v |= 1;
              res = sen_ql_mk_obj(c, base, id);
            }
            if (!NILP(res)) {
              sen_obj cons, dummy;
              sen_db_store *slot;
              cons.type = sen_ql_list;
              cons.flags = SEN_OBJ_REFERER;
              cons.u.l.cdr = NIL;
              while (LISTP(args)) {
                POP(car, args);
                if (!(msg = str_value(c, car))) { continue; }
                POP(car, args);
                if (NILP(car)) { continue; }
                if (!(slot = sen_db_class_slot(c->db, base, msg))) { break; }
                cons.u.l.car = car;
                slot_value(c, slot, res->u.o.self, &cons, &dummy);
              }
            }
            return res;
          }
          break;
        case 'r' : /* :nrecs */
        case 'R' :
          {
            cls = sen_db_store_by_id(c->db, base);
            res = sen_obj_new(c);
            res->type = sen_ql_int;
            res->u.i.i = cls->u.f.ra->header->nrecords;
            return res;
          }
          break;
        default :
          {
            /* ambiguous message. todo : return error */
            res = F;
          }
        }
      }
      break;
    case 's' :
    case 'S' :
      switch (msg[2]) {
      case 'c' : /* :scan-select */
      case 'C' :
        {
          match_spec spec;
          sen_db_store *cls = sen_db_store_by_id(c->db, base);
          sen_id id = SEN_SYM_NIL, maxid = cls->u.f.ra->header->curr_max;
          res = match_prepare(c, &spec, base, CAR(args));
          if (NILP(res)) { break; }
          while (++id <= maxid) { match_exec(c, &spec, id); }
        }
        return res;
        break;
      case 'u' : /* :suffix-search is not available*/
      case 'U' :
        return res;
        break;
      default :
        break;
      }
      break;
    }
  }
  return _native_method_class(c, args0, co);
}

static sen_obj *
_native_method_slot(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_id base;
  sen_obj *car, *res = c->code;
  if (!c->code) { return F; }
  base = c->code->u.o.self;
  POP(car, args);
  if (!(msg = str_value(c, car))) { return res; }
  switch (*msg) {
  case '\0' :
    {
      if (IDX_SLOTP(c->code)) {
        sen_obj *q;
        sen_db_store *slot;
        sen_sel_operator op = sen_sel_or;
        POP(q, args);
        if (!QUERYP(q)) {
          if (!BULKP(q)) { return F; }
          q = query_obj_new(c, q->u.b.value, q->u.b.size, sen_sel_and, 32, c->encoding);
        }
        /* TODO: specify record unit */
        /* (idxslot query ((slot1 weight1) (slot2 weight2) ...) records operator+ */
        if (!(slot = sen_db_store_by_id(c->db, c->code->u.o.self))) { return F; }
        POP(car, args);
        /* TODO: handle weights */
        POP(res, args);
        if (RECORDSP(res)) {
          char *ops;
          POP(car, args);
          if ((ops = str_value(c, car))) {
            switch (*ops) {
            case '+': op = sen_sel_or; break;
            case '-': op = sen_sel_but; break;
            case '*': op = sen_sel_and; break;
            case '>': op = sen_sel_adjust; break;
            }
          }
        } else {
          sen_db_store *cls;
          if (!(cls = sen_db_store_by_id(c->db, slot->u.i.class)) ||
              !(res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0))) {
            return F;
          }
        }
        sen_query_exec(slot->u.i.index, q->u.q.query, res->u.r.records, op);
      } else {
        char *name;
        sen_db_store *cls, *slot;
        POP(car, args);
        if (!(name = str_value(c, car))) { return F; }
        if (!(slot = sen_db_store_by_id(c->db, base))) { return F; }
        if (!(cls = sen_db_slot_class_by_id(c->db, base))) { return F; }
        res = sen_ql_class_at(c, cls, name, 0, NULL);
        if (!NILP(res)) {
          res = slot_value(c, slot, res->u.o.self, args, res);
        }
      }
    }
    break;
  }
  return res;
}

void
sen_ql_bind_symbol(sen_db_store *dbs, sen_obj *symbol)
{
  symbol->type = dbs->type;
  symbol->flags |= SEN_OBJ_NATIVE;
  symbol->u.o.self = dbs->id;
  switch (symbol->type) {
  case sen_db_class :
    symbol->u.o.func = _native_method_class;
    symbol->class = 0;
    break;
  case sen_db_obj_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.o.class;
    break;
  case sen_db_ra_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.f.class;
    break;
  case sen_db_ja_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.v.class;
    break;
  case sen_db_idx_slot :
    symbol->u.o.func = _native_method_slot;
    symbol->class = dbs->u.i.class;
    break;
  case sen_db_rel1 :
    symbol->u.o.func = _native_method_rel1;
    symbol->class = 0;
    break;
  default :
    symbol->u.o.func = _native_method_void;
    symbol->class = 0;
    break;
  }
}

static sen_obj *
_native_method_sen_query(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  sen_obj *o = NULL, *s = CAR(args);
  /* TODO: placeholder */
  if (BULKP(s)) {
    /* TODO: operator, exprs, encoding */
    o = query_obj_new(c, s->u.b.value, s->u.b.size, sen_sel_and, 32, c->encoding);
  }
  return o;
}

static sen_obj *
_native_method_sen_snip(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  /* FIXME: use uint instead of int */
  /* args: (str@bulk width@int max_results@int cond1@list cond2@list ...) */
  /* cond: (keyword@bulk [opentag@bulk closetag@bulk]) */
  sen_obj *ret = NULL, *str = CAR(args), *cur = CDR(args);
  if (BULKP(str)) {
    sen_snip *s;
    unsigned int width = 10, max_results;
    if (!LISTP(cur) || sen_obj2int(CAR(cur))) {
      return NIL;
    }
    width = CAR(cur)->u.i.i;
    cur = CDR(cur);
    if (!LISTP(cur) || sen_obj2int(CAR(cur))) {
      return NIL;
    }
    max_results = CAR(cur)->u.i.i;
    cur = CDR(cur);
    /* FIXME: mapping */
    if (!(s = sen_snip_open(c->encoding, SEN_SNIP_NORMALIZE, width, max_results,
                            NULL, 0, NULL, 0, (sen_snip_mapping *)-1))) {
      return NIL;
    }
    for (; LISTP(cur); cur = CDR(cur)) {
      if (LISTP(CAR(cur))) {
        sen_obj *cl = CAR(cur), *kw = CAR(cl);
        if (BULKP(kw)) {
          char *ot = NULL, *ct = NULL;
          uint32_t ot_l = 0, ct_l = 0;
          if (BULKP(CADR(cl))) {
            ot = CADR(cl)->u.b.value;
            ot_l = CADR(cl)->u.b.size;
            if (BULKP(CADDR(cl))) {
              ct = CADDR(cl)->u.b.value;
              ct_l = CADDR(cl)->u.b.size;
            }
          }
          if (!(sen_snip_add_cond(s, kw->u.b.value, kw->u.b.size,
                                  ot, ot_l, ct, ct_l))) {
            /* TODO: error handling */
          }
        }
      }
    }
    {
      unsigned int max_len, nresults;
      if (!(sen_snip_exec(s, str->u.b.value, str->u.b.size, &nresults, &max_len))) {
        if ((ret = sen_obj_alloc(c, sizeof(char) * max_len * nresults))) {
          unsigned int i, tlen = 0;
          for (i = 0; i < nresults; i++) {
            unsigned int len;
            if (!(sen_snip_get_result(s, i, ret->u.b.value + tlen, &len))) {
              /* TODO: concat with specified string */
              tlen += len;
            }
          }
          ret->u.b.size = tlen;
        }
      }
    }
    sen_snip_close(s);
  }
  return ret;
}

static sen_obj *
_native_method_db(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *msg;
  sen_db_store *cls;
  sen_obj *car, *res = c->code;
  POP(car, args);
  if (!(msg = str_value(c, car))) { return res; }
  if (*msg == ':') {
    switch (msg[1]) {
    case 'd' : /* :drop */
    case 'D' :
      {
        // todo : drop
      }
      break;
    case 'p' : /* :prefix-search */
    case 'P' :
      {
        char *name;
        POP(car, args);
        if (!(name = str_value(c, car))) { return F; }
        {
          sen_records *r;
          if (!(r = sen_records_open(sen_rec_document, sen_rec_none, 0))) {
            return NIL;
          }
          r->keys = c->db->keys;
          res = sen_obj_new(c);
          rec_obj_bind(res, r, 0);
        }
        sen_sym_prefix_search_with_set(c->db->keys, name, res->u.r.records->records);
        {
          sen_id *rid;
          sen_set_eh *eh;
          sen_set_cursor *sc = sen_set_cursor_open(res->u.r.records->records);
          while ((eh = sen_set_cursor_next(sc, (void **) &rid, NULL))) {
            if (strchr(_sen_sym_key(c->db->keys, *rid), '.')) {
              sen_set_del(res->u.r.records->records, eh);
            }
          }
          sen_set_cursor_close(sc);
        }
      }
      break;
    case 't' : /* :typedef */
    case 'T' :
      {
        char *name;
        sen_obj *cdr;
        sen_db_store_spec spec;
        spec.type = sen_db_class;
        spec.u.c.size = 0;
        spec.u.c.flags = SEN_INDEX_NORMALIZE;
        spec.u.c.encoding = c->encoding;
        spec.type = sen_db_raw_class;
        POP(car, args);
        if (!(name = str_value(c, car))) { return F; }
        if (sen_db_store_open(c->db, name)) { return T; /* already exists */ }
        for (cdr = args; LISTP(cdr); cdr = CDR(cdr)) {
          if (!sen_obj2int(CAR(cdr))) { spec.u.c.size = CAR(cdr)->u.i.i; }
        }
        if (!spec.u.c.size) { return F; } /* size must be assigned */
        if (!(cls = sen_db_store_create(c->db, name, &spec))) { return F; }
        res = sen_ql_mk_symbol(c, name);
        sen_ql_bind_symbol(cls, res);
      }
      break;
    }
  }
  return res;
}

static sen_obj *
_native_method_table(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *opt;
  sen_db_store *cls;
  sen_obj *car, *res = F;
  sen_db_store_spec spec;
  spec.type = sen_db_class;
  spec.u.c.size = 0;
  spec.u.c.flags = SEN_INDEX_NORMALIZE;
  spec.u.c.encoding = c->encoding;
  while (LISTP(args)) {
    POP(car, args);
    switch (car->type) {
    case sen_db_raw_class :
      cls = sen_db_store_by_id(c->db, car->u.o.self);
      if ((spec.u.c.size = cls->u.bc.element_size) == SEN_SYM_MAX_KEY_SIZE) {
        spec.u.c.size = 0;
      }
      if (spec.u.c.size > SEN_SYM_MAX_KEY_SIZE) { return F; }
      break;
    case sen_db_class :
      if (!(cls = sen_db_store_by_id(c->db, car->u.o.self))) { return F; }
      /* todo : support subrecs */
      res = rec_obj_new(c, cls, sen_rec_document, sen_rec_none, 0);
      break;
    default :
      if ((opt = str_value(c, car))) {
        switch (*opt) {
        case 'd' : /* delimited */
        case 'D' :
          spec.u.c.flags |= SEN_INDEX_DELIMITED;
          break;
        case 'e' : /* euc-jp */
        case 'E' :
          spec.u.c.encoding = sen_enc_euc_jp;
          break;
        case 'k' : /* koi8r */
        case 'K' :
          spec.u.c.encoding = sen_enc_koi8r;
          break;
        case 'l' : /* latin1 */
        case 'L' :
          spec.u.c.encoding = sen_enc_latin1;
          break;
        case 'n' : /* ngram */
        case 'N' :
          spec.u.c.flags |= SEN_INDEX_NGRAM;
          break;
        case 's' :
        case 'S' :
          switch (opt[1]) {
          case 'j' : /* shift-jis */
          case 'J' :
            spec.u.c.encoding = sen_enc_sjis;
            break;
          case 'i' : /* with-sis */
          case 'I' :
            spec.u.c.flags |= SEN_SYM_WITH_SIS;
            break;
          case 'u' : /* surrogate-key */
          case 'U' :
            spec.type = sen_db_rel1;
            spec.u.s.class = 0;
            spec.u.s.size = 1;
            break;
          }
          break;
        case 'u' : /* utf8 */
        case 'U' :
          spec.u.c.encoding = sen_enc_utf8;
          break;
        case 'v' : /* view */
        case 'V' :
          /* todo */
          break;
        default : /* numeric */
          if (sen_obj2int(car)) {
            /* todo : illegal option */
          } else {
            spec.u.c.size = car->u.i.i;
          }
          break;
        }
      } else {
        /* todo : invalid arg */
      }
    }
  }
  /* todo : support anonymous class */
  return res;
}

static sen_obj *
_native_method_ptable(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  sen_obj *car;
  char *name, *opt;
  sen_db_store_spec spec;
  spec.type = sen_db_class;
  spec.u.c.size = 0;
  spec.u.c.flags = SEN_INDEX_NORMALIZE;
  spec.u.c.encoding = c->encoding;
  POP(car, args);
  if (!(name = str_value(c, car))) { return F; }
  if (sen_db_store_open(c->db, name)) { return T; }
  while (LISTP(args)) {
    POP(car, args);
    switch (car->type) {
    case sen_db_raw_class :
      {
        sen_db_store *cls = sen_db_store_by_id(c->db, car->u.o.self);
        if (!cls) { return F; }
        if ((spec.u.c.size = cls->u.bc.element_size) == SEN_SYM_MAX_KEY_SIZE) {
          spec.u.c.size = 0;
        }
        if (spec.u.c.size > SEN_SYM_MAX_KEY_SIZE) { return F; }
      }
      break;
    case sen_db_class :
      spec.type = sen_db_rel1;
      spec.u.s.class = car->u.o.self;
      spec.u.s.size = 1;
      break;
    default :
      if ((opt = str_value(c, car))) {
        switch (*opt) {
        case 'd' : /* delimited */
        case 'D' :
          spec.u.c.flags |= SEN_INDEX_DELIMITED;
          break;
        case 'e' : /* euc-jp */
        case 'E' :
          spec.u.c.encoding = sen_enc_euc_jp;
          break;
        case 'k' : /* koi8r */
        case 'K' :
          spec.u.c.encoding = sen_enc_koi8r;
          break;
        case 'l' : /* latin1 */
        case 'L' :
          spec.u.c.encoding = sen_enc_latin1;
          break;
        case 'n' : /* ngram */
        case 'N' :
          spec.u.c.flags |= SEN_INDEX_NGRAM;
          break;
        case 's' :
        case 'S' :
          switch (opt[1]) {
          case 'j' : /* shift-jis */
          case 'J' :
            spec.u.c.encoding = sen_enc_sjis;
            break;
          case 'i' : /* with-sis */
          case 'I' :
            spec.u.c.flags |= SEN_SYM_WITH_SIS;
            break;
          case 'u' : /* surrogate-key */
          case 'U' :
            spec.type = sen_db_rel1;
            spec.u.s.class = 0;
            spec.u.s.size = 1;
            break;
          }
          break;
        case 'u' : /* utf8 */
        case 'U' :
          spec.u.c.encoding = sen_enc_utf8;
          break;
        case 'v' : /* view */
        case 'V' :
          /* todo */
          break;
        default : /* numeric */
          if (sen_obj2int(car)) {
            /* todo : illegal option */
          } else {
            spec.u.c.size = car->u.i.i;
          }
          break;
        }
      } else {
        /* todo : invalid arg */
      }
    }
  }
  {
    sen_obj *res;
    sen_db_store *cls;
    if (!(cls = sen_db_store_create(c->db, name, &spec))) { return F; }
    res = sen_ql_mk_symbol(c, name);
    sen_ql_bind_symbol(cls, res);
    return res;
  }
}

const char *
_sen_obj_key(sen_ctx *c, sen_obj *obj)
{
  sen_db_store *cls;
  switch (obj->type) {
  case sen_ql_object :
    if (!(cls = sen_db_store_by_id(c->db, obj->class))) { return NULL; }
    switch (cls->type) {
    case sen_db_class :
      return _sen_sym_key(cls->u.c.keys, obj->u.o.self);
    case sen_db_rel1 :
      {
        /* todo : return key value when cls->u.f.class exists */
        sen_obj *p = int2strobj(c, obj->u.o.self);
        return p ? p->u.b.value : NULL;
      }
    default :
      return NULL;
    }
  case sen_db_raw_class :
  case sen_db_class :
  case sen_db_obj_slot :
  case sen_db_ra_slot :
  case sen_db_ja_slot :
  case sen_db_idx_slot :
    return _sen_sym_key(c->db->keys, obj->u.o.self);
  default :
    return NULL;
  }
}

#define flags(p)         ((p)->flags)
#define issymbol(p)     (flags(p) & SEN_OBJ_SYMBOL)
#define ismacro(p)      (flags(p) & SEN_OBJ_MACRO)

static void disp_j(sen_ctx *c, sen_obj *obj, sen_rbuf *buf);

static void
disp_j_with_format(sen_ctx *c, sen_obj *args, sen_rbuf *buf)
{
  sen_obj *car, obj;
  POP(car, args);
  switch (car->type) {
  case sen_ql_records :
    {
      sen_id *rp, base;
      recinfo *ri;
      sen_obj *slots, *s, **d, *se, *t, *v;
      const sen_recordh *rh;
      int i, o, hashp = 0, offset = 0, limit = 10;
      sen_records *r = car->u.r.records;
      base = car->class;
      POP(slots, args);
      if (!LISTP(slots)) {
        disp_j(c, car, buf);
        return;
      }
      if (CAR(slots) == sen_ql_mk_symbol(c, "@")) {
        hashp = 1;
        slots = CDR(slots);
      }
      for (s = slots, d = &slots, o = 0; LISTP(s); s = CDR(s), d = &CDR(*d), o = 1 - o) {
        if (hashp && !o) {
          se = CAR(s);
        } else {
          if (!(se = slotexp_prepare(c, base, s, r))) {
            SEN_LOG(sen_log_error, "slotexp_prepare failed");
            break;
          }
        }
        *d = cons(c, se, NIL);
      }
      POP(car, args);
      if (!sen_obj2int(car)) { offset = car->u.i.i; }
      POP(car, args);
      if (!sen_obj2int(car)) { limit = car->u.i.i; }
      sen_records_rewind(r);
      for (i = 0; i < offset; i++) {
        if (!sen_records_next(r, NULL, 0, NULL)) { break; }
      }
      SEN_RBUF_PUTC(buf, '[');
      for (i = 0; i < limit; i++) {
        if (!sen_records_next(r, NULL, 0, NULL) ||
            !(rh = sen_records_curr_rec(r)) ||
            sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
          break;
        }
        if (i) { SEN_RBUF_PUTS(buf, ", "); }
        SEN_RBUF_PUTC(buf, hashp ? '{' : '[');
        for (s = slots, o = 0;; o = 1 - o) {
          POP(t, s);
          if (hashp && !o) {
            v = t;
          } else {
            obj_obj_bind(&obj, base, *rp);
            v = slotexp_exec(c, t, &obj, ri);
          }
          disp_j(c, v, buf);
          if (!LISTP(s)) { break; }
          SEN_RBUF_PUTS(buf, (hashp && !o) ? ": " : ", ");
        }
        SEN_RBUF_PUTC(buf, hashp ? '}' : ']');
      }
      SEN_RBUF_PUTC(buf, ']');
    }
    break;
  case sen_ql_object :
    {
      sen_id id = car->u.o.self, base = car->class;
      int o, hashp = 0;
      sen_obj *slots, *v;
      POP(slots, args);
      if (!LISTP(slots)) {
        disp_j(c, car, buf);
        return;
      }
      if (CAR(slots) == sen_ql_mk_symbol(c, "@")) {
        hashp = 1;
        slots = CDR(slots);
        if (!LISTP(slots)) {
          disp_j(c, car, buf);
          return;
        }
      }
      SEN_RBUF_PUTC(buf, hashp ? '{' : '[');
      for (o = 0; ; o = 1 - o) {
        if (hashp && !o) {
          v = CAR(slots);
        } else {
          sen_obj *t = slotexp_prepare(c, base, slots, NULL);
          obj_obj_bind(&obj, base, id);
          v = slotexp_exec(c, t, &obj, NULL);
        }
        disp_j(c, v, buf);
        slots = CDR(slots);
        if (!LISTP(slots)) { break; }
        SEN_RBUF_PUTS(buf, (hashp && !o) ? ": " : ", ");
      }
      SEN_RBUF_PUTC(buf, hashp ? '}' : ']');
    }
    break;
  default :
    disp_j(c, car, buf);
    break;
  }
}

static void
disp_j(sen_ctx *c, sen_obj *obj, sen_rbuf *buf)
{
  if (!obj || obj == NIL) {
    SEN_RBUF_PUTS(buf, "null");
  } else if (obj == T) {
    SEN_RBUF_PUTS(buf, "true");
  } else if (obj == F) {
    SEN_RBUF_PUTS(buf, "false");
  } else {
    switch (obj->type) {
    case sen_ql_void :
      if (issymbol(obj)) {
        const char *r = SEN_SET_STRKEY_BY_VAL(obj);
        sen_rbuf_str_esc(buf, (*r == ':') ? r + 1 : r, -1, c->encoding);
      } else {
        SEN_RBUF_PUTS(buf, "null");
      }
      break;
    case sen_ql_records :
      {
        int i;
        sen_id *rp;
        recinfo *ri;
        sen_obj o;
        const sen_recordh *rh;
        sen_records *r = obj->u.r.records;
        sen_records_rewind(r);
        obj_obj_bind(&o, obj->class, 0);
        SEN_RBUF_PUTC(buf, '[');
        for (i = 0;; i++) {
          if (!sen_records_next(r, NULL, 0, NULL) ||
              !(rh = sen_records_curr_rec(r)) ||
              sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
            break;
          }
          if (i) { SEN_RBUF_PUTS(buf, ", "); }
          o.u.o.self = *rp;
          disp_j(c, &o, buf);
        }
        SEN_RBUF_PUTC(buf, ']');
      }
      break;
    case sen_ql_list :
      if (obj->u.l.car == sen_ql_mk_symbol(c, ":")) {
        disp_j_with_format(c, obj->u.l.cdr, buf);
      } else if (obj->u.l.car == sen_ql_mk_symbol(c, "@")) {
        int o;
        SEN_RBUF_PUTC(buf, '{');
        for (obj = obj->u.l.cdr, o = 0;; o = 1 - o) {
          disp_j(c, obj->u.l.car, buf);
          if ((obj = obj->u.l.cdr) && (obj != NIL)) {
            if (LISTP(obj)) {
              SEN_RBUF_PUTS(buf, o ? ", " : ": ");
            } else {
              SEN_RBUF_PUTS(buf, " . ");
              disp_j(c, obj, buf);
              SEN_RBUF_PUTC(buf, '}');
              break;
            }
          } else {
            SEN_RBUF_PUTC(buf, '}');
            break;
          }
        }
      } else {
        SEN_RBUF_PUTC(buf, '[');
        for (;;) {
          disp_j(c, obj->u.l.car, buf);
          if ((obj = obj->u.l.cdr) && (obj != NIL)) {
            if (LISTP(obj)) {
              SEN_RBUF_PUTS(buf, ", ");
            } else {
              SEN_RBUF_PUTS(buf, " . ");
              disp_j(c, obj, buf);
              SEN_RBUF_PUTC(buf, ']');
              break;
            }
          } else {
            SEN_RBUF_PUTC(buf, ']');
            break;
          }
        }
      }
      break;
    case sen_ql_object :
      {
        const char *key = _sen_obj_key(c, obj);
        if (key) {
          sen_rbuf_str_esc(buf, key, -1, c->encoding);
        } else {
          SEN_RBUF_PUTS(buf, "<LOSTKEY>");
        }
      }
      break;
    default :
      sen_obj_inspect(c, obj, buf, SEN_OBJ_INSPECT_ESC|SEN_OBJ_INSPECT_SYM_AS_STR);
      break;
    }
  }
}

static void disp_t(sen_ctx *c, sen_obj *obj, sen_rbuf *buf);

static void
disp_t_with_format(sen_ctx *c, sen_obj *args, sen_rbuf *buf)
{
  sen_obj *car, obj;
  POP(car, args);
  switch (car->type) {
  case sen_ql_records :
    {
      sen_id *rp, base;
      recinfo *ri;
      sen_obj *slots, *s, **d, *se, *t, *v;
      const sen_recordh *rh;
      int i, o, hashp = 0, offset = 0, limit = 10;
      sen_records *r = car->u.r.records;
      base = car->class;
      POP(slots, args);
      if (!LISTP(slots)) {
        disp_t(c, car, buf);
        return;
      }
      if (CAR(slots) == sen_ql_mk_symbol(c, "@")) {
        hashp = 1;
        slots = CDR(slots);
      }
      for (s = slots, d = &slots, o = 0; LISTP(s); s = CDR(s), o = 1 - o) {
        if (hashp && !o) {
          if (s != slots) { SEN_RBUF_PUTC(buf, '\t'); }
          disp_t(c, CAR(s), buf);
        } else {
          if (!(se = slotexp_prepare(c, base, s, r))) {
            SEN_LOG(sen_log_error, "slotexp_prepare failed");
            break;
          }
          *d = cons(c, se, NIL);
          d = &CDR(*d);
        }
      }
      if (hashp) { c->output(c, SEN_QL_MORE, c->data.ptr); }
      POP(car, args);
      if (!sen_obj2int(car)) { offset = car->u.i.i; }
      POP(car, args);
      if (!sen_obj2int(car)) { limit = car->u.i.i; }
      sen_records_rewind(r);
      for (i = 0; i < offset; i++) {
        if (!sen_records_next(r, NULL, 0, NULL)) { break; }
      }
      for (i = 0; i < limit; i++) {
        if (!sen_records_next(r, NULL, 0, NULL) ||
            !(rh = sen_records_curr_rec(r)) ||
            sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
          break;
        }
        for (s = slots;;) {
          POP(t, s);
          obj_obj_bind(&obj, base, *rp);
          v = slotexp_exec(c, t, &obj, ri);
          disp_t(c, v, buf);
          if (!LISTP(s)) { break; }
          SEN_RBUF_PUTC(&c->outbuf, '\t');
        }
        c->output(c, SEN_QL_MORE, c->data.ptr);
      }
    }
    break;
  case sen_ql_object :
    {
      sen_id id = car->u.o.self, base = car->class;
      int o, hashp = 0;
      sen_obj *slots, *val, *v;
      POP(slots, args);
      if (!LISTP(slots)) {
        disp_t(c, car, buf);
        return;
      }
      if (CAR(slots) == sen_ql_mk_symbol(c, "@")) {
        hashp = 1;
        slots = CDR(slots);
        if (!LISTP(slots)) {
          disp_t(c, car, buf);
          return;
        }
        for (o = 0, val = slots; ; o = 1 - o) {
          if (!o) {
          if (val != slots) { SEN_RBUF_PUTC(buf, '\t'); }
            disp_t(c, CAR(val), buf);
          }
          val = CDR(val);
          if (!LISTP(val)) { break; }
        }
        c->output(c, SEN_QL_MORE, c->data.ptr);
      }
      for (o = 0, val = slots; ; o = 1 - o) {
        if (hashp && !o) {
          val = CDR(val);
          if (!LISTP(val)) { break; }
        } else {
          sen_obj *t = slotexp_prepare(c, base, val, NULL);
          obj_obj_bind(&obj, base, id);
          v = slotexp_exec(c, t, &obj, NULL);
          disp_t(c, v, buf);
          val = CDR(val);
          if (!LISTP(val)) { break; }
          if (val != slots) { SEN_RBUF_PUTC(buf, '\t'); }
        }
      }
    }
    break;
  default :
    disp_t(c, car, buf);
    break;
  }
}

static void
disp_t(sen_ctx *c, sen_obj *obj, sen_rbuf *buf)
{
  if (!obj || obj == NIL) {
    SEN_RBUF_PUTS(buf, "()");
  } else if (obj == T) {
    SEN_RBUF_PUTS(buf, "#t");
  } else if (obj == F) {
    SEN_RBUF_PUTS(buf, "#f");
  } else {
    switch (obj->type) {
    case sen_ql_records :
      {
        int i;
        sen_id *rp;
        recinfo *ri;
        sen_obj o;
        const sen_recordh *rh;
        sen_records *r = obj->u.r.records;
        sen_records_rewind(r);
        obj_obj_bind(&o, obj->class, 0);
        for (i = 0;; i++) {
          if (!sen_records_next(r, NULL, 0, NULL) ||
              !(rh = sen_records_curr_rec(r)) ||
              sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) {
            break;
          }
          o.u.o.self = *rp;
          disp_t(c, &o, buf);
          c->output(c, SEN_QL_MORE, c->data.ptr);
        }
      }
      break;
    case sen_ql_list :
      if (obj->u.l.car == sen_ql_mk_symbol(c, ":")) {
        disp_t_with_format(c, obj->u.l.cdr, buf);
      } else if (obj->u.l.car == sen_ql_mk_symbol(c, "@")) {
        int o0, o;
        sen_obj *val = obj->u.l.cdr;
        for (o0 = 0; o0 <= 1; o0++) {
          for (obj = val, o = o0;; o = 1 - o) {
            if (!o) { disp_t(c, obj->u.l.car, buf); }
            if ((obj = obj->u.l.cdr) && (obj != NIL)) {
              if (LISTP(obj)) {
                if (!o) { SEN_RBUF_PUTC(buf, '\t'); } /* dot pair */
              } else {
                if (!o) {
                  SEN_RBUF_PUTC(buf, '\t'); /* dot pair */
                  disp_t(c, obj, buf);
                }
                break;
              }
            } else {
              break;
            }
          }
          c->output(c, SEN_QL_MORE, c->data.ptr);
        }
      } else {
        for (;;) {
          disp_t(c, obj->u.l.car, buf);
          if ((obj = obj->u.l.cdr) && (obj != NIL)) {
            if (LISTP(obj)) {
              SEN_RBUF_PUTC(buf, '\t');
            } else {
              SEN_RBUF_PUTC(buf, '\t'); /* dot pair */
              disp_t(c, obj, buf);
              c->output(c, SEN_QL_MORE, c->data.ptr);
              break;
            }
          } else {
            c->output(c, SEN_QL_MORE, c->data.ptr);
            break;
          }
        }
      }
      break;
    default :
      sen_obj_inspect(c, obj, buf, 0);
      break;
    }
  }
}

static sen_obj *
_native_method_disp(sen_ctx *c, sen_obj *args, sen_ql_co *co)
{
  char *str;
  sen_obj *val, *fmt;
  POP(val, args);
  POP(fmt, args);
  if ((str = str_value(c, fmt))) {
    switch (str[0]) {
    case 'j' : /* json */
    case 'J' :
      disp_j(c, val, &c->outbuf);
      break;
    case 's' : /* sexp */
    case 'S' :
      break;
    case 't' : /* tsv */
    case 'T' :
      disp_t(c, val, &c->outbuf);
      break;
    case 'x' : /* xml */
    case 'X' :
      break;
    }
  } else {
    /* default */
  }
  c->output(c, SEN_QL_MORE, c->data.ptr);
  return T;
}

void
sen_ql_def_db_methods(sen_ctx *c)
{
  sen_ql_def_native_method(c, "<db>", _native_method_db);
  sen_ql_def_native_method(c, "table", _native_method_table);
  sen_ql_def_native_method(c, "ptable", _native_method_ptable);
  sen_ql_def_native_method(c, "sen-query", _native_method_sen_query);
  sen_ql_def_native_method(c, "sen-snip", _native_method_sen_snip);
  sen_ql_def_native_method(c, "disp", _native_method_disp);
}
