/* Copyright(C) 2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "senna_in.h"
#include "ql.h"
#include "sym.h"

sen_ctx *
sen_ctx_new(void)
{
  sen_ctx *c = SEN_MALLOC(sizeof(sen_ctx));
  if (!c) { return NULL; }
  c->db = NULL;
  c->phs = NIL;
  c->doing = NULL;
  c->code = NIL;
  c->dump = NIL;
  c->op = 1; // OP_T0LVL
  c->args = NIL;
  c->envir = NIL;
  c->seqno = 0;
  c->lseqno = 0;
  c->nbinds = 0;
  c->nunbinds = 0;
  c->feed_mode = sen_ql_atonce;
  c->stat = SEN_QL_WAIT_EXPR;
  c->cur = NULL;
  c->str_end = NULL;
  c->batchmode = 0;
  c->gc_verbose = 0;
  c->inbuf = NULL;
  c->co.mode = 0;
  c->objects = NULL;
  c->symbols = NULL;
  c->com = NULL;
  sen_rbuf_init(&c->outbuf, 0);
  sen_rbuf_init(&c->subbuf, 0);
  return c;
}

sen_rc
sen_ctx_initql(sen_ctx *c)
{
  if (!(c->objects = sen_set_open(sizeof(int), sizeof(sen_obj), 0))) {
    return sen_memory_exhausted;
  }
  if (!(c->symbols = sen_set_open(0, sizeof(sen_obj), 0))) {
    sen_set_close(c->objects);
    return sen_memory_exhausted;
  }
  sen_ql_def_db_methods(c);
  sen_ql_init_globals(c);
  return sen_success;
}

sen_ctx *
sen_ctx_open(sen_db *db, int flags)
{
  sen_ctx *c = sen_ctx_new();
  if (!c) { return NULL; }
  if ((c->db = db)) { c->encoding = db->keys->encoding; }
  if (flags & SEN_CTX_USEQL) {
    if (sen_ctx_initql(c)) {
      sen_ctx_close(c);
      return NULL;
    }
  }
  if (flags & SEN_CTX_BATCHMODE) { c->batchmode = 1; }
  return c;
}

sen_ctx *
sen_ctx_connect(const char *host, int port, int flags)
{
  sen_ctx *c;
  sen_com_sqtp *com = sen_com_sqtp_copen(NULL, host, port);
  if (!com) { return NULL; }
  if ((c = sen_ctx_new())) {
    c->com = com;
    return c;
  } else {
    sen_com_sqtp_close(NULL, com);
    return NULL;
  }
}

sen_rc
sen_ctx_close(sen_ctx *c)
{
  sen_rc rc = sen_success;
  if (c->objects) {
    sen_obj *o;
    sen_set_cursor *sc;
    if ((sc = sen_set_cursor_open(c->objects))) {
      while (sen_set_cursor_next(sc, NULL, (void **) &o)) { sen_obj_clear(o); }
      sen_set_cursor_close(sc);
    }
    sen_set_close(c->objects);
  }
  if (c->symbols) {
    sen_set_close(c->symbols);
  }
  if (c->com) {
    rc = sen_com_sqtp_close(NULL, c->com);
  }
  rc = sen_rbuf_fin(&c->outbuf);
  rc = sen_rbuf_fin(&c->subbuf);
  SEN_FREE(c);
  return rc;
}

sen_rc
sen_ctx_send(sen_ctx *c, char *str, unsigned int str_len, int flags)
{
  if (c->com) {
    sen_com_sqtp_header sheader;
    sheader.flags = (flags & SEN_CTX_MORE) ? (SEN_COM_SQTP_MORE|SEN_COM_SQTP_QUIET) : 0;
    sheader.size = str_len;
    return sen_com_sqtp_send(c->com, &sheader, (char *)str);
  }
  if (c->objects) {
    sen_ql_feed(c, str, str_len, 0); // todo : handle flags
    if (!(flags & SEN_COM_SQTP_QUIET) && c->output) {
      c->output(c, 0, c->data.ptr); // todo : handle flags
    }
    return sen_success;
  }
  return sen_invalid_argument;
}

sen_rc
sen_ctx_recv(sen_ctx *c, char **str, unsigned int *str_len, int *flags)
{
  if (c->com) {
    if (sen_com_sqtp_recv(c->com, &c->com->msg, &c->com_status, &c->com_info)) {
      *str = NULL;
      *str_len = 0;
      *flags = 0;
    } else {
      sen_com_sqtp_header *rheader = SEN_COM_SQTP_MSG_HEADER(&c->com->msg);
      *str = SEN_COM_SQTP_MSG_BODY(&c->com->msg);
      *str_len = rheader->size;
      *flags = (rheader->flags & SEN_COM_SQTP_TAIL) ? 0 : SEN_CTX_MORE;
    }
    return c->com->rc;
  }
  if (c->objects) {
    sen_rbuf *buf = &c->outbuf;
    unsigned int head, tail;
    int *offsets = (unsigned int *) c->subbuf.head;
    int npackets = SEN_RBUF_VSIZE(&c->subbuf) / sizeof(unsigned int);
    if (npackets < c->bufcur) { return sen_invalid_argument; }
    head = c->bufcur ? offsets[c->bufcur - 1] : 0;
    tail = c->bufcur < npackets ? offsets[c->bufcur] : SEN_RBUF_VSIZE(buf);
    *str = buf->head + head;
    *str_len = tail - head;
    *flags = c->bufcur++ < npackets ? SEN_CTX_MORE : 0;
    return sen_success;
  }
  return sen_invalid_argument;
}

sen_rc
sen_ctx_recv_handler_set(sen_ctx *c, void (*func)(sen_ctx *, int, void *), void *func_arg)
{
  c->output = func;
  c->data.ptr = func_arg;
  return sen_success;
}

sen_rc
sen_ctx_info_get(sen_ctx *c, sen_ctx_info *info)
{
  if (!c) { return sen_invalid_argument; }
  if (c->com) {
    info->fd = c->com->com.fd;
    info->com_status = c->com_status;
    info->com_info = c->com_info;
    info->outbuf = &c->com->msg;
  } else {
    info->fd = -1;
    info->com_status = 0;
    info->com_info = 0;
    info->outbuf = &c->outbuf;
  }
  return sen_success;
}

sen_obj *
sen_obj_new(sen_ctx *c)
{
  sen_obj *o;
  do {
    if (!sen_set_get(c->objects, &c->seqno, (void **) &o)) {
      /* todo : must be handled */
    }
    c->seqno++;
  } while (o->type);
  o->flags = 0;
  o->nrefs = 0;
  return o;
}

sen_obj *
sen_obj_alloc(sen_ctx *c, uint32_t size)
{
  sen_obj *o;
  void *value = SEN_MALLOC(size + 1);
  if (!value) { return NULL; }
  o = sen_obj_new(c);
  o->flags = SEN_OBJ_ALLOCATED;
  o->type = sen_ql_bulk;
  o->u.b.size = size;
  o->u.b.value = value;
  return o;
}

sen_obj *
sen_obj_clear(sen_obj *o)
{
  if (o->flags & SEN_OBJ_ALLOCATED) {
    switch (o->type) {
    case sen_ql_records :
      if (o->u.r.records) { sen_records_close(o->u.r.records); }
      break;
    case sen_ql_bulk :
      if (o->u.b.value) { SEN_FREE(o->u.b.value); }
      break;
    case sen_ql_query :
      if (o->u.q.query) { sen_query_close(o->u.q.query); }
      break;
    default :
      break;
    }
  }
  o->flags = 0;
  return o;
}

