/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _SEN_STORE_H
#define _SEN_STORE_H

#ifndef _SENNA_H
#include "senna_in.h"
#endif /* _SENNA_H */

#ifndef _SEN_SET_H
#include "set.h"
#endif /* _SEN_SET_H */

#ifndef _SEN_IO_H
#include "io.h"
#endif /* _SEN_IO_H */

#ifndef _SEN_STR_H
#include "str.h"
#endif /* _SEN_STR_H */

#ifdef __cplusplus
extern "C" {
#endif

#define SEN_ST_APPEND 1

/**** fixed sized elements ****/

typedef struct _sen_ra sen_ra;

struct _sen_ra {
  sen_io *io;
  int element_width;
  int element_mask;
  struct sen_ra_header *header;
};

sen_ra *sen_ra_create(const char *path, unsigned int element_size);
sen_ra *sen_ra_open(const char *path);
sen_rc sen_ra_info(sen_ra *ra, unsigned int *element_size, sen_id *curr_max);
sen_rc sen_ra_close(sen_ra *ra);
sen_rc sen_ra_remove(const char *path);
void *sen_ra_get(sen_ra *ra, sen_id id);
void *sen_ra_at(sen_ra *ra, sen_id id);

/**** variable sized elements ****/

typedef struct _sen_ja sen_ja;
typedef struct _sen_ja_einfo sen_ja_einfo;

struct _sen_ja {
  sen_io *io;
  struct sen_ja_header *header;
};

sen_ja *sen_ja_create(const char *path, unsigned int max_element_size);
sen_ja *sen_ja_open(const char *path);
sen_rc sen_ja_info(sen_ja *ja, unsigned int *max_element_size);
sen_rc sen_ja_close(sen_ja *ja);
sen_rc sen_ja_remove(const char *path);
sen_rc sen_ja_put(sen_ja *ja, sen_id id, const void *value, int value_len, int flags);
int sen_ja_at(sen_ja *ja, sen_id id, void *valbuf, int buf_size);

const void *sen_ja_ref(sen_ja *ja, sen_id id, uint32_t *value_len);
sen_rc sen_ja_unref(sen_ja *ja, sen_id id);

int sen_ja_size(sen_ja *ja, sen_id id);
sen_rc sen_ja_alloc(sen_ja *ja, int element_size, sen_ja_einfo *einfo, void **value);
sen_rc sen_ja_replace(sen_ja *ja, sen_id id, sen_ja_einfo *ei);

/**** db ****/

typedef enum {
  sen_db_nil = 0,
  sen_db_raw_class,         /* unstructured data class (no slots, not enumerable) */
  sen_db_structured_class,  /* structured data class (has slots, enumerable) */
  sen_db_obj_slot,          /* slot storing a reference to a structured object */
  sen_db_ra_slot,           /* slot storing a flat data, equal or less than 8 bytes */
  sen_db_ja_slot,           /* slot storing a flat data, more than 8 bytes */
  sen_db_idx_slot,          /* slot storing an inverted index */
  sen_db_object,
  sen_db_records,
  sen_db_bulk,
  sen_db_list,
  sen_db_int,
  sen_db_float,
  sen_db_native_method,
  sen_db_method,
  sen_db_query
} sen_db_type;

typedef struct _sen_db sen_db;
typedef struct _sen_db_store_rel_spec sen_db_store_rel_spec;
typedef struct _sen_db_store_spec sen_db_store_spec;
typedef struct _sen_db_store sen_db_store;

typedef enum {
  sen_db_before_update_trigger = 0,
  sen_db_after_update_trigger,
  sen_db_index_target
} sen_db_rel_type;

struct _sen_db_store_rel_spec {
  sen_db_rel_type type;
  sen_id target;
};

struct _sen_db_store_spec {
  uint32_t type;
  uint32_t n_relations;
  union {
    struct {
      unsigned int size;
      unsigned int flags;
      sen_encoding encoding;
    } c;
    struct {
      sen_id class;
      unsigned int size;
      unsigned int collection_type;
    } s;
  } u;
  sen_db_store_rel_spec relations[1];
};

#define SEN_DB_STORE_SPEC_SIZE(n) \
  ((intptr_t)(&((sen_db_store_spec *)0)->relations[n]))

sen_db *sen_db_create(const char *path, int flags, sen_encoding encoding);
sen_db *sen_db_open(const char *path);
sen_rc sen_db_close(sen_db *s);

sen_db_store *sen_db_store_create(sen_db *s, const char *name, sen_db_store_spec *spec);
sen_db_store *sen_db_store_open(sen_db *s, const char *name);
sen_rc sen_db_store_remove(sen_db *s, const char *name);
sen_rc sen_db_store_add_relation(sen_db_store *s, sen_db_store_rel_spec *r);
sen_rc sen_db_store_del_relation(sen_db_store *s, sen_db_store_rel_spec *r);

/**** sen_db_obj ****/

typedef struct _sen_db_obj sen_db_obj;
typedef struct _sen_db_ctx sen_db_ctx;
typedef struct _sen_db_cont sen_db_cont;
typedef sen_db_obj *
sen_db_method_func(sen_db_ctx *, sen_db_obj *, sen_db_cont *);
typedef sen_db_obj *
sen_db_parser(sen_db_ctx *, const char *, uint32_t);

struct _sen_db_obj {
  uint8_t type;
  uint8_t flags;
  sen_id class;
  union {
    struct {
      sen_id self;
    } o;
    struct {
      sen_records *records;
    } r;
    struct {
      char *value;
      uint32_t size;
    } b;
    struct {
      sen_db_obj *car;
      sen_db_obj *cdr;
    } l;
    struct {
      sen_db_obj *func;
    } f;
    struct {
      sen_db_method_func *func;
    } n;
    struct {
      int32_t i;
    } i;
    struct {
      double d;
    } d;
    struct {
      sen_query *query;
    } q;
  } u;
};

#define SEN_DB_CTX_HEAD 1
#define SEN_DB_CTX_TAIL 2

#define SEN_DB_CONT_BEGIN(c) if (c) { switch((c)->last) { case 0:
#define SEN_DB_CONT_WAIT(c,d) \
  (c)->last = __LINE__; (c)->data = (d); return NULL;\
  case __LINE__: (d) = (c)->data;
#define SEN_DB_CONT_END(c) (c)->last = 0; }}

struct _sen_db_cont {
  uint16_t mode;
  uint16_t last;
  void *data;
};

struct _sen_db_ctx {
  uint32_t seqno;
  sen_db *db;
  sen_set *objects;
  sen_set *bindings;
  sen_db_parser *parser;
  sen_db_method_func *doing;
  sen_db_cont cont;
  union {
    void *ptr;
    int fd;
    uint32_t u32;
    uint64_t u64;
  } data;
};

sen_db_ctx *sen_db_ctx_open(sen_db *s, sen_db_parser *parser);
sen_rc sen_db_ctx_close(sen_db_ctx *c);

sen_rc sen_db_at(sen_db_ctx *c, sen_db_obj *obj,
                    const char *key, const sen_db_obj *args, sen_db_obj **res);
sen_rc sen_db_send(sen_db_ctx *c, sen_db_obj *obj,
                    const char *message, const sen_db_obj *args, sen_db_obj **res);
sen_db_obj *sen_db_ctx_eval(sen_db_ctx *c, sen_db_obj *sexp);
void sen_db_obj_inspect(sen_db *db, sen_db_obj *obj, sen_rbuf *buf);
void sen_db_ctx_def_native_method(sen_db_ctx *c, const char *name,
                                     sen_db_method_func *func);
const char *_sen_db_obj_key(sen_db *db, sen_db_obj *obj);
sen_db_obj *sen_db_ctx_feed(sen_db_ctx *c, char *str, uint32_t str_size, int mode);
sen_db_obj *sen_db_sexp_parser(sen_db_ctx *c, const char *str, uint32_t str_size);

#define SEN_OBJ2VALUE(o,v,s) ((v) = (o)->u.b.value, (s) = (o)->u.b.size)
#define SEN_VALUE2OBJ(o,v,s) ((o)->u.b.value = (v), (o)->u.b.size = (s))

#define NILP(c) (!(c) || (c)->type == sen_db_nil)
#define OBJECTP(c) ((c) && (c)->type == sen_db_object)
#define RECORDSP(c) ((c) && (c)->type == sen_db_records)
#define BULKP(c) ((c) && (c)->type == sen_db_bulk)
#define LISTP(c) ((c) && (c)->type == sen_db_list)
#define INTP(c) ((c) && (c)->type == sen_db_int)
#define NATIVE_METHODP(c) ((c) && (c)->type == sen_db_native_method)

#define SETNIL(c) ((c)->type = sen_db_nil)
#define SETINT(c,v) ((c)->type = sen_db_int, (c)->u.i.i = (v))
#define SETBULK(c,v,s) ((c)->type = sen_db_bulk, (c)->u.b.value = (v), (c)->u.b.size = (s))

#define CAR(c) (LISTP(c) ? (c)->u.l.car : NULL)
#define CDR(c) (LISTP(c) ? (c)->u.l.cdr : NULL)
#define CAAR(c) (CAR(CAR(c)))
#define CADR(c) (CAR(CDR(c)))
#define CDAR(c) (CDR(CAR(c)))
#define CDDR(c) (CDR(CDR(c)))

/**** vgram ****/

typedef struct _sen_vgram_vnode
{
  struct _sen_vgram_vnode *car;
  struct _sen_vgram_vnode *cdr;
  sen_id tid;
  sen_id vid;
  int freq;
  int len;
} sen_vgram_vnode;

struct _sen_vgram {
  sen_sym *vgram;
};

struct _sen_vgram_buf {
  size_t len;
  sen_id *tvs;
  sen_id *tvp;
  sen_id *tve;
  sen_vgram_vnode *vps;
  sen_vgram_vnode *vpp;
  sen_vgram_vnode *vpe;
};

sen_vgram *sen_vgram_create(const char *path);
sen_vgram *sen_vgram_open(const char *path);
sen_rc sen_vgram_close(sen_vgram *vgram);
sen_rc sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms);

sen_vgram_buf *sen_vgram_buf_open(size_t len);
sen_rc sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid);
sen_rc sen_vgram_buf_close(sen_vgram_buf *b);

#ifdef __cplusplus
}
#endif

#endif /* _SEN_STORE_H */
