/* Copyright(C) 2004 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <string.h>
#include "store.h"
#include "str.h"

/* comm */

typedef struct {
  sen_store *store;
  sen_store_ctx *ctx;
  int msgno;
  int seqno;
  int headp;
  int npackets;
  int stat;
} sen_comm_session;

sen_rc message_handler(sen_comm_session *s, const char *buf, int buf_len);

#define SEN_COMM_STDOUT (NULL)

#define SEN_COMM_AUTO 0
#define SEN_COMM_TAIL 1
#define SEN_COMM_HEAD 2

typedef enum {
  sen_comm_init,
  sen_comm_sending
} sen_comm_stat;

sen_comm_session *
sen_comm_open(const char *dest)
{
  sen_comm_session *s;
  sen_init();

  if (!(s = SEN_MALLOC(sizeof(sen_comm_session)))) {
    return NULL;
  }

  if (!(s->store = sen_store_open(dest))) {
    if (!(s->store = sen_store_create(dest, 0, sen_enc_none))) {
      fprintf(stderr, "store open failed (%s)\n", dest);
      return NULL;
    }
  }
  s->ctx = sen_store_ctx_open(s->store, sen_store_sexp_parser);
  s->msgno = 0;
  s->seqno = 0;
  s->headp = 0;
  s->npackets = 0;
  s->stat = sen_comm_init;
  return s;
}

sen_rc
sen_comm_send(sen_comm_session *s, const char *buf, int buf_len, int opt, int npackets)
{
  sen_rc rc = sen_success;
  if (s) {
    if (s->stat == sen_comm_init || (opt & SEN_COMM_HEAD)) {
      s->msgno++;
      s->npackets = npackets;
      s->headp = 1;
    } else {
      s->headp = 0;
    }
    if (opt & SEN_COMM_TAIL) {
      s->stat = sen_comm_init;
    }
    s->seqno++;
    rc = message_handler(s, buf, buf_len);
  } else {
    fwrite(buf, 1, buf_len, stdout);
    if (opt == SEN_COMM_TAIL) { putchar('\n'); }
  }
  return rc;
}

sen_rc
sen_comm_close(sen_comm_session *s)
{
  if (s) {
    if (s->stat == sen_comm_sending) {
      sen_comm_send(s, NULL, 0, SEN_COMM_TAIL, 0);
    }
    sen_store_ctx_close(s->ctx);
    sen_store_close(s->store);
    SEN_FREE(s);
  }
  sen_fin();
  return sen_success;
}

/* server */

void
show_obj(sen_store *store, sen_store_obj *obj)
{
  if (obj) {
    switch (obj->type) {
    case sen_store_nil :
      fprintf(stderr, "nil object");
      obj = NULL;
      break;
    case sen_store_builtin_class :
      fprintf(stderr, "Builtin Class (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_structured_class :
      fprintf(stderr, "Structured Class (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_obj_slot :
      fprintf(stderr, "Object Slot (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_ra_slot :
      fprintf(stderr, "RA Slot (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_ja_slot :
      fprintf(stderr, "JA Slot (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_idx_slot :
      fprintf(stderr, "Index Slot (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_object :
      fprintf(stderr, "Object (%s)", _sen_store_obj_key(store, obj));
      break;
    case sen_store_records :
      fprintf(stderr, "Records(%d)", sen_records_nhits(obj->u.r.records));
      break;
    case sen_store_bulk :
      fwrite(obj->u.b.value, 1, obj->u.b.size, stderr);
      break;
    case sen_store_list :
      fputc('(', stderr);
      for (;;) {
        show_obj(store, obj->u.l.car);
        if ((obj = obj->u.l.cdr)) {
          if (obj->type == sen_store_list) {
            fputc(' ', stderr);
          } else {
            fputs(" . ", stderr);
            show_obj(store, obj->u.l.cdr);
          }
        } else {
          fputc(')', stderr);
          break;
        }
      }
      break;
    case sen_store_str :
      fprintf(stderr, "String(%d) '%s'", obj->u.b.size, (char *)obj->u.b.value);
      break;
    case sen_store_native_method :
      fprintf(stderr, "Native method");
      break;
    case sen_store_method :
      fprintf(stderr, "Method");
      break;
    case sen_store_int :
      fprintf(stderr, "%d", obj->u.i.i);
      break;
    }
  } else {
    fputs("nil", stderr);
  }
}


sen_rc
message_handler(sen_comm_session *s, const char *buf, int buf_len)
{
  sen_store_obj *r;
  r = sen_store_ctx_feed(s->ctx, buf, buf_len, SEN_STORE_CTX_HEAD|SEN_STORE_CTX_TAIL);
  if (r) show_obj(s->store, r);
  return sen_success;
}

/* client */

#define BUFSIZE 0x100000

int
main(int argc, char **argv)
{
  int buf_len;
  char buf[BUFSIZE];
  sen_comm_session *s;
  if (argc < 2) {
    fputs("usage: stest pathname\n", stderr);
    return -1;
  }
  s = sen_comm_open(argv[1]);
  while (fputs("> ", stderr), fgets(buf, BUFSIZE, stdin)) {
    buf_len = strlen(buf) - 1;
    sen_comm_send(s, buf, buf_len, buf_len ? SEN_COMM_AUTO : SEN_COMM_TAIL, 0);
    fputs("\n", stderr);
  }
  sen_comm_close(s);
  return 0;
}
