/* Copyright(C) 2004 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "inv.h"
#include "sym.h"
#include "str.h"

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>

#define NEXT_ADDR(p) (((byte *)(p)) + sizeof *(p))

typedef struct {
  uint16_t step;
  uint16_t jump;
} buffer_rec;

#define BUFFER_REC_AT(b,pos) ((buffer_rec *)(b) + (pos))

int verbose;
int check;
int unlock;

static inline void
inv_inspect(sen_inv *inv, sen_sym *lexicon)
{
  int n_terms = 0;
  int n_terms_with_hits = 0;
  int total_df = 0;
  int total_nposts = 0;
  sen_id tid = SEN_SYM_NIL;
  sen_inv_cursor *c;
  if (verbose) { puts("     tid,      df,    nposts   |  term"); }
  while ((tid = sen_sym_next(lexicon, tid))) {
    int df = 0, nposts = 0;
    n_terms++;
    if (!(c = sen_inv_cursor_open(inv, tid, 1))) { continue; }
    n_terms_with_hits++;
    while (!sen_inv_cursor_next(c)) {
      df++;
      nposts += c->post->tf;
    }
    if (verbose) {
      printf("%8d,%8d,%10d   |  %s\n", tid, df, nposts, _sen_sym_key(lexicon, tid));
    }
    total_df += df;
    total_nposts += nposts;
    sen_inv_cursor_close(c);
  }
  printf("total amount of doc freq.      %12d\n", total_df);
  printf("total number of postings       %12d\n", total_nposts);
  printf("total number of terms          %12d\n", n_terms);
  printf("total number of terms w hits   %12d\n", n_terms_with_hits);
}

static inline void
index_inspect(sen_index *i)
{
  sen_rc rc;
  sen_encoding encoding;
  int key_size, flags, initial_n_segments;
  unsigned nrecords_keys, file_size_keys, nrecords_lexicon, file_size_lexicon, inv_seg_size, inv_chunk_size;
  rc = sen_index_info(i, &key_size, &flags, &initial_n_segments, &encoding, &nrecords_keys, &file_size_keys, &nrecords_lexicon, &file_size_lexicon, &inv_seg_size, &inv_chunk_size);
  if (rc) {
    fprintf(stderr, "sen_index_info failed (%d)", rc);
    return;
  }
  printf("lock               %24u\n", *i->keys->lock);
  printf("key_size           %24d\n", key_size);
  printf("flags              %24d\n", flags);
  printf("initial_n_segments %24d\n", initial_n_segments);
  printf("encoding           %24s\n", sen_enc_string[encoding]);
  printf("nrecords_keys      %24u\n", nrecords_keys);
  printf("file_size_keys     %24u\n", file_size_keys);
  printf("nrecords_lexicon   %24u\n", nrecords_lexicon);
  printf("file_size_lexicon  %24u\n", file_size_lexicon);
  printf("inv_segment_size   %24u\n", inv_seg_size);
  printf("inv_chunk_size     %24u\n", inv_chunk_size);
  inv_inspect(i->inv, i->lexicon);
  if (check) {
    printf("n of errors in inv %24d\n", sen_inv_check(i->inv));
  }
}

int
main(int argc, char **argv)
{
  int c;
  sen_index *i;
  verbose = 0;
  check = 0;
  unlock = 0;

  for (;;) {
    c = getopt(argc, argv, "v:c:u");
    if (c == -1) { break; }
    if (c == 'v') { verbose = 1; }
    if (c == 'c') { check = 1; }
    if (c == 'u') { unlock = 1; }
  }

  if (optind >= argc) {
    fputs("Usage: itest [-v] [-c] indexfile\n", stderr);
    return -1;
  }
  if (!(i = sen_index_open(argv[optind]))) {
    fprintf(stderr, "index open failed(%s)\n", argv[1]);
    return -1;
  }
  if (unlock) { *i->keys->lock = 0; }
  index_inspect(i);
  return 0;
}
