/* Copyright(C) 2004 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _SEN_STORE_H
#define _SEN_STORE_H

#ifndef _SENNA_H
#include "senna_in.h"
#endif /* _SENNA_H */

#ifndef _SEN_SET_H
#include "set.h"
#endif /* _SEN_SET_H */

#ifndef _SEN_IO_H
#include "io.h"
#endif /* _SEN_IO_H */

#ifdef __cplusplus
extern "C" {
#endif

/**** fixed sized elements ****/

typedef struct _sen_fse sen_fse;

struct _sen_fse {
  sen_io *io;
  int element_width;
  int element_mask;
  struct sen_fse_header *header;
};

sen_fse *sen_fse_create(const char *path, unsigned int element_size);
sen_fse *sen_fse_open(const char *path);
sen_rc sen_fse_info(sen_fse *fse, unsigned int *element_size, sen_id *curr_max);
sen_rc sen_fse_close(sen_fse *fse);
sen_rc sen_fse_remove(const char *path);
void *sen_fse_get(sen_fse *fse, sen_id id);
void *sen_fse_at(sen_fse *fse, sen_id id);

/**** variable sized elements ****/

typedef struct _sen_vse sen_vse;
typedef struct _sen_vse_einfo sen_vse_einfo;

struct _sen_vse {
  sen_io *io;
  struct sen_vse_header *header;
};

sen_vse *sen_vse_create(const char *path, unsigned int max_element_size);
sen_vse *sen_vse_open(const char *path);
sen_rc sen_vse_info(sen_vse *vse, unsigned int *max_element_size);
sen_rc sen_vse_close(sen_vse *vse);
sen_rc sen_vse_remove(const char *path);
sen_rc sen_vse_put(sen_vse *vse, sen_id id, const void *value, int value_len, int flags);
int sen_vse_at(sen_vse *vse, sen_id id, void *valbuf, int buf_size);

const void *sen_vse_ref(sen_vse *vse, sen_id id, int *value_len);
sen_rc sen_vse_unref(sen_vse *vse, sen_id id);

int sen_vse_size(sen_vse *vse, sen_id id);
sen_rc sen_vse_alloc(sen_vse *vse, int element_size, sen_vse_einfo *einfo, void **value);
sen_rc sen_vse_replace(sen_vse *vse, sen_id id, sen_vse_einfo *ei);

/**** store ****/

typedef struct _sen_store sen_store;
typedef struct _sen_class sen_class;
typedef struct _sen_obj sen_obj;
typedef struct _sen_slot sen_slot;
typedef struct _sen_class_spec sen_class_spec;
typedef struct _sen_slot_spec sen_slot_spec;

struct _sen_store {
  sen_sym *keys;
  sen_vse *values;
};

struct _sen_class {
  sen_sym *keys;
};

struct _sen_obj {
  sen_class *class;
  sen_id id;
  void *value;
  int size;
};

struct _sen_slot {
  char type;
  sen_class *class;
  sen_fse *fse;
  sen_vse *vse;
};

sen_store *sen_store_create(const char *path, int flags, sen_encoding encoding);
sen_store *sen_store_open(const char *path);
sen_rc sen_store_close(sen_store *s);

sen_class *sen_class_create(sen_store *s, const char *name, const char *spec);
sen_class *sen_class_open(sen_store *s, const char *name);
sen_rc sen_class_close(sen_class *);

sen_rc sen_class_at(sen_class *c, void *key, int flags, sen_obj *res);
sen_rc sen_obj_at(sen_obj *o, sen_slot *slot, sen_obj *res);
sen_rc sen_obj_update(sen_obj *o, sen_slot *slot, sen_obj *value);
int sen_obj_key(sen_obj *o, void *keybuf, int buf_size);

#define SEN_OBJ2VALUE(o,v,s) (v = (o)->value, s = (o)->size)
#define SEN_VALUE2OBJ(o,v,s) ((o)->value = v, (o)->size = s)

sen_slot *sen_slot_create(sen_store *s, const char *name, const char *spec);
sen_slot *sen_slot_open(sen_store *s, const char *name);
sen_rc sen_slot_close(sen_slot *);

/**** vgram ****/

typedef struct _sen_vgram_vnode
{
  struct _sen_vgram_vnode *car;
  struct _sen_vgram_vnode *cdr;
  sen_id tid;
  sen_id vid;
  int freq;
  int len;
} sen_vgram_vnode;

struct _sen_vgram {
  sen_sym *vgram;
};

struct _sen_vgram_buf {
  size_t len;
  sen_id *tvs;
  sen_id *tvp;
  sen_id *tve;
  sen_vgram_vnode *vps;
  sen_vgram_vnode *vpp;
  sen_vgram_vnode *vpe;
};

sen_vgram *sen_vgram_create(const char *path);
sen_vgram *sen_vgram_open(const char *path);
sen_rc sen_vgram_close(sen_vgram *vgram);
sen_rc sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms);

sen_vgram_buf *sen_vgram_buf_open(size_t len);
sen_rc sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid);
sen_rc sen_vgram_buf_close(sen_vgram_buf *b);

#ifdef __cplusplus
}
#endif

#endif /* _SEN_STORE_H */
