/* Copyright(C) 2004,2005,2006 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include "str.h"
#include "inv.h"
#include "sym.h"
#include "store.h"
#include <string.h>
#include <stdio.h>

/* fixed sized elements */

#define SEN_FSE_IDSTR "SENNA:FSE:01.00"
#define SEN_FSE_SEGMENT_SIZE (1 << 22)

#define SEN_FSE_MAX_CACHE (4294967295U)

struct sen_fse_header {
  char idstr[16];
  unsigned element_size;
  sen_id curr_max;
  uint32_t reserved[10];
};

sen_fse *
sen_fse_create(const char *path, unsigned int element_size)
{
  sen_io *io;
  int max_segments, n_elm, w_elm;
  sen_fse *fse = NULL;
  struct sen_fse_header *header;
  unsigned actual_size;
  if (element_size > SEN_FSE_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "element_size too large (%d)", element_size);
    return NULL;
  }
  for (actual_size = 1; actual_size < element_size; actual_size *= 2) ;
  max_segments = ((SEN_ID_MAX + 1) / SEN_FSE_SEGMENT_SIZE) * actual_size;
  io = sen_io_create(path, sizeof(struct sen_fse_header),
                     SEN_FSE_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_FSE_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_FSE_IDSTR, 16);
  header->element_size = actual_size;
  header->curr_max = 0;
  if (!(fse = SEN_MALLOC(sizeof(sen_fse)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_FSE_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  fse->io = io;
  fse->header = header;
  fse->element_mask =  n_elm - 1;
  fse->element_width = w_elm;
  return fse;
}

sen_fse *
sen_fse_open(const char *path)
{
  sen_io *io;
  int n_elm, w_elm;
  sen_fse *fse = NULL;
  struct sen_fse_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_FSE_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_FSE_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "fse_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(fse = SEN_MALLOC(sizeof(sen_fse)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_FSE_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  fse->io = io;
  fse->header = header;
  fse->element_mask =  n_elm - 1;
  fse->element_width = w_elm;
  return fse;
}

sen_rc
sen_fse_info(sen_fse *fse, unsigned int *element_size, sen_id *curr_max)
{
  if (!fse) { return sen_invalid_argument; }
  if (element_size) { *element_size = fse->header->element_size; }
  if (curr_max) { *curr_max = fse->header->curr_max; }
  return sen_success;
}

sen_rc
sen_fse_close(sen_fse *fse)
{
  sen_rc rc;
  if (!fse) { return sen_invalid_argument; }
  rc = sen_io_close(fse->io);
  SEN_FREE(fse);
  return rc;
}

sen_rc
sen_fse_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  sen_io_remove(path);
  return sen_success;
}

void *
sen_fse_get(sen_fse *fse, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > SEN_ID_MAX) { return NULL; }
  seg = id >> fse->element_width;
  SEN_IO_SEG_MAP(fse->io, seg, p);
  if (!p) { return NULL; }
  if (id > fse->header->curr_max) { fse->header->curr_max = id; }
  return (void *)(((byte *)p) + ((id & fse->element_mask) * fse->header->element_size));
}

void *
sen_fse_at(sen_fse *fse, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > fse->header->curr_max) { return NULL; }
  seg = id >> fse->element_width;
  SEN_IO_SEG_MAP(fse->io, seg, p);
  if (!p) { return NULL; }
  return (void *)(((byte *)p) + ((id & fse->element_mask) * fse->header->element_size));
}

/**** variable sized elements ****/

#define SEN_VSE_IDSTR "SENNA:VSE:01.00"

#define W_OF_VSE_MAX 38
#define W_OF_VSE_SEGMENT 22
#define W_OF_VSE_MAX_SEGMENTS (W_OF_VSE_MAX - W_OF_VSE_SEGMENT)

#define W_OF_VSE_EINFO 3
#define W_OF_VSE_EINFO_IN_A_SEGMENT (W_OF_VSE_SEGMENT - W_OF_VSE_EINFO)
#define N_OF_VSE_EINFO_IN_A_SEGMENT (1U << W_OF_VSE_EINFO_IN_A_SEGMENT)
#define VSE_EINFO_MASK (N_OF_VSE_EINFO_IN_A_SEGMENT - 1)

#define VSE_SEGMENT_SIZE (1U << W_OF_VSE_SEGMENT)
#define VSE_MAX_SEGMENTS (1U << W_OF_VSE_MAX_SEGMENTS)

#define VSE_BSA_SIZE (1U << (W_OF_VSE_SEGMENT - 7))
#define VSE_N_BSEGMENTS (1U << (W_OF_VSE_MAX_SEGMENTS - 7))

#define VSE_N_ESEGMENTS (1U << 9)

#define SEN_VSE_MAX_CACHE (4294967295U)

struct _sen_vse_einfo {
  union {
    uint64_t ll;
    struct {
      uint16_t seg;
      uint16_t pos;
      uint16_t size;
      uint8_t tail[2];
    } s;
  } u;
};

#define EINFO_SET(e,_seg,_pos,_size) {\
  (e)->u.s.seg = _seg;\
  (e)->u.s.pos = (_pos) >> 4;\
  (e)->u.s.size = _size;\
  (e)->u.s.tail[0] = (((_pos) >> 14) & 0xc0) + ((_size) >> 16);\
  (e)->u.s.tail[1] = 0;\
}

#define EINFO_GET(e,_seg,_pos,_size) {\
  _seg = (e)->u.s.seg;\
  _pos = ((e)->u.s.pos + (((e)->u.s.tail[0] & 0xc0) << 10)) << 4;\
  _size = (e)->u.s.size + (((e)->u.s.tail[0] & 0x3f) << 16);\
}

typedef struct {
  uint32_t seg;
  uint32_t pos;
} vse_pos;

struct sen_vse_header {
  char idstr[16];
  unsigned max_element_size;
  unsigned max_segments;
  vse_pos free_elements[24];
  uint8_t segments[VSE_MAX_SEGMENTS];
  uint32_t esegs[VSE_N_ESEGMENTS];
  uint32_t bsegs[VSE_N_BSEGMENTS];
};


#define VSE_SEG_ESEG 1;
#define VSE_SEG_BSEG 2;
#define SEG_NOT_ASSIGNED 0xffffffff

sen_vse *
sen_vse_create(const char *path, unsigned int max_element_size)
{
  int i;
  sen_io *io;
  int max_segments;
  sen_vse *vse = NULL;
  struct sen_vse_header *header;
  if (max_element_size > VSE_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "max_element_size too large (%d)", max_element_size);
    return NULL;
  }
  max_segments = max_element_size * 128;
  if (max_segments > VSE_MAX_SEGMENTS) { max_segments = VSE_MAX_SEGMENTS; }
  io = sen_io_create(path, sizeof(struct sen_vse_header),
                     VSE_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_VSE_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_VSE_IDSTR, 16);
  for (i = 0; i < VSE_N_ESEGMENTS; i++) { header->esegs[i] = SEG_NOT_ASSIGNED; }
  for (i = 0; i < VSE_N_BSEGMENTS; i++) { header->bsegs[i] = SEG_NOT_ASSIGNED; }
  header->max_element_size = max_element_size;
  header->max_segments = max_segments;
  header->segments[0] = VSE_SEG_ESEG;
  header->esegs[0] = 0;
  if (!(vse = SEN_MALLOC(sizeof(sen_vse)))) {
    sen_io_close(io);
    return NULL;
  }
  vse->io = io;
  vse->header = header;
  return vse;
}

sen_vse *
sen_vse_open(const char *path)
{
  sen_io *io;
  sen_vse *vse = NULL;
  struct sen_vse_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_VSE_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_VSE_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "vse_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(vse = SEN_MALLOC(sizeof(sen_vse)))) {
    sen_io_close(io);
    return NULL;
  }
  vse->io = io;
  vse->header = header;
  return vse;
}

sen_rc
sen_vse_info(sen_vse *vse, unsigned int *max_element_size)
{
  if (!vse) { return sen_invalid_argument; }
  return sen_success;
}

sen_rc
sen_vse_close(sen_vse *vse)
{
  sen_rc rc;
  if (!vse) { return sen_invalid_argument; }
  rc = sen_io_close(vse->io);
  SEN_FREE(vse);
  return rc;
}

sen_rc
sen_vse_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  sen_io_remove(path);
  return sen_success;
}

sen_rc
sen_vse_put(sen_vse *vse, sen_id id, const void *value, int value_len, int flags)
{
  int rc;
  void *buf;
  sen_vse_einfo einfo;
  // todo: handle flags
  if ((rc = sen_vse_alloc(vse, value_len, &einfo, &buf))) { return rc; }
  // printf("put id=%d, value_len=%d value=%p ei=%p(%d:%d)\n", id, value_len, buf, &einfo, einfo.u.s.pos, einfo.u.s.tail[0]);
  memcpy(buf, value, value_len);
  return sen_vse_replace(vse, id, &einfo);
}

int
sen_vse_at(sen_vse *vse, sen_id id, void *valbuf, int buf_size)
{
  int len;
  const void *value = sen_vse_ref(vse, id, &len);
  if (!value) { return -1; }
  if (buf_size >= len) { memcpy(valbuf, value, len); }
  sen_vse_unref(vse, id);
  return len;
}

const void *
sen_vse_ref(sen_vse *vse, sen_id id, int *value_len)
{
  sen_vse_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_VSE_EINFO_IN_A_SEGMENT;
  pos = id & VSE_EINFO_MASK;
  pseg = &vse->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { return NULL; }
  SEN_IO_SEG_MAP(vse->io, *pseg, einfo);
  if (!einfo) { return NULL; }
  if (einfo[pos].u.s.tail[1] & 1) {
    *value_len = einfo[pos].u.s.tail[1] >> 1;
    return (void *) &einfo[pos];
  }
  {
    void *value;
    uint32_t vseg, vpos, vsize;
    EINFO_GET(&einfo[pos], vseg, vpos, vsize);
    SEN_IO_SEG_MAP(vse->io, vseg, value);
    // printf("at id=%d value=%p vseg=%d vpos=%d ei=%p(%d:%d)\n", id, value, vseg, vpos, &einfo[pos], einfo[pos].u.s.pos, einfo[pos].u.s.tail[0]);
    if (!value) { return NULL; }
    *value_len = vsize;
    return (byte *)value + vpos;
  }
}

sen_rc
sen_vse_unref(sen_vse *vse, sen_id id)
{
  // todo
  return sen_success;
}

int
sen_vse_size(sen_vse *vse, sen_id id)
{
  sen_vse_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_VSE_EINFO_IN_A_SEGMENT;
  pos = id & VSE_EINFO_MASK;
  pseg = &vse->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { return -1; }
  SEN_IO_SEG_MAP(vse->io, *pseg, einfo);
  if (!einfo) { return -1; }
  if (einfo[pos].u.s.tail[1] & 1) {
    return einfo[pos].u.s.tail[1] >> 1;
  } else {
    return einfo[pos].u.s.size + ((einfo[pos].u.s.tail[0] & 0x3f) << 16);
  }
}

sen_rc
sen_vse_alloc(sen_vse *vse, int element_size, sen_vse_einfo *einfo, void **value)
{
  int m, size;
  void *addr;
  vse_pos *vp;
  if (element_size < 8) {
    einfo->u.s.tail[1] = element_size * 2 + 1;
    *value = (void *)einfo;
    return sen_success;
  }
  if (element_size >= vse->header->max_element_size) {
    return sen_invalid_argument;
  }
  for (m = 4, size = 16; size < element_size; m++, size *= 2);
  vp = &vse->header->free_elements[m];
  if (!vp->seg) {
    int i = 0;
    while (vse->header->segments[i]) {
      if (++i >= vse->header->max_segments) { return sen_memory_exhausted; }
    }
    vse->header->segments[i] = m;
    vp->seg = i;
    vp->pos = 0;
  }
  EINFO_SET(einfo, vp->seg, vp->pos, element_size);
  SEN_IO_SEG_MAP(vse->io, vp->seg, addr);
  // printf("addr=%p seg=%d pos=%d\n", addr, vp->seg, vp->pos);
  if (!addr) { return sen_memory_exhausted; }
  *value = (byte *)addr + vp->pos;
  if ((vp->pos += size) == VSE_SEGMENT_SIZE) {
    vp->seg = 0;
    vp->pos = 0;
  }
  return sen_success;
}

sen_rc
sen_vse_free(sen_vse *vse, sen_vse_einfo *einfo)
{
  uint32_t seg, pos, size;
  if (einfo->u.s.tail[1] & 1) { return sen_success; }
  EINFO_GET(einfo, seg, pos, size);
  // free
  return sen_success;
}

sen_rc
sen_vse_replace(sen_vse *vse, sen_id id, sen_vse_einfo *ei)
{
  uint32_t lseg, *pseg, pos;
  sen_vse_einfo *einfo, eback;
  lseg = id >> W_OF_VSE_EINFO_IN_A_SEGMENT;
  pos = id & VSE_EINFO_MASK;
  pseg = &vse->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) {
    int i = 0;
    while (vse->header->segments[i]) {
      if (++i >= vse->header->max_segments) { return sen_memory_exhausted; }
    }
    vse->header->segments[i] = 1;
    *pseg = i;
  }
  SEN_IO_SEG_MAP(vse->io, *pseg, einfo);
  if (!einfo) { return sen_memory_exhausted; }
  eback = einfo[pos];
  einfo[pos] = *ei;
  // todo: SEN_ATOMIC_SET64
  sen_vse_free(vse, &eback);
  return sen_success;
}

/**** store ****/

sen_store *
sen_store_create(const char *path, int flags, sen_encoding encoding)
{
  sen_store *s;
  char buffer[PATH_MAX];
  if (!(s = SEN_MALLOC(sizeof(sen_store)))) { return NULL; }
  strcpy(buffer, path);
  strcat(buffer, ".SEN.k");
  if ((s->keys = sen_sym_create(buffer, 0, flags, encoding))) {
    strcpy(buffer, path);
    strcat(buffer, ".SEN.v");
    if ((s->values = sen_vse_create(buffer, VSE_SEGMENT_SIZE))) {
      SEN_LOG(sen_log_notice, "store created (%s) flags=%x", path, s->keys->flags);
      return s;
    }
    sen_sym_close(s->keys);
  }
  SEN_FREE(s);
  return NULL;
}

sen_store *
sen_store_open(const char *path)
{
  sen_store *s;
  char buffer[PATH_MAX];
  if (!(s = SEN_MALLOC(sizeof(sen_store)))) { return NULL; }
  strcpy(buffer, path);
  strcat(buffer, ".SEN");
  if ((s->keys = sen_sym_open(buffer))) {
    strcat(buffer, ".v");
    if ((s->values = sen_vse_open(buffer))) {
      SEN_LOG(sen_log_notice, "store opened (%s) flags=%x", path, s->keys->flags);
      return s;
    }
    sen_sym_close(s->keys);
  }
  SEN_FREE(s);
  return NULL;
}

sen_rc
sen_store_close(sen_store *s)
{
  sen_sym_close(s->keys);
  sen_vse_close(s->values);
  SEN_FREE(s);
  return sen_success;
}

sen_class *
sen_class_create(sen_store *s, const char *name, const char *spec)
{
  sen_id id;
  sen_class *c;
  char buffer[PATH_MAX];
  if (!(c = SEN_MALLOC(sizeof(sen_class)))) { return NULL; }
  if ((id = sen_sym_get(s->keys, name))) {
    if (!sen_vse_put(s->values, id, spec, strlen(spec), 0)) {
      snprintf(buffer, PATH_MAX, "%s.%07x", s->keys->io->path, id);
      c->keys = sen_sym_create(buffer, 0, 0, sen_enc_none); // todo
      return c;
    }
    sen_sym_del(s->keys, name);
  }
  SEN_FREE(c);
  return NULL;
}

sen_class *
sen_class_open(sen_store *s, const char *name)
{
  sen_id id;
  sen_class *c;
  char buffer[PATH_MAX];
  if (!(c = SEN_MALLOC(sizeof(sen_class)))) { return NULL; }
  if ((id = sen_sym_at(s->keys, name))) {
    if (sen_vse_at(s->values, id, NULL, 0) != -1) {
      snprintf(buffer, PATH_MAX, "%s.%07x", s->keys->io->path, id);
      c->keys = sen_sym_open(buffer);
      return c;
    }
  }
  SEN_FREE(c);
  return NULL;
}

sen_rc
sen_class_close(sen_class *c)
{
  sen_sym_close(c->keys);
  SEN_FREE(c);
  return sen_success;
}

sen_rc
sen_class_at(sen_class *c, void *key, int flags, sen_obj *res)
{
  res->class = c;
  res->id = flags ? sen_sym_get(c->keys, key) : sen_sym_at(c->keys, key);
  return res->id ? sen_success : sen_other_error;
}

sen_rc
sen_obj_at(sen_obj *o, sen_slot *slot, sen_obj *res)
{
  switch (slot->type) {
  case 'c' :
    res->class = slot->class;
    res->id = *((sen_id *)sen_fse_at(slot->fse, o->id));
    break;
  case 'f' :
    res->value = sen_fse_at(slot->fse, o->id);
    res->size = slot->fse->header->element_size;
    break;
  case 'v' :
    res->value = (void *)sen_vse_ref(slot->vse, o->id, &res->size); // todo : unref
    break;
  }
  return sen_success;
}

sen_rc
sen_obj_update(sen_obj *o, sen_slot *slot, sen_obj *value)
{
  sen_rc rc = sen_success;
  switch (slot->type) {
  case 'c' :
    {
      sen_id *id;
      SEN_ASSERT(slot->class == value->class);
      if (!(id = (sen_id *)sen_fse_get(slot->fse, o->id))) {
        return sen_invalid_argument;
      }
      *id = value->id;
    }
    break;
  case 'f' :
    {
      void *e;
      SEN_ASSERT(slot->fse->header->element_size == value->size);
      if (!(e = sen_fse_get(slot->fse, o->id))) {
        return sen_invalid_argument;
      }
      memcpy(e, value->value, value->size);
    }
    break;
  case 'v' :
    rc =  sen_vse_put(slot->vse, o->id, value->value, value->size, 0);
    break;
  }
  return rc;
}

int
sen_obj_key(sen_obj *o, void *keybuf, int buf_size)
{
  return sen_sym_key(o->class->keys, o->id, keybuf, buf_size);
}

sen_slot *
sen_slot_create(sen_store *s, const char *name, const char *spec)
{
  sen_id id;
  sen_slot *slot;
  char buffer[PATH_MAX];
  if (!(slot = SEN_MALLOC(sizeof(sen_slot)))) { return NULL; }
  if ((id = sen_sym_get(s->keys, name))) {
    if (!sen_vse_put(s->values, id, spec, strlen(spec), 0)) {
      snprintf(buffer, PATH_MAX, "%s.%07x", s->keys->io->path, id);
      slot->type = *spec++;
      switch (slot->type) {
      case 'c' :
        slot->class = sen_class_open(s, spec); // todo : memory leak
        slot->fse = sen_fse_create(buffer, sizeof(sen_id)); // todo : error handling
        break;
      case 'f' :
        slot->fse = sen_fse_create(buffer, atoi(spec)); // todo : error handling
        break;
      case 'v' :
        slot->vse = sen_vse_create(buffer, atoi(spec)); // todo : error handling
        break;
      }
      return slot;
    }
    sen_sym_del(s->keys, name);
  }
  SEN_FREE(slot);
  return NULL;
}

sen_slot *
sen_slot_open(sen_store *s, const char *name)
{
  sen_id id;
  sen_slot *slot;
  char buffer[PATH_MAX];
  char spec[1024]; //todo
  if (!(slot = SEN_MALLOC(sizeof(sen_slot)))) { return NULL; }
  if ((id = sen_sym_at(s->keys, name))) {
    if (!sen_vse_at(s->values, id, spec, 1024)) {
      snprintf(buffer, PATH_MAX, "%s.%07x", s->keys->io->path, id);
      slot->type = *spec;
      switch (slot->type) {
      case 'c' :
        slot->class = sen_class_open(s, spec + 1); // todo : memory leak
        slot->fse = sen_fse_open(buffer); // todo : error handling
        break;
      case 'f' :
        slot->fse = sen_fse_open(buffer); // todo : error handling
        break;
      case 'v' :
        slot->vse = sen_vse_open(buffer); // todo : error handling
        break;
      }
      return slot;
    }
  }
  SEN_FREE(slot);
  return NULL;
}

sen_rc
sen_slot_close(sen_slot *slot)
{
  switch (slot->type) {
  case 'c' :
    sen_class_close(slot->class); // todo : memory leak
    sen_fse_close(slot->fse); // todo : error handling
    break;
  case 'f' :
    sen_fse_close(slot->fse); // todo : error handling
    break;
  case 'v' :
    sen_vse_close(slot->vse); // todo : error handling
    break;
  }
  SEN_FREE(slot);
  return sen_success;
}

/**** vgram ****/

static int len_sum = 0;
static int img_sum = 0;
static int simple_sum = 0;
static int skip_sum = 0;

sen_vgram *
sen_vgram_create(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_create(path, sizeof(sen_id) * 2, 0, sen_enc_none);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram *
sen_vgram_open(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_open(path);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram_buf *
sen_vgram_buf_open(size_t len)
{
  sen_vgram_buf *b;
  if (!(b = SEN_MALLOC(sizeof(sen_vgram_buf)))) { return NULL; }
  b->len = len;
  b->tvs = b->tvp = SEN_MALLOC(sizeof(sen_id) * len);
  if (!b->tvp) { SEN_FREE(b); return NULL; }
  b->tve = b->tvs + len;
  b->vps = b->vpp = SEN_MALLOC(sizeof(sen_vgram_vnode) * len * 2);
  if (!b->vpp) { SEN_FREE(b->tvp); SEN_FREE(b); return NULL; }
  b->vpe = b->vps + len;
  return b;
}

sen_rc
sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid)
{
  uint8_t dummybuf[8], *dummyp;
  if (b->tvp < b->tve) { *b->tvp++ = tid; }
  dummyp = dummybuf;
  SEN_B_ENC(tid, dummyp);
  simple_sum += dummyp - dummybuf;
  return sen_success;
}

typedef struct {
  sen_id vid;
  sen_id tid;
} vgram_key;

sen_rc
sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms)
{
  sen_inv_updspec **u;
  if (b && b->tvs < b->tvp) {
    sen_id *t0, *tn;
    for (t0 = b->tvs; t0 < b->tvp - 1; t0++) {
      sen_vgram_vnode *v, **vp;
      sen_set_at(terms, t0, (void **) &u);
      vp = &(*u)->vnodes;
      for (tn = t0 + 1; tn < b->tvp; tn++) {
        for (v = *vp; v && v->tid != *tn; v = v->cdr) ;
        if (!v) {
          if (b->vpp < b->vpe) {
            v = b->vpp++;
          } else {
            // todo;
            break;
          }
          v->car = NULL;
          v->cdr = *vp;
          *vp = v;
          v->tid = *tn;
          v->vid = 0;
          v->freq = 0;
          v->len = tn - t0;
        }
        v->freq++;
        if (v->vid) {
          vp = &v->car;
        } else {
          break;
        }
      }
    }
    {
      sen_set *th = sen_set_open(sizeof(sen_id), sizeof(int), 0);
      if (!th) { return sen_memory_exhausted; }
      if (t0 == b->tvp) { SEN_LOG(sen_log_debug, "t0 == tvp"); }
      for (t0 = b->tvs; t0 < b->tvp; t0++) {
        sen_id vid, vid0 = *t0, vid1 = 0;
        sen_vgram_vnode *v, *v2 = NULL, **vp;
        sen_set_at(terms, t0, (void **) &u);
        vp = &(*u)->vnodes;
        for (tn = t0 + 1; tn < b->tvp; tn++) {
          for (v = *vp; v; v = v->cdr) {
            if (!v->vid && (v->freq < 2 || v->freq * v->len < 4)) {
              *vp = v->cdr;
              v->freq = 0;
            }
            if (v->tid == *tn) { break; }
            vp = &v->cdr;
          }
          if (v) {
            if (v->freq) {
              v2 = v;
              vid1 = vid0;
              vid0 = v->vid;
            }
            if (v->vid) {
              vp = &v->car;
              continue;
            }
          }
          break;
        }
        if (v2) {
          if (!v2->vid) {
            vgram_key key;
            key.vid = vid1;
            key.tid = v2->tid;
            v2->vid = sen_sym_get(vgram->vgram, (char *)&key);
          }
          vid = *t0 = v2->vid * 2 + 1;
          memset(t0 + 1, 0, sizeof(sen_id) * v2->len);
          t0 += v2->len;
        } else {
          vid = *t0 *= 2;
        }
        {
          int *tf;
          sen_set_get(th, &vid, (void **) &tf);
          (*tf)++;
        }
      }
      if (!th->n_entries) { SEN_LOG(sen_log_debug, "th->n_entries == 0"); }
      {
        int j = 0;
        int skip = 0;
        sen_set_eh *ehs, *ehp, *ehe;
        sen_set_sort_optarg arg;
        uint8_t *ps = SEN_MALLOC(b->len * 2), *pp, *pe;
        if (!ps) {
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        pp = ps;
        pe = ps + b->len * 2;
        arg.mode = sen_sort_descending;
        arg.compar = NULL;
        arg.compar_arg = (void *)(intptr_t)sizeof(sen_id);
        arg.compar_arg0 = NULL;
        ehs = sen_set_sort(th, 0, &arg);
        if (!ehs) {
          SEN_FREE(ps);
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        SEN_B_ENC(th->n_entries, pp);
        for (ehp = ehs, ehe = ehs + th->n_entries; ehp < ehe; ehp++, j++) {
          int *id = (int *)SEN_SET_INTVAL(*ehp);
          SEN_B_ENC(*SEN_SET_INTKEY(*ehp), pp);
          *id = j;
        }
        for (t0 = b->tvs; t0 < b->tvp; t0++) {
          if (*t0) {
            int *id;
            if (!sen_set_at(th, t0, (void **) &id)) {
              SEN_LOG(sen_log_error, "lookup error (%d)", *t0);
            }
            SEN_B_ENC(*id, pp);
          } else {
            skip++;
          }
        }
        len_sum += b->len;
        img_sum += pp - ps;
        skip_sum += skip;
        SEN_FREE(ehs);
        SEN_FREE(ps);
      }
      sen_set_close(th);
    }
  }
  return sen_success;
}

sen_rc
sen_vgram_buf_close(sen_vgram_buf *b)
{
  if (!b) { return sen_invalid_argument; }
  if (b->tvs) { SEN_FREE(b->tvs); }
  if (b->vps) { SEN_FREE(b->vps); }
  SEN_FREE(b);
  return sen_success;
}

sen_rc
sen_vgram_close(sen_vgram *vgram)
{
  if (!vgram) { return sen_invalid_argument; }
  SEN_LOG(sen_log_debug, "len=%d img=%d skip=%d simple=%d", len_sum, img_sum, skip_sum, simple_sum);
  sen_sym_close(vgram->vgram);
  SEN_FREE(vgram);
  return sen_success;
}
