/* Copyright(C) 2004 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>

#include "str.h"
#include "sym.h"
#include "inv.h"

#define SEN_INV_IDSTR "SENNA:INV:00.00"
#define SEN_INV_SEGMENT_SIZE 0x40000
/* SEN_INV_MAX_SEGMENT == 0x10000 >> 2 */
#define SEN_INV_CHUNK_SIZE   0x40000
#define N_CHUNKS_PER_FILE (SEN_IO_FILE_SIZE / SEN_INV_CHUNK_SIZE)
#define W_OF_SEGMENT 18
#define W_OF_ARRAY (W_OF_SEGMENT - 2)
#define ARRAY_MASK_IN_A_SEGMENT (SEN_INV_SEGMENT_SIZE / sizeof(uintptr_t) - 1)
#define BUFFER_MASK_IN_A_SEGMENT (SEN_INV_SEGMENT_SIZE - 1)
#define CHUNK_NOT_ASSIGNED 0xffffffff

#define SEGMENT_ARRAY 0x8000
#define SEGMENT_BUFFER 0x4000
#define SEGMENT_MASK (SEN_INV_MAX_SEGMENT - 1)

#define BIT11_01(x) ((x >> 1) & 0x7ff)
#define BIT31_12(x) (x >> 12)

#define SEN_INV_INITIAL_N_SEGMENTS 512
#define MAX_CHUNK_RATIO 64

#define NEXT_ADDR(p) (((byte *)(p)) + sizeof *(p))

/* segment */

inline static sen_rc
segment_get(sen_inv *inv, uint16_t type, uint32_t segno, sen_io_seginfo *si)
{
  uint16_t s;
  int i, empty = SEN_INV_MAX_SEGMENT;
  for (i = 0; i < SEN_INV_MAX_SEGMENT; i++) {
    s = inv->header->segments[i];
    if (s) {
      if (s == (type | segno)) { break; }
    } else {
      if (empty == SEN_INV_MAX_SEGMENT) { empty = i; }
    }
  }
  if (i == SEN_INV_MAX_SEGMENT) {
    if (empty == SEN_INV_MAX_SEGMENT) { return sen_memory_exhausted; }
    inv->header->segments[empty] = type | segno;
    si->segno = empty;
    if (sen_io_seg_bind(inv->seg, si)) {
      return sen_memory_exhausted;
    }
    memset(si->addr, 0, SEN_INV_SEGMENT_SIZE);
  } else {
    si->segno = i;
  }
  return sen_success;
}

inline static sen_rc
segment_new(sen_inv *inv, uint16_t type, uint32_t *segno)
{
  uint16_t s;
  int i, seg, empty = SEN_INV_MAX_SEGMENT;
  sen_rc rc = sen_success;
  char used[SEN_INV_MAX_SEGMENT];
  memset(used, 0, SEN_INV_MAX_SEGMENT);
  for (i = 0; i < SEN_INV_MAX_SEGMENT; i++) {
    if ((s = inv->header->segments[i])) {
      if (s & type) { used[s & SEGMENT_MASK]++; }
    } else {
      if (empty == SEN_INV_MAX_SEGMENT) { empty = i; }
    }
  }
  if (empty == SEN_INV_MAX_SEGMENT) { return sen_memory_exhausted; }
  if (segno && *segno < SEN_INV_MAX_SEGMENT) {
    if (used[*segno]) { return sen_invalid_argument; }
    seg = *segno;
  } else {
    for (seg = 0; used[seg]; seg++) ;
  }
  inv->header->segments[empty] = type | seg;
  switch (type) {
  case SEGMENT_ARRAY :
    inv->ainfo[seg].segno = empty;
    break;
  case SEGMENT_BUFFER :
    inv->binfo[seg].segno = empty;
    break;
  }
  if (segno) { *segno = seg; }
  return rc;
}

inline static sen_rc
load_all_segments(sen_inv *inv)
{
  sen_rc rc = sen_success;
  int seg;
  uint16_t s;
  for (seg = 0; seg < SEN_INV_MAX_SEGMENT; seg++) {
    if (!(s = inv->header->segments[seg])) { continue; }
    if (s & SEGMENT_ARRAY) {
      inv->ainfo[s & SEGMENT_MASK].segno = seg;
    }
    if (s & SEGMENT_BUFFER) {
      inv->binfo[s & SEGMENT_MASK].segno = seg;
    }
  }
  return rc;
}

/* chunk */

inline static sen_rc
chunk_new(sen_inv *inv, uint32_t *res, uint32_t size)
{
  int i, j;
  uint32_t n = size / SEN_INV_CHUNK_SIZE;
  int max_chunk = inv->header->initial_n_segments * MAX_CHUNK_RATIO;
  uint32_t base_seg = sen_io_base_seg(inv->chunk);
  if (n * SEN_INV_CHUNK_SIZE < size) { n++; }
  for (i = 0, j = -1; i < max_chunk; i++) {
    if (inv->header->chunks[i]) {
      j = i;
    } else {
      if (i - j == n) {
	if (res) { *res = j + 1; }
	while (j < i) {
	  inv->header->chunks[++j] = 1;
	}
	return sen_success;
      }
      if ((i + base_seg)/ N_CHUNKS_PER_FILE !=
	  (i + base_seg + 1) / N_CHUNKS_PER_FILE) { j = i; }
    }
  }
  sen_log("index full. set bigger value to initial_n_segments. current value = %d",
	  inv->header->initial_n_segments);
  return sen_memory_exhausted;
}

inline static sen_rc
chunk_free(sen_inv *inv, int start, uint32_t size)
{
  uint32_t i, n = size / SEN_INV_CHUNK_SIZE;
  if (n * SEN_INV_CHUNK_SIZE < size) { n++; }
  // sen_log("chunk_free start=%d size=%d(%d)", start, size, n);
  for (i = 0; i < n; i++) {
    inv->header->chunks[start + i] = 0;
  }
  return sen_success;
}

/* buffer */

typedef struct {
  uint32_t tid;
  uint32_t size_in_chunk;
  uint32_t pos_in_chunk;
  uint16_t size_in_buffer;
  uint16_t pos_in_buffer;
} buffer_term;

typedef struct {
  uint16_t step;
  uint16_t jump;
} buffer_rec;

typedef struct {
  uint32_t chunk;
  uint32_t chunk_size;
  uint32_t buffer_free;
  uint16_t nterms;
  uint16_t nterms_void;
} buffer_header;

struct sen_inv_buffer {
  buffer_header header;
  buffer_term terms[(SEN_INV_SEGMENT_SIZE - sizeof(buffer_header))/sizeof(buffer_term)];
};

typedef struct sen_inv_buffer buffer;

inline static sen_rc
buffer_at(sen_inv *inv, uint32_t pos, buffer_term **bt, buffer **b)
{
  sen_io_seginfo *si = &inv->binfo[pos >> W_OF_SEGMENT];
  //  sen_log("buffer_at > si->addr=%d", si->addr);
  if (!si->addr) {
    if (si->segno == -1) { load_all_segments(inv); }
    if (sen_io_seg_bind(inv->seg, si)) { return sen_memory_exhausted; }
  }
  if (b) { *b = si->addr; }
  if (bt) { *bt = (buffer_term *)((byte *)si->addr + (pos & BUFFER_MASK_IN_A_SEGMENT)); }
  //  sen_log("buffer_at <");
  return sen_success;
}

inline static sen_rc
buffer_alloc(sen_inv *inv, int size, uint32_t *pos,
	     buffer_term **bt, buffer_rec **br, buffer **bp, int hint)
{
  buffer *b;
  int seg, offset;
  if (size + sizeof(buffer_header) + sizeof(buffer_term) > SEN_INV_SEGMENT_SIZE) {
    return sen_invalid_argument;
  }
  for (seg = hint % inv->header->initial_n_segments;
       seg < SEN_INV_MAX_SEGMENT;
       seg += inv->header->initial_n_segments) {
    if (inv->binfo[seg].segno == -1) { break; }
    if (!(b = inv->binfo[seg].addr)) { continue; }
    if (b->header.nterms > 8191 && !b->header.nterms_void) { continue; }
    if (b->header.buffer_free < size + sizeof(buffer_term)) { continue; }
    // or flush ?
    if (b->header.chunk_size > SEN_INV_SEGMENT_SIZE * 16) {
      // sen_log("b->header.chunk_size=%d", b->header.chunk_size);
      continue;
    }
    goto exit;
  }
  if (seg >= SEN_INV_MAX_SEGMENT) {
    /* retry */
    for (seg = 0; seg < SEN_INV_MAX_SEGMENT; seg++) {
      if (!(b = inv->binfo[seg].addr)) { continue; }
      if (b->header.chunk == CHUNK_NOT_ASSIGNED // todo: more intelligent
	  && b->header.nterms < 1024
	  && b->header.buffer_free >= size + sizeof(buffer_term)) {
	goto exit;
      }      
    }
  }
  sen_log("inv=%x new seg=%d", inv, seg);
  if (segment_new(inv, SEGMENT_BUFFER, (uint32_t *)&seg) || 
      buffer_at(inv, seg * SEN_INV_SEGMENT_SIZE, NULL, &b)) {
    return sen_memory_exhausted;
  }
  memset(b, 0, SEN_INV_SEGMENT_SIZE);
  b->header.buffer_free = SEN_INV_SEGMENT_SIZE - sizeof(buffer_header);
  b->header.chunk = CHUNK_NOT_ASSIGNED;
  b->header.chunk_size = 0;
exit :
  if (b->header.nterms_void) {
    for (offset = 0; offset < b->header.nterms; offset++) {
      if (!b->terms[offset].tid) { break; }
    }
    if (offset == b->header.nterms) {
      sen_log("caution: inconsistent buffer(%d)", seg);
      b->header.nterms_void = 0;
      b->header.nterms++;
      b->header.buffer_free -= size + sizeof(buffer_term);
    } else {
      b->header.nterms_void--;
      b->header.buffer_free -= size;
    }
  } else {
    offset = b->header.nterms++;
    b->header.buffer_free -= size + sizeof(buffer_term);
  }
  *pos = seg * SEN_INV_SEGMENT_SIZE
    + sizeof(buffer_header) + sizeof(buffer_term) * offset;
  *bt = &b->terms[offset];
  *br = (buffer_rec *)(((byte *)&b->terms[b->header.nterms]) + b->header.buffer_free);
  *bp = b;
  return sen_success;
}

typedef struct {
  uint32_t rid;
  uint32_t sid;
} docid;

#define BUFFER_REC_DEL(r)  ((r)->jump = 1)
#define BUFFER_REC_DELETED(r) ((r)->jump == 1)

#define BUFFER_REC_AT(b,pos) ((buffer_rec *)(b) + (pos))
#define BUFFER_REC_POS(b,rec) ((rec) - (buffer_rec *)(b))

inline static void
buffer_term_dump(buffer *b, buffer_term *bt)
{
  int pos, rid, sid;
  uint8_t *p;
  buffer_rec *r;
  sen_log("b=(%d %d %d %d)", b->header.chunk, b->header.chunk_size, b->header.buffer_free, b->header.nterms);
  sen_log("bt=(%d %d %d %d %d)", bt->tid, bt->size_in_chunk, bt->pos_in_chunk, bt->size_in_buffer, bt->pos_in_buffer);
  for (pos = bt->pos_in_buffer; pos; pos = r->step) {
    r = BUFFER_REC_AT(b, pos);
    p = NEXT_ADDR(r);
    SEN_B_DEC(rid, p);
    SEN_B_DEC(sid, p);
    sen_log("%d=(%d:%d),(%d:%d)", pos, r->jump, r->step, rid, sid);
  }
}

static buffer_term *tmp_bt;

inline static sen_rc
check_jump(buffer *b, buffer_rec *r, int j)
{
  intptr_t i = BUFFER_REC_POS(b, r);
  uint8_t *p;
  buffer_rec *r2;
  docid id, id2;
  if (!j) { return sen_success; }
  p = NEXT_ADDR(r);
  SEN_B_DEC(id.rid, p);
  SEN_B_DEC(id.sid, p);
  if (j == 1) {
    sen_log("deleting! %d(%d:%d)", i, id.rid, id.sid);
    return sen_success;
  }
  r2 = BUFFER_REC_AT(b, j);
  p = NEXT_ADDR(r2);
  SEN_B_DEC(id2.rid, p);
  SEN_B_DEC(id2.sid, p);
  if (r2->step == i) {
    sen_log("cycle! %d(%d:%d)<->%d(%d:%d)", i, id.rid, id.sid, j, id2.rid, id2.sid);
    buffer_term_dump(b, tmp_bt);
    return sen_other_error;
  }
  if (id2.rid < id.rid || (id2.rid == id.rid && id2.sid <= id.sid)) {
    sen_log("invalid jump! %d(%d:%d)(%d:%d)->%d(%d:%d)(%d:%d)", i, r->jump, r->step, id.rid, id.sid, j, r2->jump, r2->step, id2.rid, id2.sid);
    return sen_other_error;
  }
  return sen_success;
}

inline static sen_rc
set_jump_r(buffer *b, buffer_rec *from, int to)
{
  int i, j, max_jump = 100;
  buffer_rec *r, *r2;
  for (r = from, j = to; j > 1 && max_jump--; r = BUFFER_REC_AT(b, r->step)) {
    r2 = BUFFER_REC_AT(b, j);
    if (r == r2) { break; }
    if (BUFFER_REC_DELETED(r2)) { break; }
    if (j == (i = r->jump)) { break; }
    if (j == r->step) { break; }
    if (check_jump(b, r, j)) { return sen_other_error; }
    r->jump = j;
    j = i;
    if (!r->step) { return sen_other_error; }
  }
  return sen_success;
}

#define GET_NUM_BITS(x,n) { \
  n = x; \
  n = (n & 0x55555555) + ((n >> 1) & 0x55555555); \
  n = (n & 0x33333333) + ((n >> 2) & 0x33333333); \
  n = (n & 0x0F0F0F0F) + ((n >> 4) & 0x0F0F0F0F); \
  n = (n & 0x00FF00FF) + ((n >> 8) & 0x00FF00FF); \
  n = (n & 0x0000FFFF) + ((n >>16) & 0x0000FFFF); \
}

inline static sen_rc
buffer_put(buffer *b, buffer_term *bt, buffer_rec *rnew, uint8_t *bs,
	   sen_inv_updspec *u, int size)
{
  uint8_t *p;
  docid id_curr = {0, 0}, id_start = {0, 0}, id_post = {0, 0};
  buffer_rec *r_curr, *r_start = NULL;
  uint16_t last = 0, *lastp = &bt->pos_in_buffer;
  intptr_t pos = BUFFER_REC_POS(b, rnew);
  int vdelta = 0, delta, delta0 = 0, vhops = 0, nhops = 0, reset = 1;

  tmp_bt = bt; // test

  memcpy(NEXT_ADDR(rnew), bs, size - sizeof(buffer_rec));
  //  sen_log("tid=%d u->rid=%d u->sid=%d", bt->tid, u->rid, u->sid);
  for (;;) {
    //    sen_log("*lastp=%d", *lastp);
    if (!*lastp) {
      rnew->step = 0;
      rnew->jump = 0;
      *lastp = (uint16_t)pos;
      if (bt->size_in_buffer++ > 1) {
	buffer_rec *rhead = BUFFER_REC_AT(b, bt->pos_in_buffer);
	rhead->jump = (uint16_t)pos;
	if (!(bt->size_in_buffer & 1)) {
	  int n;
	  buffer_rec *r = BUFFER_REC_AT(b, rhead->step), *r2;
	  GET_NUM_BITS(bt->size_in_buffer, n);
	  while (n-- && (r->jump > 1)) {
	    r2 = BUFFER_REC_AT(b, r->jump);
	    if (BUFFER_REC_DELETED(r2)) { break; }
	    r = r2;
	  }
	  if (r != rnew) { set_jump_r(b, r, last); }
	}
      }
      break;
    }
    r_curr = BUFFER_REC_AT(b, *lastp);
    p = NEXT_ADDR(r_curr);
    SEN_B_DEC(id_curr.rid, p);
    SEN_B_DEC(id_curr.sid, p);
    if (id_curr.rid < id_post.rid ||
	(id_curr.rid == id_post.rid && id_curr.sid < id_post.sid)) {
      sen_log("loop found!!! (%d:%d)->(%d:%d)",
	      id_post.rid, id_post.sid, id_curr.rid, id_curr.sid);
      buffer_term_dump(b, bt);
      break;
    }
    id_post.rid = id_curr.rid;
    id_post.sid = id_curr.sid;
    if (u->rid < id_curr.rid || (u->rid == id_curr.rid && u->sid <= id_curr.sid)) {
      uint16_t step = *lastp, jump = r_curr->jump;
      if (u->rid == id_curr.rid) {
	if (u->sid == 0) {
	  while (id_curr.rid == u->rid) {
	    BUFFER_REC_DEL(r_curr);
	    if (!(step = r_curr->step)) { break; }
	    r_curr = BUFFER_REC_AT(b, step);
	    p = NEXT_ADDR(r_curr);
	    SEN_B_DEC(id_curr.rid, p);
	    SEN_B_DEC(id_curr.sid, p);
	  }
	} else if (u->sid == id_curr.sid) {
	  BUFFER_REC_DEL(r_curr);
	  step = r_curr->step;
	}
      }
      rnew->step = step;
      rnew->jump = check_jump(b, rnew, jump) ? 0 : jump;
      *lastp = (uint16_t)pos;
      break;
    }

    if (reset) {
      r_start = r_curr;
      id_start.rid = id_curr.rid;
      id_start.sid = id_curr.sid;
      if (!(delta0 = u->rid - id_start.rid)) { delta0 = u->sid - id_start.sid; }
      nhops = 0;
      vhops = 1;
      vdelta = delta0 >> 1;
    } else {
      if (!(delta = id_curr.rid - id_start.rid)) { delta = id_curr.sid - id_start.sid; }
      if (vdelta < delta) {
	vdelta += (delta0 >> ++vhops);
	r_start = r_curr;
      }
      if (nhops > vhops) {
	set_jump_r(b, r_start, *lastp);
      } else {
	nhops++;
      }
    }

    last = *lastp;
    lastp = &r_curr->step;
    reset = 0;
    {
      uint16_t posj = r_curr->jump;
      if (posj > 1) {
	buffer_rec *rj = BUFFER_REC_AT(b, posj);
	if (!BUFFER_REC_DELETED(rj)) {
	  docid idj;
	  p = NEXT_ADDR(rj);
	  SEN_B_DEC(idj.rid, p);
	  SEN_B_DEC(idj.sid, p);
	  if (idj.rid < u->rid || (idj.rid == u->rid && idj.sid < u->sid)) {
	    last = posj;
	    lastp = &rj->step;
	  } else {
	    reset = 1;
	  }
	}
      }
    }
  }
  return sen_success;
}

/* array */

inline static uint32_t *
array_at(sen_inv *inv, uint32_t id)
{
  sen_io_seginfo *si;
  if (id > SEN_SYM_MAX_ID) { return NULL; }
  si = &inv->ainfo[id >> W_OF_ARRAY];
  if (!si->addr) {
    if (si->segno == -1) { load_all_segments(inv); }
    if (sen_io_seg_bind(inv->seg, si)) { return NULL; }
  }
  return (uint32_t *)(((byte *)si->addr) + 
		      (id & ARRAY_MASK_IN_A_SEGMENT) * sizeof(uint32_t));
}

inline static uint32_t *
array_get(sen_inv *inv, uint32_t id)
{
  sen_io_seginfo *si;
  uint32_t seg;
  if (id > SEN_SYM_MAX_ID) { return NULL; }
  seg = id >> W_OF_ARRAY;
  si = &inv->ainfo[seg];
  if (!si->addr) {
    if (si->segno == -1) {
      if (segment_get(inv, SEGMENT_ARRAY, seg, si)) { return NULL; }
    }
    if (sen_io_seg_bind(inv->seg, si)) { return NULL; }
  }
  return (uint32_t *)(((byte *)si->addr) +
		      (id & ARRAY_MASK_IN_A_SEGMENT) * sizeof(uint32_t));
}

/* updspec */

sen_inv_updspec *
sen_inv_updspec_open(uint32_t rid, uint32_t sid)
{
  sen_inv_updspec *u;
  if (!(u = SEN_MALLOC(sizeof(sen_inv_updspec)))) { return NULL; }
  u->rid = rid;
  u->sid = sid;
  u->score = 0;
  u->tf = 0;
  u->pos = NULL;
  u->tail = NULL;
  u->vnodes = NULL;
  return u;
}

sen_rc
sen_inv_updspec_add(sen_inv_updspec *u, int pos, int32_t weight)
{
  struct _sen_inv_pos *p;
  if (!(p = SEN_MALLOC(sizeof(struct _sen_inv_pos)))) {
    return sen_memory_exhausted;
  }
  u->score += weight;
  p->pos = pos;
  p->next = NULL;
  if (u->tail) {
    u->tail->next = p;
  } else {
    u->pos = p;
  }
  u->tail = p;
  u->tf++;
  return sen_success;
}

int
sen_inv_updspec_cmp(sen_inv_updspec *a, sen_inv_updspec *b)
{
  struct _sen_inv_pos *pa, *pb;
  if (a->rid != b->rid) { return a->rid - b->rid; }
  if (a->sid != b->sid) { return a->sid - b->sid; }
  if (a->score != b->score) { return a->score - b->score; }
  if (a->tf != b->tf) { return a->tf - b->tf; }
  for (pa = a->pos, pb = b->pos; pa && pb; pa = pa->next, pb = pb->next) {
    if (pa->pos != pb->pos) { return pa->pos - pb->pos; }
  }
  if (pa) { return 1; }
  if (pb) { return -1; }
  return 0;
}

sen_rc
sen_inv_updspec_close(sen_inv_updspec *u)
{
  struct _sen_inv_pos *p = u->pos, *q;
  while (p) {
    q = p->next;
    SEN_FREE(p);
    p = q;
  }
  SEN_FREE(u);
  return sen_success;
}

inline static uint8_t *
encode_rec(sen_inv_updspec *u, unsigned int *size, int deletep)
{
  intptr_t s, r;
  uint8_t *br, *p;
  struct _sen_inv_pos *pp;
  uint32_t lpos, tf = deletep ? 0 : u->tf;
  if (!(br = SEN_MALLOC((u->tf + 4) * 5))) {
    return NULL;
  }
  p = br;
  SEN_B_ENC(u->rid, p);
  SEN_B_ENC(u->sid, p);
  if (!u->score) {
    SEN_B_ENC(tf * 2, p);
  } else {
    SEN_B_ENC(tf * 2 + 1, p);
    SEN_B_ENC(u->score, p);
  }
  for (lpos = 0, pp = u->pos; pp && tf--; lpos = pp->pos, pp = pp->next) {
    SEN_B_ENC(pp->pos - lpos, p);
  }
  s = (p - br) + sizeof(buffer_rec);
  if ((r = s & 0x03)) { s += 4 - r; }
  *size = (unsigned int)s;
  return br;
}

inline static int
sym_deletable(uint32_t tid, sen_set *h)
{
  sen_inv_updspec **u;
  if (!h) { return 1; }
  if (!sen_set_at(h, &tid, (void **) &u)) { return 1; }
  if (!(*u)->tf || !(*u)->sid) { return 1; }
  return 0;
}

inline static void
sym_delete(sen_inv *inv, uint32_t tid, sen_set *h)
{
  uint32_t *a;
  if (!sym_deletable(tid, h)) { return; }
  if (inv->lexicon->flags & SEN_SYM_WITH_SIS) {
    while ((tid = sen_sym_del_with_sis(inv->lexicon, tid))) {
      a = array_at(inv, tid);
      if (a && *a) { break; }
      if (!sym_deletable(tid, h)) { break; }
    }
  } else {
    sen_sym_del(inv->lexicon, _sen_sym_key(inv->lexicon, tid));
  }
}

inline static sen_rc
buffer_flush(sen_inv *inv, uint32_t seg, sen_set *h)
{
  sen_io_seginfo *si;
  buffer *sb, *db;
  sen_io_win sw, dw;
  uint8_t *dc, *sc = NULL;
  uint32_t scn, dcn, ss, ds, max_dest_chunk_size;
  si = &inv->binfo[seg];
  if ((ss = si->segno) == -1) { return sen_invalid_format; }
  if (buffer_at(inv, seg * SEN_INV_SEGMENT_SIZE, NULL, &sb)) {
    return sen_memory_exhausted;
  }
  for (ds = 0; inv->header->segments[ds];) {
    if (++ds == SEN_INV_MAX_SEGMENT) { return sen_memory_exhausted; }
  }
  if (!(db = sen_io_seg_map(inv->seg, ds))) { return sen_memory_exhausted; }
  memset(db, 0, SEN_INV_SEGMENT_SIZE);

  max_dest_chunk_size = sb->header.chunk_size + SEN_INV_SEGMENT_SIZE;
  if (chunk_new(inv, &dcn, max_dest_chunk_size)) {
    return sen_memory_exhausted;
  }
  //  sen_log("db=%p ds=%d sb=%p seg=%d", db, ds, sb, seg);
  if ((scn = sb->header.chunk) != CHUNK_NOT_ASSIGNED) {
    sc = sen_io_win_map(inv->chunk, &sw, scn, 0, sb->header.chunk_size, sen_io_rdwr);
    if (!sc) {
      sen_log("io_win_map(%d, %d) failed!!", scn, sb->header.chunk_size);
      chunk_free(inv, dcn, max_dest_chunk_size);
      return sen_memory_exhausted;
    }
  }
  // dc = sen_io_win_map(inv->chunk, &dw, dcn, 0, max_dest_chunk_size, sen_io_wronly);
  dc = sen_io_win_map(inv->chunk, &dw, dcn, 0, max_dest_chunk_size, sen_io_rdwr);
  if (!dc) { 
    sen_log("io_win_map(%d, %d) failed!!", dcn, max_dest_chunk_size);
    chunk_free(inv, dcn, max_dest_chunk_size);
    if (scn != CHUNK_NOT_ASSIGNED) { sen_io_win_unmap(&sw); }
    return sen_memory_exhausted;
  }
  {
    uint8_t *bp = NULL, *cp = NULL, *cpe = NULL, *dp = dc;
    uint16_t nextb;
    buffer_rec *br;
    buffer_term *bt;
    int n = sb->header.nterms;
    int nterms_void = 0;
    memcpy(db->terms, sb->terms, n * sizeof(buffer_term));
    // sen_log(" scn=%d, dcn=%d, nterms=%d", sb->header.chunk, dcn, n);
    for (bt = db->terms; n; n--, bt++) {
      docid cid = {0, 0}, lid = {0, 0}, bid = {0, 0};
      uint32_t ndf = 0, tf2, ltf2 = 0, gap;
      if (!bt->tid) {
	nterms_void++;
	continue;
      }
      if (sc) { 
	cp = sc + bt->pos_in_chunk;
	cpe = cp + bt->size_in_chunk;
      }
      nextb = bt->pos_in_buffer;
      bt->pos_in_chunk = (uint32_t)(dp - dc);
      bt->size_in_buffer = 0;
      bt->pos_in_buffer = 0;

      // sen_log("db=%p n=%d, bt=%p tid=%d bdf=%d", db, n, bt, bt->tid, bdf);

#define GETNEXTC() { \
  if (cp < cpe && cid.rid) { \
    SEN_B_DEC(tf2, cp); \
    if (tf2 & 1) { SEN_B_SKIP(cp); } \
    tf2 >>= 1; \
    while (cp < cpe && tf2--) { SEN_B_SKIP(cp); } \
  } \
  if (cp < cpe) { \
    SEN_B_DEC(gap, cp); \
    cid.rid += gap; \
    if (gap) { cid.sid = 0; } \
    SEN_B_DEC(gap, cp); \
    cid.sid += gap; \
  } else { \
    cid.rid = 0; \
  } \
}
#define PUTNEXTC() { \
  if (cid.rid) { \
    /* sen_log("srid=%d", srid); */ \
    SEN_B_DEC(tf2, cp); \
    if (tf2) { \
      if (lid.rid > cid.rid || (lid.rid == cid.rid && lid.sid >= cid.sid)) { \
	sen_log("brokenc!! (%d:%d) -> (%d:%d)", lid.rid, lid.sid, bid.rid, bid.sid); \
      } \
      ndf++; \
      gap = cid.rid - lid.rid; \
      SEN_B_ENC(gap, dp); \
      if (gap) { SEN_B_ENC(cid.sid, dp); } else { SEN_B_ENC(cid.sid - lid.sid, dp); } \
      SEN_B_ENC(tf2, dp); \
      if (tf2 & 1) { SEN_B_COPY(dp, cp); } \
      ltf2 = tf2; \
      tf2 >>= 1; \
      while (tf2--) { SEN_B_COPY(dp, cp); } \
      lid.rid = cid.rid; \
      lid.sid = cid.sid; \
    } else { sen_log("caution: invalid chunk(%d,%d)", bt->tid, cid.rid); } \
  } \
  if (cp < cpe) { \
    SEN_B_DEC(gap, cp); \
    cid.rid += gap; \
    if (gap) { cid.sid = 0; } \
    SEN_B_DEC(gap, cp); \
    cid.sid += gap; \
  } else { \
    cid.rid = 0; \
  } \
  /* sen_log("gap=%d srid=%d", gap, srid); */ \
}
#define GETNEXTB() { \
  if (nextb) { \
    uint32_t lrid = bid.rid, lsid = bid.sid; \
    br = BUFFER_REC_AT(sb, nextb); \
    bp = NEXT_ADDR(br); \
    SEN_B_DEC(bid.rid, bp); \
    SEN_B_DEC(bid.sid, bp); \
    if (lrid > bid.rid || (lrid == bid.rid && lsid >= bid.sid)) { \
      sen_log("brokeng!! (%d:%d) -> (%d:%d)", lrid, lsid, bid.rid, bid.sid); \
    } \
    nextb = br->step; \
  } else { \
    bid.rid = 0; \
  } \
}
#define PUTNEXTB() { \
  if (bid.rid && bid.sid) { \
    SEN_B_DEC(tf2, bp); \
    if (tf2) { \
      /* sen_log("brid=%d", bid.rid); */ \
      if (lid.rid > bid.rid || (lid.rid == bid.rid && lid.sid >= bid.sid)) { \
	sen_log("brokenb!! (%d:%d) -> (%d:%d)", lid.rid, lid.sid, bid.rid, bid.sid); \
      } \
      ndf++; \
      gap = bid.rid - lid.rid; \
      SEN_B_ENC(gap, dp); \
      if (gap) { SEN_B_ENC(bid.sid, dp); } else { SEN_B_ENC(bid.sid - lid.sid, dp); } \
      SEN_B_ENC(tf2, dp); \
      if (tf2 & 1) { SEN_B_COPY(dp, bp); } \
      ltf2 = tf2; \
      tf2 >>= 1; \
      while (tf2--) { SEN_B_COPY(dp, bp); } \
      lid.rid = bid.rid; \
      lid.sid = bid.sid; \
    } \
  } \
  GETNEXTB(); \
}

      GETNEXTC();
      GETNEXTB();
      for (;;) {
	if (bid.rid) {
	  if (cid.rid) {
	    if (cid.rid < bid.rid) {
	      PUTNEXTC();
	    } else {
	      if (bid.rid < cid.rid) {
		PUTNEXTB();
	      } else {
		if (bid.sid) {
		  if (cid.sid < bid.sid) {
		    PUTNEXTC();
		  } else {
		    if (bid.sid == cid.sid) { GETNEXTC(); }
		    PUTNEXTB();
		  }
		} else {
		  GETNEXTC();
		}
	      }
	    }
	  } else {
	    PUTNEXTB();
	  }
	} else {
	  if (cid.rid) {
	    PUTNEXTC();
	  } else {
	    break;
	  }
	}
      }
      // sen_log("break: dp=%p cp=%p", dp, cp);

      bt->size_in_chunk = (uint32_t)((dp - dc) - bt->pos_in_chunk);

      if (!ndf) {
	uint32_t *a;
	if ((a = array_at(inv, bt->tid))) {
	  sen_sym_pocket_set(inv->lexicon, bt->tid, 0);
	  *a = 0;
	  sym_delete(inv, bt->tid, h);
	}
	bt->tid = 0;
	bt->pos_in_chunk = 0;
	bt->size_in_chunk = 0;
	nterms_void++;
      } else if (ndf == 1 && lid.rid < 0x100000 && lid.sid < 0x800 && ltf2 == 2) {
	uint32_t rid_, sid_, tf_, pos_;
	uint8_t *dp_ = dc + bt->pos_in_chunk;
	SEN_B_DEC(rid_, dp_);
	if (rid_ < 0x100000) {
	  SEN_B_DEC(sid_, dp_);
	  if (sid_ < 0x800) {
	    SEN_B_DEC(tf_, dp_);
	    if (tf_ == 2) {
	      SEN_B_DEC(pos_, dp_);
	      if (pos_ < 0x4000) {
		uint32_t *a;
		if ((a = array_at(inv, bt->tid))) {
		  sen_sym_pocket_set(inv->lexicon, bt->tid, pos_);
		  *a = (rid_ << 12) + (sid_ << 1) + 1;
		}
		dp = dc + bt->pos_in_chunk;
		bt->tid = 0;
		bt->pos_in_chunk = 0;
		bt->size_in_chunk = 0;
		nterms_void++;
	      }
	    }
	  }
	}
      }
      // sen_log("db=%p df=%d size=%d", db, ndf, (dp - dc) - bt->pos_in_chunk);
    }
    db->header.chunk_size = (uint32_t)(dp - dc);
    db->header.nterms_void = nterms_void;
  }
  db->header.chunk = dcn;
  db->header.buffer_free = SEN_INV_SEGMENT_SIZE
    - sizeof(buffer_header) - sb->header.nterms * sizeof(buffer_term);
  db->header.nterms = sb->header.nterms;

  sen_io_seg_unbind(inv->seg, si);
  si->segno = ds;
  sen_io_seg_bind(inv->seg, si);
  {
    uint32_t mc, ec;
    mc = max_dest_chunk_size / SEN_INV_CHUNK_SIZE;
    if (mc * SEN_INV_CHUNK_SIZE < max_dest_chunk_size) { mc++; }
    ec = db->header.chunk_size / SEN_INV_CHUNK_SIZE;
    if (ec * SEN_INV_CHUNK_SIZE < db->header.chunk_size) { ec++; }
    // sen_log(" ss=%d ds=%d inv->binfo[%d]=%p max_size=%d(%d) chunk_size=%d(%d)", ss, ds, seg, db, max_dest_chunk_size, mc, db->header.chunk_size, ec);
    while (ec < mc) {
      // sen_log("chunk[%d]=0(%d)", ec, mc);
      inv->header->chunks[db->header.chunk + ec++] = 0;
    }
  }
  inv->header->segments[ss] = 0;
  inv->header->segments[ds] = SEGMENT_BUFFER | seg;

  if (scn != CHUNK_NOT_ASSIGNED) {
    sen_io_win_unmap(&sw);
    chunk_free(inv, scn, sb->header.chunk_size);
  }
  sen_io_win_unmap(&dw);
  return sen_success;
}

/* inv */

sen_inv *
sen_inv_create(const char *path,  sen_sym *lexicon, 
	       int flags, uint32_t initial_n_segments)
{
  int i, max_chunk;
  sen_io *seg, *chunk;
  sen_inv *inv;
  char path2[PATH_MAX];
  struct sen_inv_header *header;
  if (strlen(path) + 6 >= PATH_MAX) { return NULL; }
  strcpy(path2, path);
  strcat(path2, ".c");
  if (!initial_n_segments) { initial_n_segments = SEN_INV_INITIAL_N_SEGMENTS; }
  if (initial_n_segments > SEN_INV_MAX_SEGMENT) {
    initial_n_segments = SEN_INV_MAX_SEGMENT;
  }
  max_chunk = initial_n_segments * MAX_CHUNK_RATIO;
  seg = sen_io_create(path, sizeof(struct sen_inv_header) + max_chunk,
		      SEN_INV_SEGMENT_SIZE, SEN_INV_MAX_SEGMENT,
		      sen_io_auto, SEN_INV_MAX_SEGMENT);
  if (!seg) { return NULL; }
  chunk = sen_io_create(path2, 0, SEN_INV_CHUNK_SIZE,
			max_chunk, sen_io_auto, max_chunk);
  if (!chunk) {
    sen_io_close(seg);
    return NULL;
  }
  header = sen_io_header(seg);
  memcpy(header->idstr, SEN_INV_IDSTR, 16);
  for (i = 0; i < SEN_INV_MAX_SEGMENT; i++) { header->segments[i] = 0; }
  header->initial_n_segments = initial_n_segments;
  if (!(inv = SEN_MALLOC(sizeof(sen_inv)))) {
    sen_io_close(seg);
    sen_io_close(chunk);
    return NULL;
  }
  inv->seg = seg;
  inv->chunk = chunk;
  inv->header = header;
  inv->lexicon = lexicon;
  for (i = 0; i < SEN_INV_MAX_SEGMENT; i++) {
    inv->ainfo[i].segno = -1;
    inv->ainfo[i].addr = NULL;
    inv->binfo[i].segno = -1;
    inv->binfo[i].addr = NULL;
  }
  return inv;
}

sen_inv *
sen_inv_open(const char *path, sen_sym *lexicon)
{
  int i;
  sen_io *seg, *chunk;
  sen_inv *inv;
  char path2[PATH_MAX];
  struct sen_inv_header *header;
  if (strlen(path) + 6 >= PATH_MAX) { return NULL; }
  strcpy(path2, path);
  strcat(path2, ".c");
  seg = sen_io_open(path, sen_io_auto, SEN_INV_MAX_SEGMENT);
  if (!seg) { return NULL; }
  chunk = sen_io_open(path2, sen_io_auto, SEN_INV_MAX_SEGMENT);
  if (!chunk) {
    sen_io_close(seg);
    return NULL;
  }
  header = sen_io_header(seg);
  /*
  if (memcmp(header->idstr, SEN_INV_IDSTR, 16)) {
    sen_io_close(seg);
    sen_io_close(chunk);
    return NULL;
  }
  */
  if (!(inv = SEN_MALLOC(sizeof(sen_inv)))) {
    sen_io_close(seg);
    sen_io_close(chunk);
    return NULL;
  }
  inv->seg = seg;
  inv->chunk = chunk;
  inv->header = header;
  inv->lexicon = lexicon;
  for (i = 0; i < SEN_INV_MAX_SEGMENT; i++) {
    inv->ainfo[i].segno = -1;
    inv->ainfo[i].addr = NULL;
    inv->binfo[i].segno = -1;
    inv->binfo[i].addr = NULL;
  }
  load_all_segments(inv);
  return inv;
}

sen_rc
sen_inv_close(sen_inv *inv)
{
  sen_rc rc;
  if ((rc = sen_io_close(inv->seg))) { return rc; }
  if ((rc = sen_io_close(inv->chunk))) { return rc; }
  SEN_FREE(inv);
  return rc;
}

sen_rc
sen_inv_info(sen_inv *inv, off_t *seg_size, off_t *chunk_size)
{
  sen_rc rc;

  if (seg_size) {
    if ((rc = sen_io_size(inv->seg, seg_size))) {
      return rc;
    }
  }

  if (chunk_size) {
    if ((rc = sen_io_size(inv->chunk, chunk_size))) {
      return rc;
    }
  }

  return sen_success;
}

sen_rc
sen_inv_update(sen_inv *inv, uint32_t key, sen_inv_updspec *u, sen_set *h, int hint)
{
  sen_rc r = sen_success;
  buffer *b;
  uint8_t *bs;
  buffer_rec *br = NULL;
  buffer_term *bt;
  uint32_t pos = 0, size, *a;
  // sen_log("key=%d tf=%d pos0=%d rid=%d", key, u->tf, u->pos->pos, u->rid);
  if (!u->tf || !u->sid) { return sen_inv_delete(inv, key, u, h); }
  if (!(a = array_get(inv, key))) { return sen_memory_exhausted; }
  if (!(bs = encode_rec(u, &size, 0))) { return sen_memory_exhausted; }
  for (;;) {
    if (*a) {
      if (!(*a & 1)) {
	if ((r = buffer_at(inv, *a, &bt, &b))) { goto exit; }
	if (b->header.buffer_free < size) {
	  int bfb = b->header.buffer_free;
	  uint32_t _a = *a;
	  SEN_LOG(sen_log_debug, "flushing *a=%d seg=%d(%p) free=%d",
		  *a, *a >> W_OF_SEGMENT, b, b->header.buffer_free);
	  buffer_flush(inv, *a >> W_OF_SEGMENT, h);
	  if (*a != _a) {
	    sen_log("sen_inv_update: *a changed %d->%d", *a, _a);
	    continue;
	  }
	  if ((r = buffer_at(inv, *a, &bt, &b))) {
	    sen_log("buffer not found *a=%d", *a);
	    goto exit; 
	  }
	  SEN_LOG(sen_log_debug, "flushed  *a=%d seg=%d(%p) free=%d->%d nterms=%d v=%d",
		  *a, *a >> W_OF_SEGMENT, b, bfb, b->header.buffer_free,
		  b->header.nterms, b->header.nterms_void);
	  if (b->header.buffer_free < size) {
	    sen_log("buffer(%d) is full (%d < %d) in sen_inv_update",
		    *a, b->header.buffer_free, size);
	    /* todo: must be splitted */
	    r = sen_memory_exhausted;
	    goto exit;
	  }
	}
	b->header.buffer_free -= size;
	br = (buffer_rec *)(((byte *)&b->terms[b->header.nterms])
			    + b->header.buffer_free);
      } else {
	sen_inv_updspec u2;
	uint32_t size2 = 0, v = *a;
	struct _sen_inv_pos pos2;
	pos2.pos = sen_sym_pocket_get(inv->lexicon, key);
	pos2.next = NULL;
	u2.pos = &pos2;
	u2.rid = BIT31_12(v);
	u2.sid = BIT11_01(v);
	u2.tf = 1;
	u2.score = 0;
	if (u2.rid != u->rid || u2.sid != u->sid) {
	  uint8_t *bs2 = encode_rec(&u2, &size2, 0);
	  if ((r = buffer_alloc(inv, size + size2, &pos, &bt, &br, &b, hint))) {
	    SEN_FREE(bs2);
	    goto exit;
	  }
	  bt->tid = key;
	  bt->size_in_chunk = 0;
	  bt->pos_in_chunk = 0;
	  bt->size_in_buffer = 0;
	  bt->pos_in_buffer = 0;
	  buffer_put(b, bt, br, bs2, &u2, size2);
	  br = (buffer_rec *)(((byte *)br) + size2);
	  SEN_FREE(bs2);
	}
      }
    }
    break;
  }
  if (!br) {
    if (u->rid < 0x100000 && u->sid < 0x800 &&
	u->tf == 1 && u->score == 0 && u->pos->pos < 0x4000) {
      sen_sym_pocket_set(inv->lexicon, key, u->pos->pos);
      *a = (u->rid << 12) + (u->sid << 1) + 1;
      goto exit;
    } else {
      if ((r = buffer_alloc(inv, size, &pos, &bt, &br, &b, hint))) { goto exit; }
      bt->tid = key;
      bt->size_in_chunk = 0;
      bt->pos_in_chunk = 0;
      bt->size_in_buffer = 0;
      bt->pos_in_buffer = 0;
    }
  }
  buffer_put(b, bt, br, bs, u, size);
  if (!*a || (*a & 1)) {
    *a = pos;
    sen_sym_pocket_set(inv->lexicon, key, 0);
  }
exit :
  SEN_FREE(bs);
  return r;
}

sen_rc
sen_inv_delete(sen_inv *inv, uint32_t key, sen_inv_updspec *u, sen_set *h)
{
  sen_rc r = sen_success;
  buffer *b;
  uint8_t *bs = NULL;
  buffer_rec *br;
  buffer_term *bt;
  uint32_t size, *a = array_at(inv, key);
  for (;;) {
    if (!a || !*a) { goto exit; }
    if (*a & 1) {
      uint32_t rid = BIT31_12(*a);
      uint32_t sid = BIT11_01(*a);
      if (u->rid == rid && (!u->sid || u->sid == sid)) {
	*a = 0;
	sym_delete(inv, key, h);
      }
      goto exit;
    } 
    if ((r = buffer_at(inv, *a, &bt, &b))) { goto exit; }
    if (!(bs = encode_rec(u, &size, 1))) { 
      r = sen_memory_exhausted;
      goto exit;
    }
    //  sen_log("b->header.buffer_free=%d size=%d", b->header.buffer_free, size);
    if (b->header.buffer_free < size) {
      uint32_t _a = *a;
      sen_log("flushing! b=%p free=%d, seg(%d)", b, b->header.buffer_free, *a >> W_OF_SEGMENT);
      buffer_flush(inv, *a >> W_OF_SEGMENT, h);
      if (*a != _a) {
	sen_log("sen_inv_delete: *a changed %d->%d)", *a, _a);
	continue;
      }
      if ((r = buffer_at(inv, *a, &bt, &b))) { goto exit; }
      sen_log("flushed!  b=%p free=%d, seg(%d)", b, b->header.buffer_free, *a >> W_OF_SEGMENT);
      if (b->header.buffer_free < size) {
	/* todo: must be splitted ? */
	sen_log("buffer(%d) is full (%d < %d) in sen_inv_delete",
		*a, b->header.buffer_free, size);
	r = sen_memory_exhausted;
	goto exit;
      }
    }

    b->header.buffer_free -= size;
    br = (buffer_rec *)(((byte *)&b->terms[b->header.nterms]) + b->header.buffer_free);
    buffer_put(b, bt, br, bs, u, size);
    break;
  }

exit :
  if (bs) { SEN_FREE(bs); }
  return r;
}

uint32_t
sen_inv_initial_n_segments(sen_inv *inv)
{
  return inv->header->initial_n_segments;
}

#define CHUNK_USED    1
#define BUFFER_USED   2
#define SOLE_DOC_USED 4
#define SOLE_POS_USED 8

sen_inv_cursor *
sen_inv_cursor_open(sen_inv *inv, uint32_t key)
{
  sen_inv_cursor *c  = NULL;
  uint32_t *a = array_at(inv, key);
  if (!a || !*a) { return c; }
  if (!(c = SEN_MALLOC(sizeof(sen_inv_cursor)))) { return c; }
  memset(c, 0, sizeof(sen_inv_cursor));
  if (*a & 1) {
    uint32_t v = *a;
    c->stat = 0;
    c->pb.rid = BIT31_12(v);
    c->pb.sid = BIT11_01(v);
    c->pb.tf = 1;
    c->pb.score = 0;
    c->pb.pos = sen_sym_pocket_get(inv->lexicon, key);
  } else {
    uint32_t chunk;
    buffer_term *bt;
    c->pb.rid = 0; c->pb.sid = 0; /* for check */
    if (buffer_at(inv, *a, &bt, &c->buf)) {
      SEN_FREE(c);
      return NULL;
    }
    if (bt->size_in_chunk && (chunk = c->buf->header.chunk) != CHUNK_NOT_ASSIGNED) {
      c->cp = sen_io_win_map(inv->chunk, &c->iw,
			     chunk, bt->pos_in_chunk, bt->size_in_chunk, sen_io_rdonly);
      if (!c->cp) {
	SEN_FREE(c);
	return NULL;
      }
      c->cpe = c->cp + bt->size_in_chunk;
      c->pc.rid = 0;
      c->pc.sid = 0;
    }
    c->nextb = bt->pos_in_buffer;
    c->stat = CHUNK_USED|BUFFER_USED;
  }
  return c;
}

#ifdef USE_AIO
sen_inv_cursor *
sen_inv_cursor_openv1(sen_inv *inv, uint32_t key)
{
  sen_inv_cursor *c  = NULL;
  uint32_t *a = array_at(inv, key);
  if (!a || !*a) { return c; }
  if (!(c = SEN_MALLOC(sizeof(sen_inv_cursor)))) { return c; }
  memset(c, 0, sizeof(sen_inv_cursor));
  if (*a & 1) {
    uint32_t v = *a;
    c->stat = 0;
    c->pb.rid = BIT31_12(v);
    c->pb.sid = BIT11_01(v);
    c->pb.tf = 1;
    c->pb.score = 0;
    c->pb.pos = sen_sym_pocket_get(inv->lexicon, key);
  } else {
    buffer_term *bt;
    c->pb.rid = 0; c->pb.sid = 0;
    if (buffer_at(inv, *a, &bt, &c->buf)) {
      SEN_FREE(c);
      return NULL;
    }
    c->iw.io = inv->chunk;
    c->iw.mode = sen_io_rdonly;
    c->iw.segment = c->buf->header.chunk;
    c->iw.offset = bt->pos_in_chunk;
    c->iw.size = bt->size_in_chunk;
    c->nextb = bt->pos_in_buffer;
    c->stat = CHUNK_USED|BUFFER_USED;
  }
  return c;
}

sen_rc
sen_inv_cursor_openv2(sen_inv_cursor **cursors, int ncursors)
{
  sen_rc rc = sen_success;
  int i, j = 0;
  sen_inv_cursor *c;
  sen_io_win **iws = SEN_MALLOC(sizeof(sen_io_win *) * ncursors);
  if (!iws) { return sen_memory_exhausted; }
  for (i = 0; i < ncursors; i++) {
    c = cursors[i];
    if (c->stat && c->iw.size && c->iw.segment != CHUNK_NOT_ASSIGNED) {
      iws[j++] = &c->iw;
    }
  }
  if (j) { rc = sen_io_win_mapv(iws, j); }
  for (i = 0; i < ncursors; i++) {
    c = cursors[i];
    if (c->iw.addr) {
      c->cp = c->iw.addr + c->iw.diff;
      c->cpe = c->cp + c->iw.size;
      c->pc.rid = 0;
      c->pc.sid = 0;
    }
  }
  SEN_FREE(iws);
  return rc;
}
#endif /* USE_AIO */

sen_rc
sen_inv_cursor_next(sen_inv_cursor *c)
{
  if (c->buf) {
    for (;;) {
      if (c->stat & CHUNK_USED) {
	while (c->cp < c->cpe && c->pc.rest--) { SEN_B_SKIP(c->cp); }
	if (c->cp < c->cpe) {
	  uint32_t gap;
	  SEN_B_DEC(gap, c->cp);
	  c->pc.rid += gap;
	  if (gap) { c->pc.sid = 0; }
	  SEN_B_DEC(gap, c->cp);
	  c->pc.sid += gap;
	  SEN_B_DEC(c->pc.tf, c->cp);
	  if (c->pc.tf & 1) { SEN_B_DEC(c->pc.score, c->cp); } else { c->pc.score = 0; }
	  c->pc.rest = c->pc.tf >>= 1;
	  c->pc.pos = 0;
	  // sen_log("rid=%d sid=%d tf=%d", c->pc.rid, c->pc.sid, c->pc.tf);
	} else {
	  c->pc.rid = 0;
	}
      }
      if (c->stat & BUFFER_USED) {
	if (c->nextb) {
	  uint32_t lrid = c->pb.rid, lsid = c->pb.sid; /* for check */
	  buffer_rec *br = BUFFER_REC_AT(c->buf, c->nextb);
	  c->bp = NEXT_ADDR(br);
	  SEN_B_DEC(c->pb.rid, c->bp);
	  SEN_B_DEC(c->pb.sid, c->bp);
	  if (lrid > c->pb.rid || (lrid == c->pb.rid && lsid >= c->pb.sid)) {
	    sen_log("brokend!! (%d:%d) -> (%d:%d)", lrid, lsid, c->pb.rid, c->pb.sid);
	  }
	  c->nextb = br->step;
	  SEN_B_DEC(c->pb.tf, c->bp);
	  if (c->pb.tf & 1) { SEN_B_DEC(c->pb.score, c->bp); } else { c->pb.score = 0; }
	  c->pb.rest = c->pb.tf >>= 1;
	  c->pb.pos = 0;
	} else {
	  c->pb.rid = 0;
	}
      }
      if (c->pb.rid) {
	if (c->pc.rid) {
	  if (c->pc.rid < c->pb.rid) {
	    c->stat = CHUNK_USED;
            if (c->pc.tf && c->pc.sid) { c->post = &c->pc; break; }
	  } else {
	    if (c->pb.rid < c->pc.rid) {
	      c->stat = BUFFER_USED;
	      if (c->pb.tf && c->pb.sid) { c->post = &c->pb; break; }
	    } else {
	      if (c->pb.sid) {
		if (c->pc.sid < c->pb.sid) {
		  c->stat = CHUNK_USED;
		  if (c->pc.tf && c->pc.sid) { c->post = &c->pc; break; }
		} else {
		  c->stat = BUFFER_USED;
		  if (c->pb.sid == c->pc.sid) { c->stat |= CHUNK_USED; }
		  if (c->pb.tf) { c->post = &c->pb; break; }
		}
	      } else {
		c->stat = CHUNK_USED;
	      }
	    }
	  }
	} else {
	  c->stat = BUFFER_USED;
	  if (c->pb.tf && c->pb.sid) { c->post = &c->pb; break; }
	}
      } else {
	if (c->pc.rid) {
	  c->stat = CHUNK_USED;
	  if (c->pc.tf && c->pc.sid) { c->post = &c->pc; break; }
	} else {
	  c->post = NULL;
	  return sen_other_error;
	}
      }
    }
  } else {
    if (c->stat & SOLE_DOC_USED) {
      c->post = NULL;
      return sen_other_error;
    } else {
      c->post = &c->pb;
      c->stat |= SOLE_DOC_USED;
    }
  }
  return sen_success;
}

sen_rc
sen_inv_cursor_next_pos(sen_inv_cursor *c)
{
  uint32_t gap;
  sen_rc rc = sen_success;
  if (c->buf) {
    if (c->post == &c->pc) {
      if (c->pc.rest) {
	c->pc.rest--;
	SEN_B_DEC(gap, c->cp);
	c->pc.pos += gap;
      } else {
	rc = sen_other_error;
      }
    } else if (c->post == &c->pb) {
      if (c->pb.rest) {
	c->pb.rest--;
	SEN_B_DEC(gap, c->bp);
	c->pb.pos += gap;
      } else {
	rc = sen_other_error;
      }
    } else {
      rc = sen_other_error;
    }
  } else {
    if (c->stat & SOLE_POS_USED) {
      rc = sen_other_error;
    } else {
      c->stat |= SOLE_POS_USED;
    }
  }
  return rc;
}

sen_rc
sen_inv_cursor_close(sen_inv_cursor *c)
{
  if (!c) { return sen_invalid_argument; }
  if (c->cp) { sen_io_win_unmap(&c->iw); }
  SEN_FREE(c);
  return sen_success;
}

uint32_t
sen_inv_estimate_size(sen_inv *inv, uint32_t key)
{
  uint32_t *a = array_at(inv, key);
  if (a && *a) {
    if (*a & 1) {
      return 1;
    } else {
      buffer *buf;
      buffer_term *bt;
      if (buffer_at(inv, *a, &bt, &buf)) { return 0; }
      // buffer_term_dump(buf, bt);
      return (bt->size_in_chunk >> 2) + bt->size_in_buffer + 2;
    }
  } else {
    return 0;
  }
}

int
sen_inv_entry_info(sen_inv *inv, unsigned key, unsigned *a, unsigned *pocket,
		   unsigned *chunk, unsigned *chunk_size, unsigned *buffer_free,
		   unsigned *nterms, unsigned *nterms_void, unsigned *tid,
		   unsigned *size_in_chunk, unsigned *pos_in_chunk,
		   unsigned *size_in_buffer, unsigned *pos_in_buffer)
{
  buffer *b;
  buffer_term *bt;
  uint32_t *ap = array_at(inv, key);
  *pocket = sen_sym_pocket_get(inv->lexicon, key);
  if (!ap) { return 0; }
  *a = *ap;
  if (!*a) { return 1; }
  if (*a & 1) { return 2; }
  if (buffer_at(inv, *a, &bt, &b)) { return 3; }
  *chunk = b->header.chunk;
  *chunk_size = b->header.chunk_size;
  *buffer_free = b->header.buffer_free;
  *nterms = b->header.nterms;
  *tid = bt->tid;
  *size_in_chunk = bt->size_in_chunk;
  *pos_in_chunk = bt->pos_in_chunk;
  *size_in_buffer = bt->size_in_buffer;
  *pos_in_buffer = bt->pos_in_buffer;
  return 4;
}
