package org.seasar.framework.container.impl;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.List;

import org.seasar.framework.aop.Aspect;
import org.seasar.framework.aop.proxy.AopProxy;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.PropertyDesc;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.container.AspectDef;
import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.container.ArgDef;
import org.seasar.framework.container.CyclicReferenceRuntimeException;
import org.seasar.framework.container.MethodDef;
import org.seasar.framework.container.PropertyDef;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.exception.EmptyRuntimeException;
import org.seasar.framework.reflect.ClassUtil;
import org.seasar.framework.reflect.ConstructorUtil;
import org.seasar.framework.sel.Expression;
import org.seasar.framework.sel.parser.SelParser;

/**
 * @author higa
 *
 */
public final class ComponentDefImpl implements ComponentDef {

	private Class componentClass_;
	private String componentName_;
	private S2Container container_;
	private Object component_;
	private List argDefs_ = new ArrayList();
	private List propertyDefs_ = new ArrayList();
	private List methodDefs_ = new ArrayList();
	private List aspectDefs_ = new ArrayList();
	private Expression expression_;
	private boolean instantiating_ = false;

	public ComponentDefImpl(Class componentClass)
		throws EmptyRuntimeException {

		this(componentClass, null);
	}

	public ComponentDefImpl(Class componentClass, String componentName)
		throws EmptyRuntimeException {

		if (componentClass == null) {
			throw new EmptyRuntimeException("componentClass");
		}
		componentClass_ = componentClass;
		componentName_ = componentName;
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#getComponent()
	 */
	public Object getComponent() {
		assembleIfFirst();
		return component_;
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#getComponentClass()
	 */
	public final Class getComponentClass() {
		return componentClass_;
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#getComponentName()
	 */
	public final String getComponentName() {
		return componentName_;
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#getContainer()
	 */
	public final S2Container getContainer() {
		return container_;
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#setContainer(org.seasar.framework.container.S2Container)
	 */
	public final void setContainer(S2Container container) {
		container_ = container;
		for (int i = 0; i < argDefs_.size(); ++i) {
			getArgDef(i).setContainer(container);
		}
		for (int i = 0; i < propertyDefs_.size(); ++i) {
			getPropertyDef(i).setContainer(container);
		}
		for (int i = 0; i < methodDefs_.size(); ++i) {
			getMethodDef(i).setContainer(container);
		}
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#addArgDef(org.seasar.framework.container.ArgDef)
	 */
	public void addArgDef(ArgDef argDef) {
		if (container_ != null) {
			argDef.setContainer(container_);
		}
		argDefs_.add(argDef);
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#addPropertyDef(org.seasar.framework.container.PropertyDef)
	 */
	public void addPropertyDef(PropertyDef propertyDef) {
		if (container_ != null) {
			propertyDef.setContainer(container_);
		}
		propertyDefs_.add(propertyDef);
	}
	
	/**
	 * @see org.seasar.framework.container.ComponentDef#addMethodDef(org.seasar.framework.container.MethodDef)
	 */
	public void addMethodDef(MethodDef methodDef) {
		if (container_ != null) {
			methodDef.setContainer(container_);
		}
		methodDefs_.add(methodDef);
	}
	
	/**
	 * @see org.seasar.framework.container.ComponentDef#addAspectDef(org.seasar.framework.container.AspectDef)
	 */
	public void addAspectDef(AspectDef aspectDef) {
		if (container_ != null) {
			aspectDef.setContainer(container_);
		}
		aspectDefs_.add(aspectDef);
	}

	/**
	 * @see org.seasar.framework.container.ComponentDef#setExpression(java.lang.String)
	 */
	public void setExpression(String str) {
		expression_ = new SelParser(str).parseExpression();
	}

	private ArgDef getArgDef(int index) {
		return (ArgDef) argDefs_.get(index);
	}

	private PropertyDef getPropertyDef(int index) {
		return (PropertyDef) propertyDefs_.get(index);
	}
	
	private MethodDef getMethodDef(int index) {
		return (MethodDef) methodDefs_.get(index);
	}
	
	private AspectDef getAspectDef(int index) {
		return (AspectDef) aspectDefs_.get(index);
	}

	private void assembleIfFirst() {
		if (component_ == null) {
			synchronized (this) {
				if (component_ == null) {
					assemble();
				}
			}
		}
	}

	private void assemble() {
		if (instantiating_) {
			throw new CyclicReferenceRuntimeException(componentClass_);
		}
		instantiating_ = true;
		if (argDefs_.size() > 0) {
			assembleByArgDefs();
		} else if (propertyDefs_.size() > 0) {
			assembleByPropertyDefs();
		} else if (expression_ != null) {
			assembleByExpression();
		} else {
			Constructor constructor = getSuitableConstructor();
			if (constructor != null) {
				assembleAutoByConstructor(constructor);
			} else {
				assembleAutoByProperties();
			}
		}
		assembleByMethodDefs();
	}

	private void assembleByArgDefs() {
		Object[] args = new Object[argDefs_.size()];
		for (int i = 0; i < args.length; ++i) {
			args[i] = getArgDef(i).getValue();
		}
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(componentClass_);
		if (aspectDefs_.size() > 0) {
			AopProxy aopProxy = new AopProxy(componentClass_, getAspects());
			component_ = aopProxy.create(
				beanDesc.getSuitableConstructor(args).getParameterTypes(), args);
		} else {
			component_ = beanDesc.newInstance(args);
		}
		instantiating_ = false;
	}

	private void assembleByPropertyDefs() {
		createComponentForDefaultConstructor();
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(componentClass_);
		for (int i = 0; i < propertyDefs_.size(); ++i) {
			PropertyDef propDef = getPropertyDef(i);
			Object value = propDef.getValue();
			PropertyDesc propDesc =
				beanDesc.getPropertyDesc(propDef.getPropertyName());
			propDesc.setValue(component_, value);
		}
	}
	
	private void assembleByMethodDefs() {
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(componentClass_);
		for (int i = 0; i < methodDefs_.size(); ++i) {
			MethodDef methodDef = getMethodDef(i);
			beanDesc.invoke(component_, methodDef.getMethodName(),
				methodDef.getArgs());
		}
	}
	
	private void createComponentForDefaultConstructor() {
		if (aspectDefs_.size() > 0) {
			AopProxy aopProxy = new AopProxy(componentClass_, getAspects());
			component_ = aopProxy.create();
		} else {
			component_ = ClassUtil.newInstance(componentClass_);
		}
		instantiating_ = false;
	}
	
	private Aspect[] getAspects() {
		Aspect[] aspects = new Aspect[aspectDefs_.size()];
		for (int i = 0; i < aspectDefs_.size(); ++i) {
			AspectDef aspectDef = getAspectDef(i);
			aspects[i] = aspectDef.getAspect();
		}
		return aspects;
	}

	private void assembleByExpression() {
		component_ = expression_.evaluateValue(container_.getSelContext());
		instantiating_ = false;
	}

	private void assembleAutoByConstructor(Constructor constructor) {
		Class[] argTypes = constructor.getParameterTypes();
		Object[] args = new Object[argTypes.length];
		for (int i = 0; i < argTypes.length; ++i) {
			args[i] = container_.getComponent(argTypes[i]);
		}
		if (aspectDefs_.size() > 0) {
			AopProxy aopProxy = new AopProxy(componentClass_, getAspects());
			component_ = aopProxy.create(argTypes, args);
		} else {
			component_ = ConstructorUtil.newInstance(constructor, args);
		}
		instantiating_ = false;
	}

	private Constructor getSuitableConstructor() {
		int argSize = 0;
		Constructor constructor = null;
		Constructor[] constructors = componentClass_.getConstructors();
		for (int i = 0; i < constructors.length; ++i) {
			int tempArgSize = constructors[i].getParameterTypes().length;
			if (tempArgSize == 0) {
				return null;
			}
			if (tempArgSize > argSize) {
				constructor = constructors[i];
				argSize = tempArgSize;
			}
		}
		return constructor;
	}

	private void assembleAutoByProperties() {
		createComponentForDefaultConstructor();
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(componentClass_);
		for (int i = 0; i < beanDesc.getPropertyDescSize(); ++i) {
			PropertyDesc propDesc = beanDesc.getPropertyDesc(i);
			if (propDesc.getWriteMethod() != null) {
				Object value =
					container_.getComponent(propDesc.getPropertyType());
				propDesc.setValue(component_, value);
			}
		}
	}
}
