package org.seasar.transaction;

import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.xa.Xid;

import org.seasar.log.Logger;
import org.seasar.message.MessageFormatter;
import org.seasar.util.SeasarException;

public final class TransactionManagerImpl implements TransactionManager {

    private static TransactionManagerImpl _instance = new TransactionManagerImpl();
    private ThreadLocal _threadAttachTx = new ThreadLocal();

    private TransactionManagerImpl() { }

    public static TransactionManagerImpl getInstance() {
        return _instance;
    }
    
    public void begin() throws NotSupportedException, SystemException {
        TransactionImpl tx = (TransactionImpl) _threadAttachTx.get();
        if (tx != null && tx.getStatus() != Status.STATUS_NO_TRANSACTION) {
            throw new NotSupportedException(
                    MessageFormatter.getMessage("ESSR0316", null));
        }
        tx = attachTransaction(tx);
        tx.begin();
    }
    
    public void begin2() throws SeasarException {
    	try {
    		begin();
    	} catch (NotSupportedException ex) {
    		throw SeasarException.convertSeasarException(ex);
    	} catch (SystemException ex) {
    		throw SeasarException.convertSeasarException(ex);
    	}
	}

    public void commit() throws RollbackException, HeuristicMixedException,
            HeuristicRollbackException, SecurityException, IllegalStateException,
            SystemException {

        TransactionImpl tx = getCurrent();
        if (tx == null) {
            throw new IllegalStateException(
                    MessageFormatter.getMessage("ESSR0311", null));
        }
        try {
            tx.commit();
        } finally {
            detachTransaction(tx);
        }
    }
    
    public void commit2() throws SeasarException {
    	try {
    		commit();
    	} catch (Exception ex) {
    		throw SeasarException.convertSeasarException(ex);
    	}
    }

    public Transaction suspend() {
        TransactionImpl tx = getCurrent();
        if (tx == null) {
            throw new IllegalStateException(
                    MessageFormatter.getMessage("ESSR0311", null));
        }
        suspendInternal(tx);
        return tx;
    }

    public void resume(final Transaction resumeTx)
             throws InvalidTransactionException, IllegalStateException, SystemException {

        if (resumeTx == null) {
            throw new InvalidTransactionException(
                    MessageFormatter.getMessage("ESSR0007", new Object[]{"Transaction"}));
        }
        TransactionImpl tx = getCurrent();
        if (tx != null) {
            throw new IllegalStateException(
                    MessageFormatter.getMessage("ESSR0317", null));
        }
        setCurrent((TransactionImpl) resumeTx);
        ((TransactionImpl) resumeTx).resume();
    }
    
    public void resume2(final Transaction resumeTx) throws SeasarException {
    	try {
    		resume(resumeTx);
    	} catch (Exception ex) {
    		throw SeasarException.convertSeasarException(ex);
    	}
    }

    public void rollback() throws IllegalStateException, SecurityException,
            SystemException {

        TransactionImpl tx = getCurrent();
        if (tx == null) {
            throw new IllegalStateException(
                    MessageFormatter.getMessage("ESSR0311", null));
        }
        try {
            tx.rollback();
        } finally {
            detachTransaction(tx);
        }
    }
    
    public void rollback2() {
    	try {
    		if (getStatus() != Status.STATUS_NO_TRANSACTION) {
  		  		rollback();
    		}
    	} catch (Exception ex) {
    		Logger.getLogger(getClass()).log("ESSR0017", new Object[]{ex}, ex);
    	}
    }

    public void setRollbackOnly() throws IllegalStateException, SystemException {
        Transaction tx = getTransaction();
        if (tx == null) {
            throw new IllegalStateException(
                    MessageFormatter.getMessage("ESSR0311", null));
        }
        tx.setRollbackOnly();
    }

    public void setTransactionTimeout(final int timeout) throws SystemException {
    }

    public int getStatus() {
        TransactionImpl tx = getCurrent();
        if (tx != null) {
            return tx.getStatus();
        } else {
            return Status.STATUS_NO_TRANSACTION;
        }
    }

    public Transaction getTransaction() {
        return getCurrent();
    }
    
    public boolean preRequiredProcess() throws SeasarException {
		boolean began = false;			
		int preStatus = getStatus();
		if (preStatus == Status.STATUS_NO_TRANSACTION) {
    		begin2();
    		began = true;
		}
        return began;
	}
	
	public void postNormalRequiredProcess(final boolean began) throws SeasarException {
        if (began) {
        	commit2();
        }
    }
    
    public void postExceptionRequiredProcess(final boolean began) {
        if (began) {
        	try {
        		if (getStatus() != Status.STATUS_NO_TRANSACTION) {
      		  		rollback();
        		}
        	} catch (Exception ex) {
        		Logger.getLogger(getClass()).log("ESSR0017", new Object[]{ex}, ex);
        	}
        }
    }
    
    public TransactionImpl preRequiresNewProcess() throws SeasarException {
    	TransactionImpl preTx = getCurrent();
    	if (preTx != null) {
    		suspendInternal(preTx);
    	}
    	begin2();
        return preTx;
	}
	
	public void postNormalRequiresNewProcess() throws SeasarException {
    	commit2();
    }
    
    public void postExceptionRequiresNewProcess() {
        rollback2();
    }
    
    public void finalRequiresNewProcess(final TransactionImpl preTx) throws SeasarException {
        if (preTx != null) {
        	resume2(preTx);
        }
    }

    private TransactionImpl getCurrent() {
        TransactionImpl tx = (TransactionImpl) _threadAttachTx.get();
        if (tx != null && tx.getStatus() == Status.STATUS_NO_TRANSACTION) {
        	return null;
        }
        return tx;
    }

    private void setCurrent(final TransactionImpl current) {
        _threadAttachTx.set(current);
    }

    private TransactionImpl attachTransaction(TransactionImpl tx) {
        Xid xid = new XidImpl();
        if (tx != null) {
            tx.init(xid);
        } else {
            tx = new TransactionImpl(xid);
        }
        _threadAttachTx.set(tx);
        return tx;
    }

    private void detachTransaction(TransactionImpl tx) {
        if (tx != null) {
            tx.destroy();
        }
    }
    
    private void suspendInternal(TransactionImpl tx) {
		tx.suspend();
		setCurrent(null);
	}
}