package org.seasar.nazuna;

import java.lang.reflect.Method;

import org.seasar.util.Assertion;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarException;

public final class RuletFacade {
	
	private Rulet _rulet;
	private Method _doExecuteMethod;
	private Method _doValidateMethod;
	private NazunaTransAttribute _transAttribute;
	
	private RuletFacade(Rulet rulet, Method doExecuteMethod, Method doValidateMethod,
			NazunaTransAttribute transAttribute) {
				
		_rulet = rulet;
		_doExecuteMethod = doExecuteMethod;
		_doValidateMethod = doValidateMethod;
		_transAttribute = transAttribute;
	}
	
	public static final RuletFacade create(Class ruletClass) {
		Assertion.assertNotNull("ruletClass", ruletClass);
		
		Rulet rulet = (Rulet) Reflector.newInstance(ruletClass);
		RuletConfig ruletConfig = new RuletConfig();
		rulet.setRuletConfig(ruletConfig);
		Method doExecuteMethod = findMethod(ruletClass, "doExecute");
		Assertion.assertFound("doExecute", doExecuteMethod);
		Method doValidateMethod = findMethod(ruletClass, "doValidate");
		NazunaTransAttribute transAttribute = getTransAttribute(rulet);
		return new RuletFacade(rulet, doExecuteMethod, doValidateMethod, transAttribute);
	}
	
	public final Rulet getRulet() {
		return _rulet;
	}
	
	public final Method getDoExecuteMethod() {
		return _doExecuteMethod;
	}
	
	public final Method getDoValidateMethod() {
		return _doValidateMethod;
	}
	
	public final NazunaTransAttribute getTransAttribute() {
		return _transAttribute;
	}
	
	public final void setTransAttribute(NazunaTransAttribute transAttribute) {
		Assertion.assertNotNull("transAttribute", transAttribute);
		
		_transAttribute = transAttribute;
	}

	public final void init() throws SeasarException {
		_rulet.init();
	}
	
	public final void destroy() throws SeasarException {
		_rulet.destroy();
	}
	
	public final Object execute(Object[] args) throws SeasarException {
		if (_doValidateMethod != null) {
			Reflector.invoke(_doValidateMethod, _rulet, args);
		}
		return _transAttribute.invoke(_doExecuteMethod, _rulet, args);
	}
	
	private static final Method findMethod(Class ruletClass, String methodName) {
		Method[] methods = ruletClass.getDeclaredMethods();
		for (int i = 0; i < methods.length; ++i) {
			if (methods[i].getName().equals(methodName)) {
				return methods[i];
			}
		}
		Class superClass = ruletClass.getSuperclass();
		if (superClass != null && !superClass.equals(Rulet.class)) {
			return findMethod(superClass, methodName);
		}
		return null;
	}
	
	private static final NazunaTransAttribute getTransAttribute(Rulet rulet) {
		if (rulet instanceof RequiredTx) {
			return NazunaTransAttribute.REQUIRED;
		} else if (rulet instanceof RequiresNewTx) {
			return NazunaTransAttribute.REQUIRES_NEW;
		} else if (rulet instanceof MandatoryTx) {
			return NazunaTransAttribute.MANDATORY;
		} else {
			return NazunaTransAttribute.SUPPORTS;
		}
	}
}

