/**************************************************************************
 FolderMenu - easy access to project folders from menu.
 
 Copyright (C) 2013-2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.osdn.users.yutang.omegat.plugin.foldermenu.filepreview;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import jp.osdn.users.yutang.omegat.plugin.foldermenu.L10n;
import org.omegat.util.LFileCopy;
import org.omegat.util.Log;
import static org.omegat.util.Platform.OsType.WIN32;
import static org.omegat.util.Platform.OsType.WIN64;
import static org.omegat.util.Platform.getOsType;
import org.omegat.util.StaticUtils;

/**
 * Word 文書をプレビュー用に開きます。
 * 
 * 
 * @author Yu Tang
 */
public class WordPreview implements IPreview {
    
    private static final String WSF_NAME = "WordPreview.wsf";
    private static boolean _isMSWordAvailable;
    private static File _wsf;

    private final File originalFile;
    private long originalFileLastModified; // will update at each every time compiling target docs
    private final String windowTitle;
    private final File temporaryFile;      // Primary temp file
    private final File temporaryFile2;     // Secondary temp file

    public WordPreview(final File originalFile) throws IOException {
        this.originalFile = originalFile;
        this.originalFileLastModified = originalFile.lastModified();
        this.temporaryFile = getTempFile(originalFile);
        this.windowTitle = StaticUtils.format(
                L10n.Entry.WORD_WINDOW_CAPTION.toString(),
                originalFile.getName());
        this.temporaryFile2 = getTempFile2(this.temporaryFile);
    }

    static {
        // _isMSWordAvailable
        switch (getOsType()) {
            case WIN64:
            case WIN32:
                new Thread() {
                    @Override
                    public void run() {
                        _isMSWordAvailable = getMSWordAvailable();
                        if (_isMSWordAvailable) {
                            Log.log("FolderMenu: WordPreview function is available.");
                        } else {
                            Log.log("FolderMenu: WordPreview function is not available.");
                        }
                    }
                }.start();
                break;
            default: // Mac, Linux and others
                _isMSWordAvailable = false;
                break;
        }

        // _wsf
        File tempDir = new File(System.getProperty("java.io.tmpdir"));
        _wsf = new File(tempDir, WSF_NAME);
    }

    public static boolean isAvailable(File file) {
        return isAvailable() &&
                file.isFile() && 
                file.getName().toLowerCase().endsWith(".docx");
    }
    
    public static boolean isAvailable() {
        return _isMSWordAvailable;
    }

    public static void init() {
        // force executing static initializer
    }

    private static boolean getMSWordAvailable() {
        final int RET_OK = 0;
        try {
            Command command = new Command();
            String s;
            if (RET_OK == command.execDOS("assoc", ".docx")) {
                s = command.getStdout();
                // s's data example）
                // -----------------------------------------------------
                //.docx=Word.Document.12
                //<-(\r\n)
                // -----------------------------------------------------
                // 末尾に空行が入るので注意。
                // 他のパターンとして、AOO/LO 環境で以下のようなケースもあり。
                //.docx=OpenOffice.Docx
                //.docx=LibreOffice.WriterDocument.1
                if (s.toLowerCase().startsWith(".docx=word.document.")) {
                    String classString = s.substring(".docx=".length()).replaceAll("\\r\\n", "");
                    if (RET_OK == command.exec("reg", "query", "HKCR\\" + classString + "\\shell\\open\\command", "/ve")) {
                        s = command.getStdout();
                        // s's data example）
                        // -----------------------------------------------------
                        //<-(\r\n)
                        //HKEY_CLASSES_ROOT\Word.document.12\shell\open\command
                        //(既定) REG_SZ "C:\PROGRA~2\MICROS~4\OFFICE11\WINWORD.EXE" /n /dde
                        //<-(\r\n)
                        // -----------------------------------------------------
                        // 前後に空行が入るので注意。
                        return s.toUpperCase().indexOf("\\WINWORD.EXE") > -1;
                    }
                }
            }
        } catch (Exception ex) {
            Log.log(ex);
        }
        return false;
    }

    private static File getWSF() throws IOException {
        if (! _wsf.exists()) {
            InputStream in = WordPreview.class.getResourceAsStream(WSF_NAME);
            try {
                LFileCopy.copy(in, _wsf);
            } finally {
                in.close();
            }
            _wsf.deleteOnExit();
        }
        return _wsf;
    }

    @Override
    public String[] getTempFiles() {
        // ここでは、残留する可能性のある一時ファイルをすべて申告します。
        String[] paths = new String[3];
        try {
            paths[0] = this.temporaryFile.getCanonicalPath();
            paths[1] = this.temporaryFile2.getCanonicalPath();

            // MS Word が作成する（プレビュー用ファイルの）一時ファイルも、強制
            // 終了時などには残留する可能性があるため、ここで申告しておきます。
            final String PREFIX = "~$";
            String parent = this.temporaryFile.getParent();
            String name = this.temporaryFile.getName();
            paths[2] = (new File(parent, PREFIX + name.substring(2))).getCanonicalPath();
        } catch (IOException ex) {
            Log.log(ex);
        }
        return paths;
    }

    @Override
    public void activate() {
        try {
            final Command command = new Command();
            final String job = "activate";
            final int ret = command.execWSF(job, this.windowTitle);
        } catch (IOException ex) {
            Log.log(ex);
        } catch (InterruptedException ex) {
            Log.log(ex);
        }
    }

    @Override
    public void open() {

        // 以下の処理は少し時間がかかるため、別スレッドに処理を委譲します。
        new Thread() {
            @Override
            public void run() {
                try {
                    // 起動前にファイルをコピーする。
                    // OmegaT は訳文ファイルの作成時に、既存の訳文ファイルを上書きする。
                    // そのため、オリジナルのファイルをそのまま開くとファイルがロックされ、
                    // 次回のコンパイル時に上書きできずに失敗する。それを避けるために、
                    // プレビュー専用の一時ファイルをコピーして、そちらを開く。
                    // コピー先は、temp フォルダーではなく、オリジナルと同じフォルダー内に
                    // コピーする。文書に相対パスで画像リンクなどが張られている場合のリンク
                    // 切れを防ぐため。
                    // そのままコピーすると FolderMenu プラグインのメニュー上で一時ファイル
                    // が見えてしまうため、hidden 属性を付けておく。
                    LFileCopy.copy(originalFile, temporaryFile);

                    // make temp file hidden on Windows
                    addHiddenFileAttribute(temporaryFile);

                    // Desktop.getDesktop().open(temp);
                    // 上記のようにして一時ファイルを開くと、場合によっては Word
                    // の MRU に一時ファイルを開いた履歴が大量に残ってしまう。
                    // これを回避するため、WSH を経由して COM オートメーションで
                    // 処理する。

                    // open the document
                    Command command = new Command();
                    String document = temporaryFile.getCanonicalPath();
                    String document2 = temporaryFile2.getCanonicalPath();
                    String job = "open";
                    int ret = command.execWSF(job, document, document2, windowTitle);

                    if (! command.stderr.isEmpty()) {
                        Log.log("Word error(" + ret + "): " + command.stderr);
                    }
                    onWordApplicationQuit(ret);
                 } catch (IOException ex) {
                    Log.log(ex);
                } catch (InterruptedException ex) {
                    Log.log(ex);
                }
            }
        }.start();
     }

    private File getTempFile(final File originalFile) throws IOException {
        String prefix = "_WordPreview";
        String name = originalFile.getName();
        String suffix = name.substring(name.lastIndexOf("."));
        File parentFolder = originalFile.getParentFile();
        File tempFile = File.createTempFile(prefix, suffix, parentFolder);
        tempFile.deleteOnExit();
        return tempFile;
    }

    // foo.ext => foo(2).ext
    private File getTempFile2(final File primaryTempFile) throws IOException {
        String name = primaryTempFile.getName();
        int lastDotPos = name.lastIndexOf(".");
        String baseName = name.substring(0, lastDotPos);
        String extension = name.substring(lastDotPos);
        String fileName = baseName + "(2)" + extension;
        File parentFolder = primaryTempFile.getParentFile();
        File tempFile2 = new File(parentFolder, fileName);
        tempFile2.deleteOnExit();
        return tempFile2;
    }

    private void addHiddenFileAttribute(File file) {
        try {
            new ProcessBuilder("attrib","+H", file.getCanonicalPath()).start();
        } catch (IOException ex) {
            Log.log(ex);
        }
    }

    private void onWordApplicationQuit(final int returnCode) {
        try {
            // remove this from Previews collection
            FilePreview.delete(originalFile);

            // try to delete temporary file
            temporaryFile.delete();
            
            // try to delete WSF file
            if (FilePreview.size(WordPreview.class) == 0) {
                _wsf.delete();
            }
            
        } catch (IOException ex) {
            Log.log(ex);
        }
    }

    @Override
    public void close() {
        try {
            // close the document
            final Command command = new Command();
            final String job = "close";
            final String document = temporaryFile.getCanonicalPath();
            command.execWSF(job, document);
        } catch (IOException ex) {
            Log.log(ex);
        } catch (InterruptedException ex) {
            Log.log(ex);
        }
    }

    @Override
    public void reload() {
        if (! isOriginalFileUpdated()) {
            return; 
        }

        try {
            File temp = getTempFile(originalFile);

            // copy the file to avoid locking the file unnecessarily
            LFileCopy.copy(originalFile, temp);

            // rename to secondary temp file (and pass it to WSF)
            temp.renameTo(temporaryFile2);

            // make temp file hidden on Windows
            addHiddenFileAttribute(temporaryFile2);

            // update lastModified value
            this.originalFileLastModified = originalFile.lastModified();
        } catch (IOException ex) {
            Log.log(ex);
        }
    }
    
    private boolean isOriginalFileUpdated() {
        return this.originalFileLastModified != this.originalFile.lastModified();
    }

    // バッファあふれ非対応のため、少量のテキスト（だいたい 500文字ていど）が
    // 予想される場合のみ利用してください。
    // また同期実行です。プロセスの終了を待機してから制御を返します。
    protected static class Command {

        private int exitCode = 0;
        private String stdout = "";
        private String stderr = "";

        public int getExitCode() {
            return exitCode;
        }

        public String getStdout() {
            return stdout;
        }

        public String getStderr() {
            return stderr;
        }

        public int exec(String... command) 
                throws IOException, InterruptedException {
            return startProcessAndWait(Arrays.asList(command));
        }

        public int execDOS(String... command) 
                throws IOException, InterruptedException {
            List<String> commands = new ArrayList<String>(command.length + 2);
            commands.add("cmd.exe");
            commands.add("/c");
            commands.addAll(Arrays.asList(command));
            
            return startProcessAndWait(commands);
        }

        public int execWSF(String job, String... command) 
                throws IOException, InterruptedException {
            String script = getWSF().getCanonicalPath();
            List<String> commands = new ArrayList<String>(command.length + 4);
            commands.add("cscript.exe");
            commands.add("//nologo");
            commands.add("//Job:" + job);
            commands.add(script);
            commands.addAll(Arrays.asList(command));

            return startProcessAndWait(commands);
        }

        private int startProcessAndWait(List<String> command) 
                throws IOException, InterruptedException {
            ProcessBuilder pb = new ProcessBuilder(command);
            Process process = pb.start();
            exitCode = process.waitFor();  // 0: succeed
            stdout = getString(process.getInputStream());
            stderr = getString(process.getErrorStream());
            return exitCode;
        }

        private String getString(InputStream is) throws IOException {
            byte[] b = new byte[1024];
            int size = is.read(b);
            if (size > 0) {
                return new String(b, 0, size);
            } else {
                return "";
            }
        }

    }
}
