﻿using System;
using System.Text;
using System.Runtime.InteropServices;

namespace DiskRefresh
{
    /// <summary>
    /// Setup APIを利用して、デバイスの存在判定・有効化・無効化を行う
    /// </summary>
    /// <remarks>
    /// 参考URL:
    /// https://stackoverflow.com/questions/4097000/how-do-i-disable-a-system-device-programatically
    /// </remarks>
    class SetupDi
    {
        /// <summary>
        /// 検出されたデバイスのフレンドリネーム
        /// </summary>
        private static string foundFriendlyName = null;
        
        /// <summary>
        /// ディスクドライブを表すGUID
        /// </summary>
        /// <remarks>devguid.hのGUID_DEVCLASS_DISKDRIVEを参照</remarks>
        private static Guid devClassDiskDriveGuid = new Guid(
            0x4d36e967,
            0xe325, 0x11ce,
            0xbf, 0xc1, 0x08, 0x00, 0x2b, 0xe1, 0x03, 0x18);

        /// <summary>
        /// SetupDiGetClassDevsのフラグ(現在存在するデバイスだけを返す)
        /// </summary>
        /// <remarks>SetupAPI.hのDIGCF_PRESENTを参照</remarks>
        private const uint DIGCF_PRESENT = 2;

        /// <summary>
        /// Win32エラーコード(これ以上有効なデータはない)
        /// </summary>
        /// <remarks>WinError.hのERROR_NO_MORE_ITEMSを参照</remarks>
        private const uint ERROR_NO_MORE_ITEMS = 259;

        /// <summary>
        /// SetupDiGetDeviceRegistryPropertyのフラグ(デバイスのフレンドリネーム)
        /// </summary>
        /// <remarks>SetupAPI.hのSPDRP_FRIENDLYNAMEを参照</remarks>
        private const uint SPDRP_FRIENDLYNAME = 0x000c;

        /// <summary>
        /// クラスインストーラファンクションの機能コード(プロパティの変更)
        /// </summary>
        /// <remarks>SetupAPI.hのDIF_PROPERTYCHANGEを参照</remarks>
        private const uint DIF_PROPERTYCHANGE = 0x0012;

        /// <summary>
        /// デバイス状態の変更コード(デバイスの有効化)
        /// </summary>
        /// <remarks>SetupAPI.hのDICS_ENABLEを参照</remarks>
        private const uint DICS_ENABLE = 0x0001;

        /// <summary>
        /// デバイス状態の変更コード(デバイスの無効化)
        /// </summary>
        /// <remarks>SetupAPI.hのDICS_DISABLEを参照</remarks>
        private const uint DICS_DISABLE = 0x0002;

        /// <summary>
        /// プロパティ変更のスコープを示すコード(すべてのハードウェアプロファイルに適用)
        /// </summary>
        /// <remarks>SetupAPI.hのDICS_FLAG_GLOBALを参照</remarks>
        private const uint DICS_FLAG_GLOBAL = 0x0001;

        /// <summary>
        /// SP_DEVINFO_DATA(デバイス情報エレメントの構造体)
        /// </summary>
        /// <remarks>SetupApi.hのSP_DEVINFO_DATAを参照</remarks>
        [StructLayout(LayoutKind.Sequential)]
        private struct SP_DEVINFO_DATA
        {
            public UInt32 cbSize;
            public Guid classGuid;
            public UInt32 devInst;
            public IntPtr reserved;
        }

        /// <summary>
        /// SP_CLASSINSTALL_HEADER(クラスインストールパラメータの先頭に配置されるヘッダの構造体)
        /// </summary>
        /// <remarks>SetupApi.hのSP_CLASSINSTALL_HEADERを参照</remarks>
        [StructLayout(LayoutKind.Sequential)]
        private struct SP_CLASSINSTALL_HEADER
        {
            public UInt32 cbSize;
            public UInt32 InstallFunction;
        }

        /// <summary>
        /// SP_PROPCHANGE_PARAMS(DIF_PROPERTYCHANGEに対応したクラスインストールパラメータ構造体)
        /// </summary>
        /// <remars>SetupApi.hのSP_PROPCHANGE_PARAMSを参照</remars>
        [StructLayout(LayoutKind.Sequential)]
        private struct SP_PROPCHANGE_PARAMS
        {
            public SP_CLASSINSTALL_HEADER ClassInstallHeader;
            public UInt32 StateChange;
            public UInt32 Scope;
            public UInt32 HwProfile;
        }
        /// <summary>
        /// SetupDiGetClassDevs(デバイス情報セットを取得する)
        /// </summary>
        /// <param name="ClassGuid">セットアップクラスまたはインタフェースクラスのクラスGUIDへのポインタ</param>
        /// <param name="Enumerator">返されたデバイスをフィルタ処理するためのPnP列挙子</param>
        /// <param name="parent">トップレベルウィンドウのハンドル</param>
        /// <param name="flags">制御オプション(DIGCF_ALLCLASSES,DIGCF_DEVICEINTERFACE,DIGCF_PRESENT,DIGCF_PROFILE)</param>
        /// <returns>デバイス情報セットのハンドルまたはNULL</returns>
        [DllImport("setupapi.dll", SetLastError = true)]
        static extern IntPtr SetupDiGetClassDevsW(
            [In] ref Guid ClassGuid,
            [MarshalAs(UnmanagedType.LPWStr)]string Enumerator,
            IntPtr parent,
            UInt32 flags);

        /// <summary>
        /// SetupDiDestroyDeviceInfoList(デバイス情報セットを破棄する)
        /// </summary>
        /// <param name="handle">デバイス情報セットのハンドル</param>
        /// <returns>ステータス(false:失敗)</returns>
        [DllImport("setupapi.dll", SetLastError = true)]
        static extern bool SetupDiDestroyDeviceInfoList(IntPtr handle);

        /// <summary>
        /// SetupDiEnumDeviceInfo(デバイス情報エレメントを取得する)
        /// </summary>
        /// <param name="deviceInfoSet">デバイス情報セットのハンドル</param>
        /// <param name="memberIndex">取得したいデバイス情報エレメントのインデックス(0オリジン)</param>
        /// <param name="deviceInfoData">SP_DEVINFO_DATA構造体へのポインタ(メンバcbSizeを初期化すること)</param>
        /// <returns>ステータス(false:失敗)</returns>
        [DllImport("setupapi.dll", SetLastError = true)]
        static extern bool SetupDiEnumDeviceInfo(IntPtr deviceInfoSet,
            UInt32 memberIndex,
            [Out] out SP_DEVINFO_DATA deviceInfoData);

        /// <summary>
        /// SetupDiGetDeviceRegistryProperty(PnPデバイスのプロパティを取得する)
        /// </summary>
        /// <param name="DeviceInfoSet">デバイス情報セットのハンドル</param>
        /// <param name="DeviceInfoData">デバイス情報エレメント</param>
        /// <param name="Property">取得するプロパティ</param>
        /// <param name="PropertyRegDataType">プロパティのレジストリデータ種別の格納先</param>
        /// <param name="PropertyBuffer">プロパティを格納するバッファへのポインタ</param>
        /// <param name="PropertyBufferSize">プロパティを格納するバッファのサイズ</param>
        /// <param name="RequiredSize">必要なバッファサイズの格納先</param>
        /// <returns>ステータス(false:失敗)</returns>
        [DllImport("setupapi.dll", SetLastError = true)]
        static extern bool SetupDiGetDeviceRegistryPropertyW(
            IntPtr DeviceInfoSet,
            [In] ref SP_DEVINFO_DATA DeviceInfoData,
            UInt32 Property,
            [Out] out UInt32 PropertyRegDataType,
            IntPtr PropertyBuffer,
            UInt32 PropertyBufferSize,
            [In, Out] ref UInt32 RequiredSize);

        /// <summary>
        /// SetupDiSetClasssInstallParams(デバイス情報エレメントにクラスインストールパラメータを設定する)
        /// </summary>
        /// <param name="deviceInfoSet">デバイス情報セットのハンドル</param>
        /// <param name="deviceInfoData">デバイス情報エレメント</param>
        /// <param name="classInstallParams">クラスインストールパラメータへのポインタ</param>
        /// <param name="ClassInstallParamsSize">クラスインストールパラメータのサイズ</param>
        /// <returns>ステータス(false:失敗)</returns>
        [DllImport("setupapi.dll", SetLastError = true)]
        static extern bool SetupDiSetClassInstallParams(
            IntPtr deviceInfoSet,
            [In] ref SP_DEVINFO_DATA deviceInfoData,
            [In] ref SP_PROPCHANGE_PARAMS classInstallParams,
            UInt32 ClassInstallParamsSize);

        /// <summary>
        /// SetupDiChangeState(DIF_PROPERTYCHANGEリクエストのデフォルトハンドラ)
        /// </summary>
        /// <param name="deviceInfoSet">デバイス情報セットのハンドル</param>
        /// <param name="deviceInfoData">デバイス情報エレメント</param>
        /// <returns>ステータス(false:失敗)</returns>
        [DllImport("setupapi.dll", SetLastError = true)]
        static extern bool SetupDiChangeState(
            IntPtr deviceInfoSet,
            [In] ref SP_DEVINFO_DATA deviceInfoData);

        /// <summary>
        /// actionDevice()に渡す機能コード
        /// </summary>
        /// <remarks>Noneの場合、デバイスの存在有無チェックのみを行う</remarks>
        private enum actionType
        {
            // 何もしない(存在チェックのみ行う)
            None,

            // デバイスを有効化する
            Enable,

            // デバイスを無効化する
            Disable
        }

        /// <summary>
        /// デバイスが存在するか調べる
        /// </summary>
        /// <param name="friendlyName">デバイスのフレンドリネーム(部分一致)</param>
        /// <returns>ステータス(true:存在する、false:存在しない)</returns>
        public static bool isDevicePresent(string friendlyName)
        {
            return actionDevice(friendlyName, actionType.None);
        }

        /// <summary>
        /// 存在するデバイスのフレンドリネームを取得する
        /// </summary>
        /// <returns>デバイスのフレンドリネーム(見つからなかった場合はnull)</returns>
        /// <remarks>事前にisDevicePresentを呼び出しておくこと</remarks>
        public static string getPresentFriendlyName()
        {
            return foundFriendlyName;
        }

        /// <summary>
        /// デバイスを有効化する
        /// </summary>
        /// <param name="friendlyName">デバイスのフレンドリネーム(部分一致)</param>
        /// <returns>ステータス(true:有効化成功、false:有効化失敗)</returns>
        /// <remarks>Administrator権限(UACによる権限昇格)が必要</remarks>
        public static bool enableDevice(string friendlyName)
        {
            return actionDevice(friendlyName, actionType.Enable);
        }

        /// <summary>
        /// デバイスを無効化する
        /// </summary>
        /// <param name="friendlyName">デバイスのフレンドリネーム(部分一致)</param>
        /// <returns>ステータス(true:無効化成功、false:無効化失敗)</returns>
        /// <remarks>Administrator権限(UACによる権限昇格)が必要</remarks>
        public static bool disableDevice(string friendlyName)
        {
            return actionDevice(friendlyName, actionType.Disable);
        }

        /// <summary>
        /// デバイスの存在確認・有効化・無効化を行う 
        /// </summary>
        /// <param name="friendlyName">デバイスのフレンドリネーム(部分一致)</param>
        /// <param name="type">デバイスに対するアクション(actionType列挙型)</param>
        /// <returns>ステータス(true:成功、false:失敗)</returns>
        private static bool actionDevice(string friendlyName, actionType type)
        {
            // 戻り値をtrueで初期化
            bool result = true;

            // 検出フレンドリネームをnullで初期化
            foundFriendlyName = null;

            // デバイス情報セットをNULLで初期化
            IntPtr info = IntPtr.Zero;

            // SP_DEVINFO_DATA構造体を作成し、メンバcbSizeを初期化
            SP_DEVINFO_DATA devdata = new SP_DEVINFO_DATA();
            devdata.cbSize = (UInt32)Marshal.SizeOf(devdata);

            // デバイス情報セットを取得(現在存在するディスクドライブを指定)
            info = SetupDiGetClassDevsW(
                ref devClassDiskDriveGuid,
                null,
                IntPtr.Zero,
                DIGCF_PRESENT);

            // NULLであれば失敗
            if (info == IntPtr.Zero)
            {
                result = false;
            }

            // 成功している場合
            if (result)
            {
                // インデックスを増やしながらループ
                for (uint index = 0; ; index++)
                {
                    // デバイス情報エレメントを取得
                    result = SetupDiEnumDeviceInfo(
                        info,
                        index,
                        out devdata);

                    // 失敗すれば
                    if (result == false)
                    {
                        // GetLastError()がERROR_NO_MORE_ITEMSか調べる
                        if (Marshal.GetLastWin32Error() == ERROR_NO_MORE_ITEMS)
                        {
                            // 列挙終了としてresult=falseでbreak
                            break;
                        }
                    }

                    // フレンドリネームを取得
                    string friendly = getFriendlyName(info, devdata);

                    // フレンドリネームの取得に成功して
                    if (friendly != null)
                    {
                        // フレンドリネームが部分一致すれば
                        if (friendly.Contains(friendlyName))
                        {
                            // フレンドリネームを記憶
                            foundFriendlyName = friendly;
                            
                            // この時点でresult=trueで抜ける
                            break;
                        }
                    }
                }
            }

            // 列挙に成功していて
            if (result != false)
            {
                // アクションがあれば実施
                if (type != actionType.None)
                {
                    result = actionDeviceSub(info, devdata, type);
                }
            }

            // デバイス情報セットが存在すれば
            if (info != IntPtr.Zero)
            {
                // デバイス情報セットを解放
                SetupDiDestroyDeviceInfoList(info);
                info = IntPtr.Zero;
            }

            return result;
        }

        /// <summary>
        /// デバイスの有効化または無効化を行う
        /// </summary>
        /// <param name="info">デバイス情報セット</param>
        /// <param name="devdata">デバイス情報エレメント</param>
        /// <param name="type">デバイスに対するアクション(actionType列挙型)</param>
        /// <returns>ステータス(true:成功、false:失敗)</returns>
        private static bool actionDeviceSub(IntPtr info, SP_DEVINFO_DATA devdata, actionType type)
        {
            // 戻り値をtrueで初期化
            bool result = true;

            // SP_CLASSINSTALL_HEADERをDIF_PROPERTYCHANGEで初期化
            SP_CLASSINSTALL_HEADER header = new SP_CLASSINSTALL_HEADER();
            header.cbSize = (UInt32)Marshal.SizeOf(header);
            header.InstallFunction = DIF_PROPERTYCHANGE;

            // SP_PROPCHANGE_PARAMSをDICS_ENABLEまたはDICS_DISABLEで初期化
            SP_PROPCHANGE_PARAMS propchangeparams = new SP_PROPCHANGE_PARAMS();
            propchangeparams.ClassInstallHeader = header;
            switch (type)
            {
                // デバイスの有効化
                case actionType.Enable:
                    propchangeparams.StateChange = DICS_ENABLE;
                    break;
                // デバイスの無効化
                case actionType.Disable:
                    propchangeparams.StateChange = DICS_DISABLE;
                    break;
            }
            // すべてのハードウェアプロファイルに適用する
            propchangeparams.Scope = DICS_FLAG_GLOBAL;
            propchangeparams.HwProfile = 0;

            // デバイス情報エレメントにクラスインストールパラメータを設定
            result = SetupDiSetClassInstallParams(info,
                ref devdata,
                ref propchangeparams,
                (UInt32)Marshal.SizeOf(propchangeparams));

            // 成功すれば
            if (result != false)
            {
                // DIF_PROPERTYCHANGEを発行
                result = SetupDiChangeState(
                    info,
                    ref devdata);
            }

            return result;
        }

        /// <summary>
        /// デバイスのフレンドリネームを取得する
        /// </summary>
        /// <param name="info">デバイス情報セットのハンドル</param>
        /// <param name="devdata">デバイス情報エレメント</param>
        /// <returns>デバイスのフレンドリネーム(取得失敗時はnull)</returns>
        private static string getFriendlyName(IntPtr info, SP_DEVINFO_DATA devdata)
        {
            IntPtr buffer;
            uint regtype = 0;
            uint required = 0;
            string friendly = null;

            // バッファを512バイト固定で確保
            buffer = Marshal.AllocHGlobal(0x200);

            // バッファが有効な場合
            if (buffer != IntPtr.Zero)
            {
                // PnPデバイスのプロパティ(属性)を取得
                bool result = SetupDiGetDeviceRegistryPropertyW(
                    info,
                    ref devdata,
                    SPDRP_FRIENDLYNAME,
                    out regtype,
                    buffer,
                    0x200,
                    ref required);

                // 成功した場合は
                if (result != false)
                {
                    // requiredに応じた配列を確保し、コピー
                    byte[] lbuffer = new byte[required];
                    Marshal.Copy(buffer, lbuffer, 0, (int)required);

                    // フレンドリネームを取得
                    friendly = Encoding.Unicode.GetString(lbuffer);

                    // 終端の\0があれば削除する
                    friendly = friendly.Replace("\0", "");
                }

                // バッファを解放
                Marshal.FreeHGlobal(buffer);
                buffer = IntPtr.Zero;
            }

            return friendly;
        }
    }
}
