package bindata;

import static org.junit.Assert.*;
import static org.hamcrest.CoreMatchers.*;
import org.junit.*;


public class BinaryDatumTest {
	
	private BinaryDatum datum;
	
	@Before
	public void setUp() {
		datum = new BinaryDatum();
	}

	@Test
	public void testInitialLength() {
		assertThat(datum.length(), is(0));
	}
	
	@Test
	public void testAppendFirstHeader() {
		datum.append((byte)12);
		assertThat(datum.at(0), is(12));
		assertThat(datum.length(), is(1));
	}
	
	@Test
	public void testAppendBadFirstHeader() {
		datum.append((byte)11);
		assertThat(datum.at(0), is(-1));
		assertThat(datum.length(), is(0));
	}
	
	@Test
	public void testAppendSecondHeader() {
		datum.append((byte)12);
		datum.append((byte)48);
		assertThat(datum.at(0), is(12));
		assertThat(datum.at(1), is(48));
		assertThat(datum.length(), is(2));
	}
	
	@Test
	public void testAppendBadSecondHeader() {
		datum.append((byte)12);
		datum.append((byte)47);
		assertThat(datum.at(0), is(-1));
		assertThat(datum.at(1), is(-1));
		assertThat(datum.length(), is(0));
	}
	
	@Test
	public void testAppendLength() {
		datum.append((byte)12);
		datum.append((byte)48);
		datum.append((byte)(BinaryDatum.MIN_CONTENT_LENGTH + 1));
		assertThat(datum.at(2), is(BinaryDatum.MIN_CONTENT_LENGTH + 1));
		assertThat(datum.length(), is(3));
	}

	@Test
	public void testAppendMinLength() {
		datum.append((byte)12);
		datum.append((byte)48);
		datum.append((byte)(BinaryDatum.MIN_CONTENT_LENGTH));
		assertThat(datum.at(2), is(BinaryDatum.MIN_CONTENT_LENGTH));
		assertThat(datum.length(), is(3));
	}

	@Test
	public void testAppendSmallerLength() {
		datum.append((byte)12);
		datum.append((byte)48);
		datum.append((byte)(BinaryDatum.MIN_CONTENT_LENGTH - 1));
		assertThat(datum.at(2), is(-1));
		assertThat(datum.length(), is(0));
	}
	
	@Test
	public void testAppendMaxLength() {
		datum.append((byte)12);
		datum.append((byte)48);
		datum.append((byte)(BinaryDatum.MAX_CONTENT_LENGTH));
		assertThat(datum.at(2), is(BinaryDatum.MAX_CONTENT_LENGTH));
		assertThat(datum.length(), is(3));
	}
	
	@Test
	public void testAppendLargerLength() {
		datum.append((byte)12);
		datum.append((byte)48);
		datum.append((byte)(BinaryDatum.MAX_CONTENT_LENGTH + 1));
		assertThat(datum.at(2), is(-1));
		assertThat(datum.length(), is(0));
	}
	
	@Test
	public void testAppendContent() {
		byte[] data = new byte[] { 12, 48, 9, 1, 2 , 3, 4, 5 };
		for (byte each : data) {
			datum.append(each);
		}
		assertThat(datum.length(), is(data.length));
	}
	
	@Test
	public void testAppendIgnore() {
		byte[] data = new byte[] { 12, 48, 9, 1, 2 , 3, 4, 5 };
		int sum = 0;
		for (byte each : data) {
			sum += each;
			datum.append(each);
		}
		datum.append((byte)(sum & 0xFF));
		assertThat(datum.length(), is(data.length + 1));
		datum.append((byte)6);
		assertThat(datum.length(), is(data.length + 1));
	}
	
	@Test
	public void testAppendChecksum() {
		byte[] data = new byte[] { 12, 48, 7, 1, 2 , 3 };
		int sum = 0;
		for (byte each : data) {
			sum += each;
			datum.append(each);
		}
		datum.append((byte)(sum & 0xFF));
		assertThat(datum.length(), is(data.length + 1));
	}
	
	@Test
	public void testAppendWrongChecksum() {
		byte[] data = new byte[] { 12, 48, 9, 1, 2 , 3, 4, 5 };
		int sum = 0;
		for (byte each : data) {
			sum += each;
			datum.append(each);
		}
		assertThat(datum.length(), is(data.length));
		datum.append((byte)((sum & 0xFF) + 1));
		assertThat(datum.length(), is(0));
	}
	
	@Test
	public void testClear() {
		byte[] data = new byte[] { 12, 48, 9, 1, 2 , 3, 4, 5 };
		int sum = 0;
		for (byte each : data) {
			sum += each;
			datum.append(each);
		}
		datum.append((byte)(sum & 0xFF));
		assertThat(datum.length(), is(data.length + 1));
		datum.clear();
		assertThat(datum.length(), is(0));
	}

	@Test
	public void testIsComplete() {
		byte[] data = new byte[] { 12, 48, 9, 1, 2 , 3, 4, 5 };
		int sum = 0;
		for (byte each : data) {
			sum += each;
			datum.append(each);
		}
		assertThat(datum.isComplete(), is(false));
		datum.append((byte)(sum & 0xFF));
		assertThat(datum.isComplete(), is(true));
	}

	@Test
	public void testToString() {
		byte[] data = new byte[] { 12, 48, 11, 1, 2 , 3, 4, 5, 6, 7 };
		int sum = 0;
		for (byte each : data) {
			sum += each;
			datum.append(each);
		}
		datum.append((byte)(sum & 0xFF));
		String str = datum.toString();
		String header = str.substring(0, 5);
		String body = str.substring(5);
		body = body.substring(body.indexOf(","));
		assertThat(header, is("1248,"));
		assertThat(body, is(",1029,1543,"));
	}

	@Test
	public void testToStringIncomplete() {
		byte[] data = new byte[] { 12, 48, 9, 1, 2 , 3, 4, 5 };
		for (byte each : data) {
			datum.append(each);
		}
		String str = datum.toString();
		assertThat(str, is(""));
	}

}
