package bindata;

public class BinaryDatum {
	static final int MAX_CONTENT_LENGTH = 15;
	static final int MIN_CONTENT_LENGTH = 7;  // header id(2bytes) + length(1byte) + time(3bytes) + checksum(1byte)
	private static final int CONTENT_LENGTH_INDEX = 2;
	private byte[] content = new byte[MAX_CONTENT_LENGTH];
	private StringBuilder converter = null;
	private int length = 0;
	private long completeTime = 0;
	
	public BinaryDatum() {
		content[0] = 12;
		content[1] = 48;
	}
	
	public BinaryDatum(BinaryDatum initialDatum) {
		System.arraycopy(initialDatum.content, 0, content, 0, initialDatum.length());
		length = initialDatum.length();
	}
	
	public void append(byte value) {
		switch (length) {
		case 0:
			appendHeaderFirstId(value);
			break;
		case 1:
			appendHeaderSecondId(value);
			break;
		case CONTENT_LENGTH_INDEX:
			appendLength(value);
			break;
		default:
			if (length < content[CONTENT_LENGTH_INDEX] - 1) {
				appendBody(value);
			}
			else if (length == content[CONTENT_LENGTH_INDEX] - 1) {
				appendChecksum(value);
			}
			else {
				// ignore value.
			}
			break;
		}
	}

	private void appendHeaderFirstId(byte value) {
		if (value != 12) {
			return;
		}
		length ++;
	}

	private void appendHeaderSecondId(byte value) {
		if (value != 48) {
			clear();
			return;
		}
		length ++;
	}

	private void appendLength(byte value) {
		if (value < MIN_CONTENT_LENGTH) {
			clear();
			return;
		}
		if (value > MAX_CONTENT_LENGTH) {
			clear();
			return;
		}
		content[length ++] = value;
	}

	private void appendBody(byte value) {
		content[length ++] = value;
	}

	private void appendChecksum(byte value) {
		if (calcCheckSum() != value) {
			clear();
			return;
		}
		content[length ++] = value;
		completeTime = System.nanoTime();
	}

	public void clear() {
		length = 0;
	}
	
	public boolean isComplete() {
		return length > CONTENT_LENGTH_INDEX && length >= content[CONTENT_LENGTH_INDEX];
	}
	
	public String toString() {
		if (!isComplete()) {
			return "";
		}
		if (converter == null) {
			converter = new StringBuilder(64);
		}
		else {
			converter.setLength(0);
		}
		converter.append("1248,");
		// converter.append((((content[3] & 0xFF) << 16) | ((content[4] & 0xFF) << 8) | (content[5] & 0xFF))).append(",");
		// converter.append(calendar.getTimeInMillis());
		// converter.append(System.nanoTime()).append(",");
		converter.append(completeTime).append(",");
		for (int i = 6; i < content[CONTENT_LENGTH_INDEX] - 1; i += 2) {
			converter.append((content[i] << 8) | content[i + 1]).append(",");
		}
		return converter.toString();
	}
	
	public int at(int index) {
		if (index >= length) {
			return -1;
		}
		return content[index];
	}
	
	public int length() {
		return length;
	}

	private byte calcCheckSum() {
		int sum = 0;
		final int length = content[CONTENT_LENGTH_INDEX] - 1; 
		for (int i = 0; i < length; i ++) {
			sum += content[i];
		}
		return (byte) (sum & 0xFF);
	}
}
