﻿using System;
using System.Collections.Generic;
using System.Text;
#if MMF_DIC
using System.IO.MemoryMappedFiles;
#endif

namespace NMeCab.Core
{
	public static class StrUtils
	{
		private const byte Nul = (byte)0;

		/// <summary>
		/// バイト配列の中から終端が\0で表された文字列を取り出す。
		/// </summary>
		/// <remarks>
		/// バイト配列の長さはInt32.MaxValueを超えていても良い。
		/// </remarks>
		/// <param name="bytes">バイト配列</param>
		/// <param name="enc">文字エンコーディング</param>
		/// <returns>文字列（\0は含まない）</returns>
		public static string GetString (byte[] bytes, Encoding enc)
		{
			return StrUtils.GetString (bytes, 0, enc);
		}

		/// <summary>
		/// バイト配列の中から終端が\0で表された文字列を取り出す。
		/// </summary>
		/// <remarks>
		/// バイト配列の長さはInt32.MaxValueを超えていても良い。
		/// </remarks>
		/// <param name="bytes">バイト配列</param>
		/// <param name="offset">オフセット位置</param>
		/// <param name="enc">文字エンコーディング</param>
		/// <returns>文字列（\0は含まない）</returns>
		public static string GetString (byte[] bytes, int offset, Encoding enc)
		{
			//バイト長のカウント
			int byteCount = 0;
			while (bytes[byteCount+offset] != Nul) { //終端\0に到達するまでシーク
				checked {
					byteCount++;
				} //文字列のバイト長がInt32.MaxValueを超えたならエラー
			}

			return enc.GetString (bytes, offset, byteCount);
		}

#if MMF_DIC

        /// <summary>
        /// MemoryMappedViewAccessorから終端が\0で表された文字列を取り出す。
        /// </summary>
        /// <remarks>
        /// MemoryMappedViewAccessorの容量はInt32.MaxValueを超えていても良い。
        /// </remarks>
        /// <param name="accessor">MemoryMappedViewAccessor</param>
        /// <param name="index">オフセット位置</param>
        /// <param name="enc">文字エンコーディング</param>
        /// <param name="buffSize">内部で使用するバッファの初期サイズ</param>
        /// <returns>文字列（\0は含まない）</returns>
        public static string GetString(MemoryMappedViewAccessor accessor, long offset, Encoding enc,
                                        int buffSize = 128)
        {
            byte[] buff = new byte[buffSize]; //IO回数削減のためのバッファ配列
            accessor.ReadArray<byte>(offset, buff, 0, buffSize); //初期読込

            //バイト長のカウント
            int byteCount = 0;
            while (buff[byteCount] != Nul) //終端\0に到達するまでシーク
            {
                byteCount++;

                if (byteCount == buffSize) //バッファ配列の終端
                {
                    //バッファ配列の拡張と追加読込
                    checked { buffSize *= 2; } //Int32.MaxValueを超えたならエラー
                    byte[] newBuff = new byte[buffSize];
                    Buffer.BlockCopy(buff, 0, newBuff, 0, byteCount);
                    accessor.ReadArray<byte>(offset + byteCount, newBuff, byteCount, buffSize - byteCount);
                    buff = newBuff;
                }
            }

            //バッファ配列を文字列にデコード
            return enc.GetString(buff, 0, byteCount);
        }

#endif

	}
}
