/**
 * Moxkiriya standalone Wiki.
 * Blocklevel element parser, processor creator.
 * 
 * @author Ryuhei Terada
 * See the '<a href="{@docRoot}/copyright.html">Copyright</a>'
 */package com.wiki.standalone.moxkiriya.parser.blockparser;

import java.util.LinkedHashMap;
import java.util.Set;

import com.wiki.standalone.moxkiriya.WikiEngine;

/**
 * WikiBlockParserインスタンスを生成する。
 *
 */
public class WikiBlockParserCreator {
	/**	TOC構築クラス */
	private WikiTOCBuilder wikiTOCBuilder_;
	
	/** Wiki engine */
	private WikiEngine wikiEngine_;
	
	interface Creator {
		public WikiBlockParser create(WikiEngine wikiEngine);
	};
	
	/** WikiBlockParser Creator Table*/
	private LinkedHashMap<String, Creator> creatorMap 
		= new LinkedHashMap<String, Creator>() {

		private static final long serialVersionUID = 1L;

		{
			put(WikiHnBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiHnBlockParser(wikiEngine, wikiTOCBuilder_);
				}
			} );
			put(WikiHrBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiHrBlockParser(wikiEngine);
				}
			} );
			put(WikiTableBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiTableBlockParser(wikiEngine);
				}
			} );
			put(WikiUnorderedListBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiUnorderedListBlockParser(wikiEngine);
				}
			} );
			put(WikiOrderedListBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiOrderedListBlockParser(wikiEngine);
				}
			} );
			put(WikiBlockquoteBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiBlockquoteBlockParser(wikiEngine);
				}
			} );
			put(WikiPreBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiPreBlockParser(wikiEngine);
				}
			} );
			put(WikiCodeBlockParser.NOTATION_REGEX, new Creator() {
				@Override 
				public WikiBlockParser create(WikiEngine wikiEngine) {
					return new WikiCodeBlockParser(wikiEngine);
				}
			} );
		}
	};

	/**
	 * Constructor
	 * @param wikiRepository
	 */
	public WikiBlockParserCreator(WikiEngine wikiEngine) {
		wikiTOCBuilder_ = new WikiTOCBuilder();
		wikiEngine_ = wikiEngine;
	}
	
	/**
	 * lineのSyntaxにマッチするWikiBlockParserのインスタンスを生成する。
	 * @param line
	 * @return WikiBlockParser
	 */
	public WikiBlockParser create(String line) {
		WikiBlockParser parser = null;
		Set<String>     keySet = creatorMap.keySet();

		for(String key: keySet) {
			if(line.toLowerCase().matches(key) == true) {
				parser = creatorMap.get(key).create(wikiEngine_);
				break;
			}
		}
		
		if (parser == null) {
			parser = new WikiDefaultBlockParser(wikiEngine_);
		}
		
		return parser;
	}
	
	/** 
	 * TOC(目次)ノードを取得する
	 * @param firstLine
	 * @return
	 */
	public String getTOC(String firstLine) {
		return wikiTOCBuilder_.buildTOC(firstLine);
	}
}
