/****************************************************************************
Copyright (c) 2010-2012 cocos2d-x.org
Copyright (c) 2008-2010 Ricardo Quesada
Copyright (c) 2011      Zynga Inc.

http://www.cocos2d-x.org

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
****************************************************************************/
package org.cocos2d.draw_nodes;

import static javax.microedition.khronos.opengles.GL10.GL_COLOR_ARRAY;
import static javax.microedition.khronos.opengles.GL10.GL_FLOAT;
import static javax.microedition.khronos.opengles.GL10.GL_LINES;
import static javax.microedition.khronos.opengles.GL10.GL_LINE_LOOP;
import static javax.microedition.khronos.opengles.GL10.GL_LINE_STRIP;
import static javax.microedition.khronos.opengles.GL10.GL_POINTS;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_2D;
import static javax.microedition.khronos.opengles.GL10.GL_TEXTURE_COORD_ARRAY;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;

import javax.microedition.khronos.opengles.GL10;

import org.cocos2d.cocoa.CCGeometry.CCPoint;
import org.cocos2d.cocoa.CCGeometry.CCRect;
import org.cocos2d.include.ccMacros;
import org.cocos2d.include.ccTypes.ccColor4F;
import org.cocos2d.shaders.CCGLProgram;
import org.cocos2d.shaders.CCShaderCache;
import org.cocos2d.shaders.ccGLStateCache;
import org.cocos2d.support.CCPointExtension;

import android.opengl.GLES20;

/*
*
* IMPORTANT       IMPORTANT        IMPORTANT        IMPORTANT
*
*
* LEGACY FUNCTIONS
*
* USE CCDrawNode instead
*
*/

/**
 * @file
 * Drawing OpenGL ES primitives.
 * - ccDrawPoint, ccDrawPoints
 * - ccDrawLine
 * - ccDrawRect, ccDrawSolidRect
 * - ccDrawPoly, ccDrawSolidPoly
 * - ccDrawCircle
 * - ccDrawQuadBezier
 * - ccDrawCubicBezier
 * - ccDrawCatmullRom
 * - ccDrawCardinalSpline
 * 
 * You can change the color, point size, width by calling:
 * - ccDrawColor4B(), ccDrawColor4F()
 * - ccPointSize()
 * - glLineWidth()
 * 
 * @warning These functions draws the Line, Point, Polygon, immediately. They aren't batched. If you are going to make a game that depends on these primitives, I suggest creating a batch. Instead you should use CCDrawNode
 * 
 */

/**
 * @addtogroup global
 * @{
 */

public class CCDrawingPrimitives {

	/** Initializes the drawing primitives */
	public static void ccDrawInit() {
		lazy_init();
	}

	/** Frees allocated resources by the drawing primitives */
	public static void ccDrawFree() {
		s_pShader = null;
		s_bInitialized = false;
	}

	/** draws a point given x and y coordinate measured in points */
	public static void ccDrawPoint(final CCPoint point) {
		lazy_init();

		FloatBuffer p = FloatBuffer.allocate(2 * 1);
		p.put(0, point.x);
		p.put(1, point.y);
		p.position(0);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);
		s_pShader.use();
		s_pShader.setUniformsForBuiltins();

		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);
		s_pShader.setUniformLocationWith1f(s_nPointSizeLocation, s_fPointSize);

		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, p);

		GLES20.glDrawArrays(GLES20.GL_POINTS, 0, 1);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}


	/** draws an array of points.
	 * @since v0.7.2
	 */
	public static void ccDrawPoints(final CCPoint [] points, int numberOfPoints) {
		lazy_init();

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);
		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);
		s_pShader.setUniformLocationWith1f(s_nPointSizeLocation, s_fPointSize);

		FloatBuffer newPoints = FloatBuffer.allocate(2 * numberOfPoints);

		for(int i=0; i<numberOfPoints;i++) {
			newPoints.put(i * 2 + 0, points[i].x);
			newPoints.put(i * 2 + 1, points[i].y);
		}
		newPoints.position(0);
		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, newPoints);

		GLES20.glDrawArrays(GLES20.GL_POINTS, 0, numberOfPoints);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}

	/** draws a line given the origin and destination point measured in points */
	public static void ccDrawLine(final CCPoint origin, final CCPoint destination) {
		lazy_init();

		FloatBuffer vertices = FloatBuffer.allocate(2 * 2);
		vertices.put(origin.x);
		vertices.put(origin.y);
		vertices.put(destination.x);
		vertices.put(destination.y);
		vertices.position(0);

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);
		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, vertices);
		GLES20.glDrawArrays(GL_LINES, 0, 2);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}

	/** draws a rectangle given the origin and destination point measured in points. */
	public static void ccDrawRect(final CCPoint origin, final CCPoint destination) {
		ccDrawLine(CCPoint.Make(origin.x, origin.y), CCPoint.Make(destination.x, origin.y));
		ccDrawLine(CCPoint.Make(destination.x, origin.y), CCPoint.Make(destination.x, destination.y));
		ccDrawLine(CCPoint.Make(destination.x, destination.y), CCPoint.Make(origin.x, destination.y));
		ccDrawLine(CCPoint.Make(origin.x, destination.y), CCPoint.Make(origin.x, origin.y));
	}

	/** draws a solid rectangle given the origin and destination point measured in points.
	 * @since 1.1
	 */
	public static void ccDrawSolidRect(final CCPoint origin, final CCPoint destination, final ccColor4F color) {
		CCPoint [] vertices = {
				origin,
				CCPointExtension.ccp(destination.x, origin.y),
				destination,
				CCPointExtension.ccp(origin.x, destination.y)
		};

		ccDrawSolidPoly(vertices, 4, color);
	}

	/** draws a polygon given a pointer to CCPoint coordinates and the number of vertices measured in points.
	 * The polygon can be closed or open
	 */
	public static void ccDrawPoly(final CCPoint [] poli, int numberOfPoints, boolean closePolygon) {
		lazy_init();

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);

		FloatBuffer newPoli = FloatBuffer.allocate(2 * numberOfPoints);
		for(int i=0; i<numberOfPoints;i++) {
			newPoli.put(poli[i].x);
			newPoli.put(poli[i].y);
		}
		newPoli.position(0);
		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, newPoli);

		if( closePolygon ) {
			GLES20.glDrawArrays(GLES20.GL_LINE_LOOP, 0, numberOfPoints);
		} else {
			GLES20.glDrawArrays(GLES20.GL_LINE_STRIP, 0, numberOfPoints);
		}

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}

	/** draws a solid polygon given a pointer to CGPoint coordinates, the number of vertices measured in points, and a color.
	 */
	public static void ccDrawSolidPoly(final CCPoint [] poli, int numberOfPoints, final ccColor4F color) {
		lazy_init();

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {color.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);

		FloatBuffer newPoli = FloatBuffer.allocate(2 * numberOfPoints);
		for(int i=0; i<numberOfPoints;i++) {
			newPoli.put(poli[i].x);
			newPoli.put(poli[i].y);
		}
		newPoli.position(0);

		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, newPoli);

		GLES20.glDrawArrays(GLES20.GL_TRIANGLE_FAN, 0, numberOfPoints);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}


	/** draws a circle given the center, radius and number of segments. */
	public static void ccDrawCircle(final CCPoint center, float radius, float angle, int segments, boolean drawLineToCenter, float scaleX, float scaleY) {
		lazy_init();

		int additionalSegment = 1;
		if (drawLineToCenter) {
			additionalSegment++;
		}

		final float coef = 2.0f * (float)Math.PI/segments;

		FloatBuffer vertices = FloatBuffer.allocate(2 * (segments + 2));

		for(int i = 0;i <= segments; i++) {
			float rads = i*coef;
			float j = (float) (radius * Math.cos(rads + angle) * scaleX + center.x);
			float k = (float) (radius * Math.sin(rads + angle) * scaleY + center.y);

			vertices.put(i*2, j);
			vertices.put(i*2+1,  k);
		}
		vertices.put((segments+1)*2, center.x);
		vertices.put((segments+1)*2+1, center.y);
		vertices.position(0);

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);

		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, vertices);
		GLES20.glDrawArrays(GLES20.GL_LINE_STRIP, 0, segments+additionalSegment);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}

	public static void ccDrawCircle(final CCPoint center, float radius, float angle, int segments, boolean drawLineToCenter) {
		ccDrawCircle(center, radius, angle, segments, drawLineToCenter, 1.0f, 1.0f);
	}

	/** draws a quad bezier path
	 * @warning This function could be pretty slow. Use it only for debugging purposes.
	 * @since v0.8
	 */
	public static void ccDrawQuadBezier(final CCPoint origin, final CCPoint control, final CCPoint destination, int segments) {
		lazy_init();

		FloatBuffer vertices = FloatBuffer.allocate(2 * (segments + 1));

		float t = 0.0f;
		for(int i = 0; i < segments; i++) {
			vertices.put((float) (Math.pow(1 - t, 2) * origin.x + 2.0f * (1 - t) * t * control.x + t * t * destination.x));
			vertices.put((float) (Math.pow(1 - t, 2) * origin.y + 2.0f * (1 - t) * t * control.y + t * t * destination.y));
			t += 1.0f / segments;
		}
		vertices.put(destination.x);
		vertices.put(destination.y);
		vertices.position(0);

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);

		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, vertices);
		GLES20.glDrawArrays(GLES20.GL_LINE_STRIP, 0, segments + 1);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}

	/** draws a cubic bezier path
	 * @warning This function could be pretty slow. Use it only for debugging purposes.
	 * @since v0.8
	 */
	public static void ccDrawCubicBezier(final CCPoint origin, final CCPoint control1, final CCPoint control2, final CCPoint destination, int segments) {
		lazy_init();

		FloatBuffer vertices = FloatBuffer.allocate(2 * (segments + 1));

		float t = 0;
		for(int i = 0; i < segments; i++) {
			vertices.put((float) (Math.pow(1 - t, 3) * origin.x + 3.0f * Math.pow(1 - t, 2) * t * control1.x + 3.0f * (1 - t) * t * t * control2.x + t * t * t * destination.x));
			vertices.put((float) (Math.pow(1 - t, 3) * origin.y + 3.0f * Math.pow(1 - t, 2) * t * control1.y + 3.0f * (1 - t) * t * t * control2.y + t * t * t * destination.y));
			t += 1.0f / segments;
		}
		vertices.put(destination.x);
		vertices.put(destination.y);
		vertices.position(0);

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);

		GLES20.glVertexAttribPointer(CCGLProgram.kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, vertices);
		GLES20.glDrawArrays(GLES20.GL_LINE_STRIP, 0, segments + 1);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}

	/** draws a Catmull Rom path.
	 * @warning This function could be pretty slow. Use it only for debugging purposes.
	 * @since v2.0
	 */
/* TODO
	public static void ccDrawCatmullRom(CCPointArray *arrayOfControlPoints, int segments) {
		ccDrawCardinalSpline(points, 0.5f, segments);
	}
*/
	/** draws a Cardinal Spline path.
	 * @warning This function could be pretty slow. Use it only for debugging purposes.
	 * @since v2.0
	 */
/* TODO
	public static void ccDrawCardinalSpline( CCPointArray *config, float tension,  unsigned int segments ) {
		lazy_init();

		FloatBuffer vertices = FloatBuffer.allocate(2 * (segments + 1));

		int p;
		float lt;
		float deltaT = 1.0f / config.count();

		for(int i=0; i < segments+1;i++) {

			float dt = (float)i / segments;

			// border
			if( dt == 1 ) {
				p = config.count() - 1;
				lt = 1;
			} else {
				p = dt / deltaT;
				lt = (dt - deltaT * (float)p) / deltaT;
			}

			// Interpolate
			CCPoint pp0 = config.getControlPointAtIndex(p-1);
			CCPoint pp1 = config.getControlPointAtIndex(p+0);
			CCPoint pp2 = config.getControlPointAtIndex(p+1);
			CCPoint pp3 = config.getControlPointAtIndex(p+2);

			CCPoint newPos = ccCardinalSplineAt( pp0, pp1, pp2, pp3, tension, lt);
			vertices.put(newPos.x);
			vertices.put(newPos.y);
		}
		vertices.position(0);

		s_pShader.use();
		s_pShader.setUniformsForBuiltins();
		float [] color_r = {s_tColor.r};
		s_pShader.setUniformLocationWith4fv(s_nColorLocation, color_r, 1);

		ccGLStateCache.ccGLEnableVertexAttribs(ccGLStateCache.kCCVertexAttribFlag_Position);

		GLES20.glVertexAttribPointer(kCCVertexAttrib_Position, 2, GLES20.GL_FLOAT, false, 0, vertices);
		GLES20.glDrawArrays(GL_LINE_STRIP, 0, (GLsizei) segments + 1);

		ccMacros.CC_INCREMENT_GL_DRAWS(1);
	}
*/
	/** set the drawing color with 4 unsigned bytes
	 * @since v2.0
	 */
	public static void ccDrawColor4B(int r, int g, int b, int a) {
		s_tColor.r = r/255.0f;
		s_tColor.g = g/255.0f;
		s_tColor.b = b/255.0f;
		s_tColor.a = a/255.0f;
	}

	/** set the drawing color with 4 floats
	 * @since v2.0
	 */
	public static void ccDrawColor4F(float r, float g, float b, float a) {
		s_tColor.r = r;
		s_tColor.g = g;
		s_tColor.b = b;
		s_tColor.a = a;
	}

	/** set the point size in points. Default 1.
	 * @since v2.0
	 */
	public static void ccPointSize(float pointSize) {
		s_fPointSize = pointSize * ccMacros.CC_CONTENT_SCALE_FACTOR();

		//TODO :glPointSize( pointSize );

	}

	private static boolean s_bInitialized = false;
	private static CCGLProgram s_pShader = null;
	private static int s_nColorLocation = -1;
	private static ccColor4F s_tColor = new ccColor4F(1.0f,1.0f,1.0f,1.0f);
	private static int s_nPointSizeLocation = -1;
	private static float s_fPointSize = 1.0f;

	private static void lazy_init() {
		if(! s_bInitialized) {

			//
			// Position and 1 color passed as a uniform (to simulate glColor4ub )
			//
			s_pShader = CCShaderCache.sharedShaderCache().programForKey(CCGLProgram.kCCShader_Position_uColor);

			s_nColorLocation = GLES20.glGetUniformLocation(s_pShader.getProgram(), "u_color");
			ccMacros.CHECK_GL_ERROR_DEBUG();
			s_nPointSizeLocation = GLES20.glGetUniformLocation(s_pShader.getProgram(), "u_pointSize");
			ccMacros.CHECK_GL_ERROR_DEBUG();

			s_bInitialized = true;
		}
	}

	// TODO legacy -->

	private static FloatBuffer tmpFloatBuf;

	private static FloatBuffer getVertices(int size) {
		if(tmpFloatBuf == null || tmpFloatBuf.capacity() < size) {
	        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * size);
	        vbb.order(ByteOrder.nativeOrder());
	        tmpFloatBuf = vbb.asFloatBuffer();
		}
		tmpFloatBuf.rewind();
		return tmpFloatBuf;
	}
	
    /** draws a point given x and y coordinate */
    public static void ccDrawPoint(GL10 gl, CCPoint pnt) {
//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * 1);
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * 1);

        vertices.put(pnt.x);
        vertices.put(pnt.y);
        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL10.GL_TEXTURE_2D);
        gl.glDisableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL10.GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        gl.glDrawArrays(GL_POINTS, 0, 1);

        // restore default state
        gl.glEnableClientState(GL10.GL_COLOR_ARRAY);
        gl.glEnableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL10.GL_TEXTURE_2D);	
    }

    /** draws an array of points.
      @since v0.7.2
    */
    public static void ccDrawPoints(GL10 gl, CCPoint points[], int numberOfPoints) {
//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * numberOfPoints);
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * numberOfPoints);

        for (int i = 0; i < numberOfPoints; i++) {
            vertices.put(points[i].x);
            vertices.put(points[i].y);
        }
        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL_TEXTURE_2D);
        gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        gl.glDrawArrays(GL_POINTS, 0, numberOfPoints);

        // restore default state
        gl.glEnableClientState(GL_COLOR_ARRAY);
        gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL_TEXTURE_2D);	
    }

    /** draws a line given the origin and destination point */
    public static void ccDrawLine(GL10 gl, CCPoint origin, CCPoint destination) {
//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * 2);
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * 2);

        vertices.put(origin.x);
        vertices.put(origin.y);
        vertices.put(destination.x);
        vertices.put(destination.y);
        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL_TEXTURE_2D);
        gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        gl.glDrawArrays(GL_LINES, 0, 2);

        // restore default state
        gl.glEnableClientState(GL_COLOR_ARRAY);
        gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL_TEXTURE_2D);	
    }

    public static void ccDrawRect(GL10 gl, CCRect rect) {
        CCPoint[] poli = new CCPoint[4];

        poli[0] = CCPointExtension.ccp(rect.origin.x, rect.origin.y);
        poli[1] = CCPointExtension.ccp(rect.origin.x + rect.size.width, rect.origin.y);
        poli[2] = CCPointExtension.ccp(rect.origin.x + rect.size.width, rect.origin.y + rect.size.height);
        poli[3] = CCPointExtension.ccp(rect.origin.x, rect.origin.y + rect.size.height);

        ccDrawPoly(gl, poli, poli.length, true);
    }

    /**
     * draws a poligon given a pointer to CGPoint coordiantes and the number of vertices.
     * The polygon can be closed or open
    */
    public static void ccDrawPoly(GL10 gl, CCPoint poli[], int numberOfPoints, boolean closePolygon) {
//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * numberOfPoints);
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * numberOfPoints);

        for (int i = 0; i < numberOfPoints; i++) {
            vertices.put(poli[i].x);
            vertices.put(poli[i].y);
        }
        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL_TEXTURE_2D);
        gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        if (closePolygon)
            gl.glDrawArrays(GL_LINE_LOOP, 0, numberOfPoints);
        else
            gl.glDrawArrays(GL_LINE_STRIP, 0, numberOfPoints);

        // restore default state
        gl.glEnableClientState(GL_COLOR_ARRAY);
        gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL_TEXTURE_2D);	
    }

    /** draws a circle given the center, radius and number of segments. */
    public static void ccDrawCircle(GL10 gl, CCPoint center, float r, float a,
            int segments, boolean drawLineToCenter) {

//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * (segments + 2));
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * (segments + 2));

        int additionalSegment = 1;

        if (drawLineToCenter)
            additionalSegment++;

        final float coef = 2.0f * (float) Math.PI / segments;
        for (int i = 0; i <= segments; i++) {
            float rads = i * coef;
            float j = (float) (r * Math.cos(rads + a) + center.x);
            float k = (float) (r * Math.sin(rads + a) + center.y);

            vertices.put(j);
            vertices.put(k);
        }
        vertices.put(center.x);
        vertices.put(center.y);
        
        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL_TEXTURE_2D);
        gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        gl.glDrawArrays(GL_LINE_STRIP, 0, segments + additionalSegment);

        // restore default state
        gl.glEnableClientState(GL_COLOR_ARRAY);
        gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL_TEXTURE_2D);	
    }

    /** draws a quad bezier path since v0.8 */
    public static void ccDrawQuadBezier(GL10 gl, CCPoint origin, CCPoint control,
            CCPoint destination, int segments) {

//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * (segments + 1));
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * (segments + 1));
        
        float t = 0.0f;
        for(int i = 0; i < segments; i++) {
            float x = (float)Math.pow(1 - t, 2) * origin.x + 2.0f * (1 - t) * t * control.x + t * t * destination.x;
            float y = (float)Math.pow(1 - t, 2) * origin.y + 2.0f * (1 - t) * t * control.y + t * t * destination.y;
            vertices.put(x);
            vertices.put(y);
            t += 1.0f / segments;
        }
        vertices.put(destination.x);
        vertices.put(destination.y);

        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL10.GL_TEXTURE_2D);
        gl.glDisableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL10.GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        gl.glDrawArrays(GL_LINE_STRIP, 0, segments + 1);

        // restore default state
        gl.glEnableClientState(GL10.GL_COLOR_ARRAY);
        gl.glEnableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL10.GL_TEXTURE_2D);	
    }

    /**
     * draws a cubic bezier path
     @since v0.8
    */
    public static void ccDrawCubicBezier(GL10 gl, CCPoint origin, CCPoint control1, CCPoint control2,
            CCPoint destination, int segments) {

//        ByteBuffer vbb = ByteBuffer.allocateDirect(4 * 2 * (segments + 1));
//        vbb.order(ByteOrder.nativeOrder());
//        FloatBuffer vertices = vbb.asFloatBuffer();
        FloatBuffer vertices = getVertices(2 * (segments + 1));

        float t = 0;
        for(int i = 0; i < segments; i++)
        {
            float x = (float)Math.pow(1 - t, 3) * origin.x + 3.0f * (float)Math.pow(1 - t, 2) * t * control1.x + 3.0f * (1 - t) * t * t * control2.x + t * t * t * destination.x;
            float y = (float)Math.pow(1 - t, 3) * origin.y + 3.0f * (float)Math.pow(1 - t, 2) * t * control1.y + 3.0f * (1 - t) * t * t * control2.y + t * t * t * destination.y;
            vertices.put(x);
            vertices.put(y);
            t += 1.0f / segments;
        }
        vertices.put(destination.x);
        vertices.put(destination.y);

        vertices.position(0);

        // Default GL states: GL_TEXTURE_2D, GL_VERTEX_ARRAY, GL_COLOR_ARRAY, GL_TEXTURE_COORD_ARRAY
        // Needed states: GL_VERTEX_ARRAY, 
        // Unneeded states: GL_TEXTURE_2D, GL_TEXTURE_COORD_ARRAY, GL_COLOR_ARRAY	
        gl.glDisable(GL_TEXTURE_2D);
        gl.glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glDisableClientState(GL_COLOR_ARRAY);

        gl.glVertexPointer(2, GL_FLOAT, 0, vertices);
        gl.glDrawArrays(GL_LINE_STRIP, 0, segments + 1);
        
        // restore default state
        gl.glEnableClientState(GL_COLOR_ARRAY);
        gl.glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        gl.glEnable(GL_TEXTURE_2D);	

    }
    
}

// end of global group
/// @}
