package org.cocos2d.menus;

import org.cocos2d.nodes.CCNode;
import org.cocos2d.protocols.CCRGBAProtocol;
import org.cocos2d.types.CGPoint;
import org.cocos2d.types.ccColor3B;

/**
 * @addtogroup GUI
 * @{
 * @addtogroup menu
 * @{
 */


/** @brief CCMenuItemSprite accepts CCNode<CCRGBAProtocol> objects as items.
 * The images has 3 different states:
 * - unselected image
 * - selected image
 * - disabled image
 * 
 * @since v0.8.0
 */
public class CCMenuItemSprite extends CCMenuItem {

	/** the image used when the item is not selected */
	protected CCNode m_pNormalImage;

	public void setNormalImage(CCNode pImage) {
		if(pImage != m_pNormalImage) {
			if(pImage != null) {
				addChild(pImage, 0, kNormalTag);
				pImage.setAnchorPoint(CGPoint.ccp(0.0f, 0.0f));
			}

			if(m_pNormalImage != null) {
				removeChild(m_pNormalImage,  true);
			}

			m_pNormalImage = pImage;
			this.setContentSize(m_pNormalImage.getContentSize());
			this.updateImagesVisibility();
		}
	}

	public CCNode getNormalImage() {
		return m_pNormalImage;
	}

	/** the image used when the item is selected */
	protected CCNode m_pSelectedImage;

	public void setSelectedImage(CCNode pImage) {
		if(pImage != m_pSelectedImage) {
			if(pImage != null) {
				addChild(pImage);
				pImage.setAnchorPoint(CGPoint.ccp(0.0f, 0.0f));
			}

			if(m_pSelectedImage != null) {
				removeChild(m_pSelectedImage, true);
			}

			m_pSelectedImage = pImage;
			this.updateImagesVisibility();
		}
	}

	public CCNode getSelectedImage() {
		return m_pSelectedImage;
	}

	/** the image used when the item is disabled */
	protected CCNode m_pDisabledImage;

	public void setDisabledImage(CCNode image) {
		if(image != m_pDisabledImage) {
			if (image != null) {
				addChild(image);
				image.setAnchorPoint(CGPoint.ccp(0.0f, 0.0f));
			}

			if (m_pDisabledImage != null) {
				removeChild(m_pDisabledImage, true);
			}

			m_pDisabledImage = image;
			this.updateImagesVisibility();
		}
	}

	public CCNode getDisabledImage(){
		return m_pDisabledImage;
	}

	public CCMenuItemSprite() {
		m_pNormalImage = null;
		m_pSelectedImage = null;
		m_pDisabledImage = null;
	}

	/** creates a menu item with a normal, selected and disabled image*/
	public static CCMenuItemSprite create(CCNode normalSprite, CCNode selectedSprite) {
		return CCMenuItemSprite.create(normalSprite, selectedSprite, null);
	}

	public static CCMenuItemSprite create(CCNode normalSprite, CCNode selectedSprite, CCNode disabledSprite) {
		return CCMenuItemSprite.create(normalSprite, selectedSprite, disabledSprite, null, null);
	}

	/** creates a menu item with a normal and selected image with target/selector */
	public static CCMenuItemSprite create(CCNode normalSprite, CCNode selectedSprite, Object target, String selector) {
		return CCMenuItemSprite.create(normalSprite, selectedSprite, null, target, selector);
	}

	/** creates a menu item with a normal,selected  and disabled image with target/selector */
	public static CCMenuItemSprite create(CCNode normalSprite, CCNode selectedSprite, CCNode disabledSprite, Object target, String selector) {
		CCMenuItemSprite pRet = new CCMenuItemSprite();
		pRet.initWithNormalSprite(normalSprite, selectedSprite, disabledSprite, target, selector);
		return pRet;
	}

	/** initializes a menu item with a normal, selected  and disabled image with target/selector */
	public boolean initWithNormalSprite(CCNode normalSprite, CCNode selectedSprite, CCNode disabledSprite, Object target, String selector) {
		super.initWithTarget(target, selector);
		setNormalImage(normalSprite);
		setSelectedImage(selectedSprite);
		setDisabledImage(disabledSprite);

		if(m_pNormalImage != null) {
			this.setContentSize(m_pNormalImage.getContentSize());
		}

		setCascadeColorEnabled(true);
		setCascadeOpacityEnabled(true);

		return true;
	}

	/**
	 * @since v0.99.5
	 */

	@Override
	public void selected() {
		super.selected();

		if(m_pNormalImage != null) {
			if(m_pDisabledImage != null) {
				m_pDisabledImage.setVisible(false);
			}

			if (m_pSelectedImage != null) {
				m_pNormalImage.setVisible(false);
				m_pSelectedImage.setVisible(true);
			} else {
				m_pNormalImage.setVisible(true);
			}
		}
	}

	@Override
	public void unselected() {
		super.unselected();
		if(m_pNormalImage != null) {
			m_pNormalImage.setVisible(true);

			if(m_pSelectedImage != null) {
				m_pSelectedImage.setVisible(false);
			}

			if (m_pDisabledImage != null) {
				m_pDisabledImage.setVisible(false);
			}
		}
	}

	@Override
	public void setEnabled(boolean bEnabled) {
		if(m_bEnabled != bEnabled) {
			super.setEnabled(bEnabled);
			this.updateImagesVisibility();
		}
	}

	@Override
	public void setOpacityModifyRGB(boolean bValue) {
	}

	@Override
	public boolean isOpacityModifyRGB() {
		return false;
	}

	protected void updateImagesVisibility() {
		if(m_bEnabled) {
			if(m_pNormalImage != null) m_pNormalImage.setVisible(true);
			if(m_pSelectedImage != null) m_pSelectedImage.setVisible(false);
			if( m_pDisabledImage != null ) m_pDisabledImage.setVisible(false);
		} else {
			if( m_pDisabledImage != null) {
				if(m_pNormalImage != null) m_pNormalImage.setVisible(false);
				if(m_pSelectedImage != null) m_pSelectedImage.setVisible(false);
				if(m_pDisabledImage != null) m_pDisabledImage.setVisible(true);
			} else {
				if(m_pNormalImage != null) m_pNormalImage.setVisible(true);
				if(m_pSelectedImage != null) m_pSelectedImage.setVisible(false);
				if(m_pDisabledImage != null) m_pDisabledImage.setVisible(false);
			}
		}
	}



	// TODO legacy -->

    // CocosNodeRGBA protocol
    public void setOpacity(int opacity) {
        ((CCRGBAProtocol) m_pNormalImage).setOpacity(opacity);
        ((CCRGBAProtocol) m_pSelectedImage).setOpacity(opacity);
        if (m_pDisabledImage != null)
            ((CCRGBAProtocol) m_pDisabledImage).setOpacity(opacity);
    }

    public void setColor(ccColor3B color) {
        ((CCRGBAProtocol) m_pNormalImage).setColor(color);
        ((CCRGBAProtocol) m_pSelectedImage).setColor(color);
        if (m_pDisabledImage != null)
            ((CCRGBAProtocol) m_pDisabledImage).setColor(color);
    }

    public ccColor3B getColor() {
        return ((CCRGBAProtocol) m_pNormalImage).getColor();
    }

    public int getOpacity() {
        return ((CCRGBAProtocol) m_pNormalImage).getOpacity();
    }
}

// end of GUI group
/// @}
/// @}
