package org.cocos2d.menus;

import org.cocos2d.actions.base.CCAction;
import org.cocos2d.actions.interval.CCScaleTo;
import org.cocos2d.nodes.CCNode;
import org.cocos2d.protocols.CCLabelProtocol;
import org.cocos2d.protocols.CCRGBAProtocol;
import org.cocos2d.types.CGPoint;
import org.cocos2d.types.ccColor3B;

/**
 * @addtogroup GUI
 * @{
 * @addtogroup menu
 * @{
 */

/** @brief An abstract class for "label" CCMenuItemLabel items 
 * Any CCNode that supports the CCLabelProtocol protocol can be added.
 * Supported nodes:
 * - CCBitmapFontAtlas
 * - CCLabelAtlas
 * - CCLabelTTF
 */
public class CCMenuItemLabel extends CCMenuItem {

	/** the color that will be used to disable the item */
	protected ccColor3B m_tDisabledColor;

	public ccColor3B getDisabledColor() {
		return m_tDisabledColor;
	}

	public void setDisabledColor(ccColor3B color) {
		m_tDisabledColor = color;
	}

	/** Label that is rendered. It can be any CCNode that implements the CCLabelProtocol */
	protected CCNode m_pLabel;

	public CCNode getLabel() {
		return m_pLabel;
	}

	public void setLabel(CCNode label) {

		if(label != null) {
			addChild(label);
			label.setAnchorPoint(CGPoint.ccp(0.0f, 0.0f));
			setContentSize(label.getContentSize());
		}

		if(m_pLabel != null) {
			removeChild(m_pLabel, true);
		}

		m_pLabel = label;
	}

	public CCMenuItemLabel() {
		m_pLabel = null;
		m_fOriginalScale = 0.0f;
	}

	/** creates a CCMenuItemLabel with a Label, target and selector */
	public static CCMenuItemLabel create(CCNode label, Object target, String selector) {
		CCMenuItemLabel pRet = new CCMenuItemLabel();
		pRet.initWithLabel(label, target, selector);
		return pRet;
	}

	/** creates a CCMenuItemLabel with a Label. Target and selector will be nil */
	public static CCMenuItemLabel create(CCNode label) {
		CCMenuItemLabel pRet = new CCMenuItemLabel();
		pRet.initWithLabel(label, null, null);
		return pRet;
	}

	/** initializes a CCMenuItemLabel with a Label, target and selector */
	public boolean initWithLabel(CCNode label, Object target, String selector) {
		super.initWithTarget(target, selector);
		m_fOriginalScale = 1.0f;
		m_tColorBackup = ccColor3B.ccWHITE;
		setDisabledColor(ccColor3B.ccc3(126,126,126));
		this.setLabel(label);

		setCascadeColorEnabled(true);
		setCascadeOpacityEnabled(true);

		return true;
	}

	/** sets a new string to the inner label */
	public void setString(String label) {
		((CCLabelProtocol)m_pLabel).setString(label);
		this.setContentSize(m_pLabel.getContentSize());
	}

	// super methods

	@Override
	public void activate() {
		if(m_bEnabled) {
			this.stopAllActions();
			this.setScale(m_fOriginalScale);
			super.activate();
		}
	}

	@Override
	public void selected() {
		// subclass to change the default action
		if(m_bEnabled) {
			super.selected();

			CCAction action = getActionByTag(kZoomActionTag);
			if(action != null) {
				this.stopAction(action);
			} else {
				m_fOriginalScale = this.getScale();
			}

			CCAction zoomAction = CCScaleTo.action(0.1f, m_fOriginalScale * 1.2f);
			zoomAction.setTag(kZoomActionTag);
			runAction(zoomAction);
		}
	}

	@Override
	public void unselected() {
		// subclass to change the default action
		if(m_bEnabled) {
			super.unselected();
			this.stopActionByTag(kZoomActionTag);
			CCAction zoomAction = CCScaleTo.action(0.1f, m_fOriginalScale);
			zoomAction.setTag(kZoomActionTag);
			this.runAction(zoomAction);
		}
	}

	/** Enable or disabled the CCMenuItemFont
	 * @warning setEnabled changes the RGB color of the font
	 */
	public void setEnabled(boolean enabled) {
		if(m_bEnabled != enabled) {
			if(enabled == false) {
				m_tColorBackup = ((CCRGBAProtocol) m_pLabel).getColor();
				((CCRGBAProtocol)m_pLabel).setColor(m_tDisabledColor);
			} else {
				((CCRGBAProtocol)m_pLabel).setColor(m_tColorBackup);
			}
		}
		super.setEnabled(enabled);
	}

	protected ccColor3B m_tColorBackup;
	protected float m_fOriginalScale;
}

// end of GUI group
/// @}
/// @}
