#
# Copyright (c) 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#

module Luce
  # Parses email-style blockquotes: `> quote`.
  class BlockquoteSyntax < BlockSyntax
    def pattern : Regex
      Luce.blockquote_pattern
    end

    def parse_child_lines(parser : BlockParser) : Array(String)
      # Grab all of the lines that form the blockquote, stripping off the ">".
      child_lines = [] of String

      while !parser.done?
        current_line = parser.current
        match = pattern.match(parser.current)
        if !match.nil?
          # A block quote marker consists of a `>` together with an optional
          # following space of indentation, see
          # https://spec.commonmark.org/0.30/#block-quote-marker.
          marker_start = match.match.index('>') || 0
          marker_end = if current_line.size > 1
                         next_char = current_line.codepoint_at(marker_start + 1)
                         has_space = next_char == Charcode::TAB || next_char == Charcode::SPACE
                         marker_start + (has_space ? 2 : 1)
                       else
                         marker_start + 1
                       end
          child_lines << current_line[marker_end..]
          parser.advance
          next
        end

        last_line = child_lines.last

        # A paragraph continuation is OK. This is content that cannot
        # be parsed as any other syntax except Paragraph, and it
        # doesn't match the bar in a Setext header.
        # Because indented code blocks cannot interrupt paragraphs, a line
        # matched CodeBlockSyntax is also paragraph continuation text.
        other_matched = parser.block_syntaxes.find(&.can_parse?(parser))
        if (other_matched.is_a? ParagraphSyntax && last_line.empty? == false && false == Luce.code_fence_pattern.matches?(last_line)) ||
           (other_matched.is_a? CodeBlockSyntax && false == Luce.indent_pattern.matches?(last_line))
          child_lines << parser.current
          parser.advance
        else
          break
        end
      end

      child_lines
    end

    def parse(parser : BlockParser) : Node
      child_lines = parse_child_lines(parser)

      # Recursively parse the contents of the blockquote.
      children = BlockParser.new(child_lines, parser.document).parse_lines

      Element.new("blockquote", children)
    end
  end
end
