#include "Mix/Private/Graphics/DX11/VertexBuffer.h"
#include "Mix/Private/Graphics/DX11/Device.h"

namespace Mix{ namespace Graphics{ namespace DX11{

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::DX11::VertexBuffer
////////////////////////////////////////////////////////////////////////////////////////////////////

VertexBuffer* VertexBuffer::CreateInstance(	Mix::Graphics::DX11::Device* pDevice,
											UInt32 count, UInt32 stride,
											Boolean bDynamic, UInt32 resizeStep )
{
	return MIX_LIB_NEW_T( Mix::Memory::SECTION_GRAPHICS, VertexBuffer, pDevice, count, stride, bDynamic, resizeStep );
}

VertexBuffer::VertexBuffer(	Mix::Graphics::DX11::Device* pDevice,
							UInt32 count, UInt32 stride,
							Boolean bDynamic, UInt32 resizeStep ) :
DeviceObject( pDevice ),
m_pDevice( pDevice ),
m_Stride( stride ),
m_bDynamic( bDynamic ),
m_ResizeStep( resizeStep ),
m_pVertexBuffer( NULL ),
m_bLock( False ),
m_bResize( False ),
m_pBuffer( NULL ),
m_pSource( NULL ),
m_Capacity( count ),
m_Count( 0 )
{
	MIX_ADD_REF( m_pDevice );
}

VertexBuffer::~VertexBuffer( void )
{
	MIX_LIB_FREE( m_pBuffer );
	MIX_RELEASE( m_pVertexBuffer );
	MIX_RELEASE( m_pDevice );
}

HRESULT VertexBuffer::Initialize( const void* pInitialData )
{
	HRESULT ret = m_pDevice->CreateVertexBuffer( m_Capacity, m_Stride, m_bDynamic, pInitialData, &m_pVertexBuffer );
	if( ret != S_OK )
	{
		return ret;
	}

	if( ( m_ResizeStep > 0 ) &&
		( m_bDynamic == True ) )
	{
		m_pBuffer = static_cast<UInt8*>( MIX_LIB_MALLOC( Mix::Memory::SECTION_GRAPHICS, m_Stride * m_Capacity ) );
		if( m_pBuffer == NULL )
		{
			return E_OUTOFMEMORY;
		}
	}

	if( pInitialData != NULL )
	{
		m_Count = m_Capacity;
	}

	return S_OK;
}

ID3D11Buffer* VertexBuffer::GetInsterface( void )
{
	return m_pVertexBuffer;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::IVertexBuffer
////////////////////////////////////////////////////////////////////////////////////////////////////

Boolean VertexBuffer::Lock( void )
{
	if( ( m_bLock == True ) ||
		( m_bDynamic == False ) ||
		( m_pVertexBuffer == NULL ) )
	{
		return False;
	}

	if( m_ResizeStep > 0 )
	{
		m_pSource = &( m_pBuffer[0] );
	}
	else
	{
		if( m_pDevice->Map( m_pVertexBuffer, m_bDynamic, m_Mapped ) == S_OK )
		{
			m_pSource = static_cast<UInt8*>( m_Mapped.pData );
		}
		else
		{
			return False;
		}
	}

	m_Count = 0;
	m_bLock = True;

	return True;
}

void VertexBuffer::Unlock( void )
{
	if( ( m_bLock == False ) ||
		( m_pVertexBuffer == NULL ) )
	{
		return;
	}

	if( m_ResizeStep > 0 )
	{
		if( m_bResize == True )
		{
			HRESULT ret;
			ID3D11Buffer* pTempBuffer;

			ret = m_pDevice->CreateVertexBuffer( m_Capacity, m_Stride, m_bDynamic, m_pBuffer, &pTempBuffer );
			if( ret == S_OK )
			{
				MIX_RELEASE( m_pVertexBuffer );
				m_pVertexBuffer = pTempBuffer;
#ifdef _DEBUG
				MIX_LOG_INFO( L"o[ebNXobt@̃TCYύX : %s[%s] : %s[%d]", Mix::STR_DEBUGNAME, GetDebugName(), Mix::STR_CAPACITY, m_Capacity );
#endif //_DEBUG
			}
			else
			{
#ifdef _DEBUG
				MIX_LOG_ERROR( L"o[ebNXobt@̃TCY̕ύXɎs : %s[%s] %s[%s]", Mix::STR_DEBUGNAME, GetDebugName(), Mix::STR_RESULT, m_pDevice->GetD3DResultText( ret ) );
#else //_DEBUG
				MIX_LOG_ERROR( L"o[ebNXobt@̃TCY̕ύXɎs : %s[%s]", Mix::STR_RESULT, m_pDevice->GetD3DResultText( ret ) );
#endif //_DEBUG
			}

			m_bResize = False;
		}

		if( m_pDevice->Map( m_pVertexBuffer, m_bDynamic, m_Mapped ) == S_OK )
		{
			UInt32 copyCount = ( m_Capacity > m_Count )? m_Count : m_Capacity;
			Mix::Memory::Copy( m_Mapped.pData, m_pBuffer, ( copyCount * m_Stride ) );
			m_pDevice->Unmap( m_pVertexBuffer );
		}
	}
	else
	{
		m_pDevice->Unmap( m_pVertexBuffer );
	}

	m_bLock = False;
}

Boolean VertexBuffer::Push( const void* pData, UInt32 count )
{
	void* pDst = Push( count );
	if( pDst == NULL )
	{
		return False;
	}

	Mix::Memory::Copy( pDst, pData, ( m_Stride * count ) );

	return True;
}

void* VertexBuffer::Push( UInt32 count )
{
	if( m_bLock == False )
	{
		return NULL;
	}

	UInt32 size;
	UInt32 preCur;

	size = ( m_Count + count );

	if( m_ResizeStep > 0 )
	{
		if( size <= m_Capacity )
		{
			preCur = m_Count;
			m_Count += count;
		}
		else
		{
			UInt32 newCount;
			void* pTempBuff;
			
			newCount = ( size / m_ResizeStep );
			if( ( size % m_ResizeStep ) > 0 )
			{
				newCount++;
			}

			newCount *= m_ResizeStep;

			pTempBuff = MIX_LIB_REALLOC( m_pBuffer, ( newCount * m_Stride ) );
			if( pTempBuff != NULL )
			{
				m_pBuffer = reinterpret_cast<UInt8*>( pTempBuff );

				m_pSource = &( m_pBuffer[0] );
				m_Capacity = newCount;

				preCur = m_Count;
				m_Count += count;

				m_bResize = True;
			}
			else
			{
				return NULL;
			}
		}
	}
	else
	{
		if( size <= m_Capacity )
		{
			preCur = m_Count;
			m_Count += count;
		}
		else
		{
			return NULL;
		}
	}

	return ( m_pSource + ( preCur * m_Stride ) );
}

void VertexBuffer::Clear( void )
{
	m_Count = 0;
}

UInt32 VertexBuffer::GetStride( void ) const
{
	return m_Stride;
}

UInt32 VertexBuffer::GetCount( void ) const
{
	return m_Count;
}

UInt32 VertexBuffer::GetCapacity( void ) const
{
	return m_Capacity;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics::IDeviceObject
////////////////////////////////////////////////////////////////////////////////////////////////////

Mix::Graphics::IDeviceObject::TYPE VertexBuffer::GetType( void ) const
{
	return Mix::Graphics::IDeviceObject::VERTEX_BUFFER;
}

}}}
