#include "Mix/Geometry/Frustum.h"
#include "Mix/Geometry.h"

namespace Mix{ namespace Geometry{

Frustum::Frustum( void )
{
}

Frustum::Frustum( const Mix::Matrix4x4& viewProjMat ) :
m_ViewProjMat( viewProjMat )
{
	Update( viewProjMat );
}

void Frustum::Update( const Mix::Matrix4x4& viewProjMat )
{
	Mix::Geometry::Plane* p;

	m_ViewProjMat = viewProjMat;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	//Near
	p = &( m_Planes[Frustum::PLANE_NEAR] );
	p->normal.x = m_ViewProjMat.m03 + m_ViewProjMat.m02;
	p->normal.y = m_ViewProjMat.m13 + m_ViewProjMat.m12;
	p->normal.z = m_ViewProjMat.m23 + m_ViewProjMat.m22;
	p->d = m_ViewProjMat.m33 + m_ViewProjMat.m32;

	//Far
	p = &( m_Planes[Frustum::PLANE_FAR] );
	p->normal.x = m_ViewProjMat.m03 - m_ViewProjMat.m02;
	p->normal.y = m_ViewProjMat.m13 - m_ViewProjMat.m12;
	p->normal.z = m_ViewProjMat.m23 - m_ViewProjMat.m22;
	p->d = m_ViewProjMat.m33 - m_ViewProjMat.m32;

	//Top
	p = &( m_Planes[Frustum::PLANE_TOP] );
	p->normal.x = m_ViewProjMat.m03 - m_ViewProjMat.m01;
	p->normal.y = m_ViewProjMat.m13 - m_ViewProjMat.m11;
	p->normal.z = m_ViewProjMat.m23 - m_ViewProjMat.m21;
	p->d = m_ViewProjMat.m33 - m_ViewProjMat.m31;

	//Bottom
	p = &( m_Planes[Frustum::PLANE_BOTTOM] );
	p->normal.x = m_ViewProjMat.m03 + m_ViewProjMat.m01;
	p->normal.y = m_ViewProjMat.m13 + m_ViewProjMat.m11;
	p->normal.z = m_ViewProjMat.m23 + m_ViewProjMat.m21;
	p->d = m_ViewProjMat.m33 + m_ViewProjMat.m31;

	//Left
	p = &( m_Planes[Frustum::PLANE_LEFT] );
	p->normal.x = m_ViewProjMat.m03 + m_ViewProjMat.m00;
	p->normal.y = m_ViewProjMat.m13 + m_ViewProjMat.m10;
	p->normal.z = m_ViewProjMat.m23 + m_ViewProjMat.m20;
	p->d = m_ViewProjMat.m33 + m_ViewProjMat.m30;

	//Right
	p = &( m_Planes[Frustum::PLANE_RIGHT] );
	p->normal.x = m_ViewProjMat.m03 - m_ViewProjMat.m00;
	p->normal.y = m_ViewProjMat.m13 - m_ViewProjMat.m10;
	p->normal.z = m_ViewProjMat.m23 - m_ViewProjMat.m20;
	p->d = m_ViewProjMat.m33 - m_ViewProjMat.m30;

	//K
	for( UInt32 i = 0; i < 6; i++ )
	{
		m_Planes[i].Normalize();
	}
}

const Mix::Matrix4x4& Frustum::GetViewProjectionMatrix( void ) const
{
	return m_ViewProjMat;
}

const Mix::Geometry::Plane& Frustum::GetPlane( Mix::Geometry::Frustum::PLANE_TYPE type ) const
{
	return m_Planes[type];
}

Boolean Frustum::Contains( const Mix::Vector3& point ) const
{
	Mix::Vector3 pos;

	pos = m_ViewProjMat * point;
	if( ( pos.x >= -1.0f ) &&
		( pos.x <=  1.0f ) &&
		( pos.y >= -1.0f ) &&
		( pos.y <=  1.0f ) &&
		( pos.z >=  0.0f ) &&
		( pos.z <=  1.0f ) )
	{
		return True;
	}

	return False;
}

Boolean Frustum::Contains( const Mix::Geometry::Sphere& sphere ) const
{
	const Mix::Vector3& pos = sphere.center;
	const Float32 radius = sphere.radius;

	const Mix::Geometry::Plane* pPlane = &( m_Planes[0] );
	const Mix::Geometry::Plane* pPlaneEnd = pPlane + 6;

	Float32 dist;

	while( pPlane != pPlaneEnd )
	{
#if 1
		const Mix::Vector3& norm = pPlane->normal;
		dist = ( norm.x * pos.x ) + ( norm.y * pos.y ) + ( norm.z * pos.z ) + pPlane->d;
#else
		dist = Mix::Vector3::Dot( pPlane->normal, pos ) + pPlane->d;
#endif

		if( -radius > dist )
		{
			return False;
		}

		pPlane++;
	}

	return True;
}

Boolean Frustum::ContainsTBLR( const Mix::Geometry::Sphere& sphere ) const
{
	const Mix::Vector3& pos = sphere.center;
	const Float32 radius = sphere.radius;

	const Mix::Geometry::Plane* pPlane = &( m_Planes[2] );
	const Mix::Geometry::Plane* pPlaneEnd = pPlane + 4;

	Float32 dist;

	while( pPlane != pPlaneEnd )
	{
#if 1
		const Mix::Vector3& norm = pPlane->normal;
		dist = ( norm.x * pos.x ) + ( norm.y * pos.y ) + ( norm.z * pos.z ) + pPlane->d;
#else
		dist = Mix::Vector3::Dot( pPlane->normal, pos ) + pPlane->d;
#endif
		if( -radius > dist )
		{
			return False;
		}

		pPlane++;
	}

	return True;
}

Boolean Frustum::Contains( const Mix::Geometry::AABB& aabb ) const
{
	const Mix::Geometry::Plane* pPlane = &( m_Planes[0] );
	const Mix::Geometry::Plane* pPlaneEnd = pPlane + 6;

	const Mix::Vector3* pPointBegin = &( aabb.points[0] );
	const Mix::Vector3* pPointEnd = pPointBegin + 8;
	const Mix::Vector3* pPoint = NULL;

	Float32 dist;
	UInt32 count;

	while( pPlane != pPlaneEnd )
	{
#if 1
		const Mix::Vector3& norm = pPlane->normal;
#endif
		count = 8;

		pPoint = pPointBegin;
		while( pPoint != pPointEnd )
		{
#if 1
			dist = ( norm.x * pPoint->x ) + ( norm.y * pPoint->y ) + ( norm.z * pPoint->z ) + pPlane->d;
#else
			dist = Mix::Vector3::Dot( pPlane->normal, *pPoint ) + pPlane->d;
#endif
			if( dist < 0.0f )
			{
				MIX_ASSERT( count > 0 );
				count--;
			}

			pPoint++;
		}

		if( count == 0 )
		{
			return False;
		}

		pPlane++;
	}

	return True;
}

Boolean Frustum::ContainsTBLR( const Mix::Geometry::AABB& aabb ) const
{
	const Mix::Geometry::Plane* pPlane = &( m_Planes[2] );
	const Mix::Geometry::Plane* pPlaneEnd = pPlane + 4;

	const Mix::Vector3* pPointBegin = &( aabb.points[0] );
	const Mix::Vector3* pPointEnd = pPointBegin + 8;
	const Mix::Vector3* pPoint = NULL;

	Float32 dist;
	UInt32 count;

	while( pPlane != pPlaneEnd )
	{
#if 1
		const Mix::Vector3& norm = pPlane->normal;
#endif
		count = 8;

		pPoint = pPointBegin;
		while( pPoint != pPointEnd )
		{
#if 1
			dist = ( norm.x * pPoint->x ) + ( norm.y * pPoint->y ) + ( norm.z * pPoint->z ) + pPlane->d;
#else
			dist = Mix::Vector3::Dot( pPlane->normal, *pPoint ) + pPlane->d;
#endif
			if( dist < 0.0f )
			{
				MIX_ASSERT( count > 0 );
				count--;
			}

			pPoint++;
		}

		if( count == 0 )
		{
			return False;
		}

		pPlane++;
	}

	return True;
}

}}
