#include "Mix/Vector4.h"
#include "Mix/Vector3.h"

namespace Mix{

Vector4::Vector4( void )
{
	x = 0.0f;
	y = 0.0f;
	z = 0.0f;
	w = 1.0f;
}

Vector4::Vector4( const Vector4& v )
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = v.w;
}

Vector4::Vector4( const Vector3& v )
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = 1.0f;
}

Vector4::Vector4( Float32 vx, Float32 vy, Float32 vz )
{
	x = vx;
	y = vy;
	z = vz;
	w = 1.0f;
}

Vector4::Vector4( Float32 vx, Float32 vy, Float32 vz, Float32 vw )
{
	x = vx;
	y = vy;
	z = vz;
	w = vw;
}

Boolean Vector4::operator == ( const Vector4& v ) const
{
	if( ( MIX_FLOAT_EQUAL( x, v.x ) == False ) ||
		( MIX_FLOAT_EQUAL( y, v.y ) == False ) ||
		( MIX_FLOAT_EQUAL( z, v.z ) == False ) ||
		( MIX_FLOAT_EQUAL( w, v.w ) == False ) )
	{
		return False;
	}

	return True;
}

Boolean Vector4::operator != ( const Vector4& v ) const
{
	if( ( MIX_FLOAT_EQUAL( x, v.x ) == False ) ||
		( MIX_FLOAT_EQUAL( y, v.y ) == False ) ||
		( MIX_FLOAT_EQUAL( z, v.z ) == False ) ||
		( MIX_FLOAT_EQUAL( w, v.w ) == False ) )
	{
		return True;
	}

	return False;
}

Vector4& Vector4::operator = ( const Vector4& v )
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = v.w;

	return *this;
}

Vector4& Vector4::operator = ( const Vector3& v )
{
	x = v.x;
	y = v.y;
	z = v.z;
	w = 1.0f;

	return *this;
}

Vector4& Vector4::operator += ( const Vector4& v )
{
	x += v.x;
	y += v.y;
	z += v.z;
	w += v.w;

	return *this;
}

Vector4& Vector4::operator -= ( const Vector4& v )
{
	x -= v.x;
	y -= v.y;
	z -= v.z;
	w -= v.w;

	return *this;
}

Vector4& Vector4::operator *= ( const Vector4& v )
{
	x *= v.x;
	y *= v.y;
	z *= v.z;
	w *= v.w;

	return *this;
}

Vector4& Vector4::operator /= ( const Vector4& v )
{
	x = MIX_FLOAT_DIV( x, v.x );
	y = MIX_FLOAT_DIV( y, v.y );
	z = MIX_FLOAT_DIV( z, v.z );
	w = MIX_FLOAT_DIV( w, v.w );

	return *this;
}

Vector4 Vector4::operator + ( const Vector4& v ) const
{
	Vector4 ret( *this );

	ret.x += v.x;
	ret.y += v.y;
	ret.z += v.z;
	ret.w += v.w;

	return ret;
}

Vector4 Vector4::operator - ( const Vector4& v ) const
{
	Vector4 ret( *this );

	ret.x -= v.x;
	ret.y -= v.y;
	ret.z -= v.z;
	ret.w -= v.w;

	return ret;
}

Vector4 Vector4::operator * ( const Vector4& v ) const
{
	Vector4 ret( *this );

	ret.x *= v.x;
	ret.y *= v.y;
	ret.z *= v.z;
	ret.w *= v.w;

	return ret;
}

Vector4 Vector4::operator / ( const Vector4& v ) const
{
	Vector4 ret( *this );

	ret.x = MIX_FLOAT_DIV( ret.x, v.x );
	ret.y = MIX_FLOAT_DIV( ret.y, v.y );
	ret.z = MIX_FLOAT_DIV( ret.z, v.z );
	ret.w = MIX_FLOAT_DIV( ret.w, v.w );

	return ret;
}

Vector4 Vector4::operator +( void ) const
{
	return *this;
}

Vector4 Vector4::operator -( void ) const
{
	return ( *this * -1.0f );
}

Vector4 Vector4::operator * ( Float32 s ) const
{
	Vector4 ret( *this );

	ret.x *= s;
	ret.y *= s;
	ret.z *= s;
	ret.w *= s;

	return ret;
}

Vector4 Vector4::operator / ( Float32 s ) const
{
	Vector4 ret;

	if( MIX_FLOAT_IS_ZERO( s ) == True )
	{
		ret.x = 0.0f;
		ret.y = 0.0f;
		ret.z = 0.0f;
		ret.w = 0.0f;
	}
	else
	{
		Float32 invS = 1.0f / s;

		ret.x = x * invS;
		ret.y = y * invS;
		ret.z = z * invS;
		ret.w = w * invS;
	}

	return ret;
}

Vector4& Vector4::operator *= ( Float32 s )
{
	x *= s;
	y *= s;
	z *= s;
	w *= s;

	return *this;
}

Vector4& Vector4::operator /= ( Float32 s )
{
	if( MIX_FLOAT_IS_ZERO( s ) == True )
	{
		x = 0.0f;
		y = 0.0f;
		z = 0.0f;
		w = 0.0f;
	}
	else
	{
		Float32 invS = 1.0f / s;

		x *= invS;
		y *= invS;
		z *= invS;
		w *= invS;
	}

	return *this;
}

void Vector4::Set( Float32 vx, Float32 vy, Float32 vz, Float32 vw )
{
	x = vx;
	y = vy;
	z = vz;
	w = vw;
}

Float32 Vector4::Normalize( void )
{
	Float32 length = GetLength();

	if( MIX_FLOAT_IS_ZERO( length ) == True )
	{
		x = 0.0f;
		y = 0.0f;
		z = 0.0f;
	}
	else
	{
		Float32 invLength = ( 1.0f / length );

		x *= invLength;
		y *= invLength;
		z *= invLength;
	}

	return length;
}

Vector4 Vector4::ToNormalize( Float32* pLength ) const
{
	Mix::Vector4 tmp( *this );
	Float32 length = tmp.Normalize();

	if( pLength == NULL )
	{
		tmp.Normalize();
	}
	else
	{
		*pLength = tmp.Normalize();
	}

	return tmp;
}

void Vector4::Saturate( void )
{
	x = MIX_CLAMP( x, 0.0f, 1.0f );
	y = MIX_CLAMP( y, 0.0f, 1.0f );
	z = MIX_CLAMP( z, 0.0f, 1.0f );
	w = MIX_CLAMP( w, 0.0f, 1.0f );
}

Vector4 Vector4::ToSaturate( void ) const
{
	Mix::Vector4 tmp( *this );

	tmp.Saturate();

	return tmp;
}

Float32 Vector4::GetLengthSqr( void ) const
{
	return ( x * x ) + ( y * y ) + ( z * z );
}

Float32 Vector4::GetLength( void ) const
{
	return ::sqrtf( ( x * x ) + ( y * y ) + ( z * z ) );
}

Float32 Vector4::GetLengthF( void ) const
{
	return Mix::SqrtF( ( x * x ) + ( y * y ) + ( z * z ) );
}

Float32 Vector4::Dot( const Vector4& lv, const Vector4& rv )
{
	return ( lv.x * rv.x ) + ( lv.y * rv.y ) + ( lv.z * rv.z ) + ( lv.w * rv.w );
}

Vector4 Vector4::Cross( const Vector4& lv, const Vector4& rv )
{
	Float32 y2z3 = ( lv.y * rv.z );
	Float32 z2y3 = ( lv.z * rv.y );
	Float32 z2x3 = ( lv.z * rv.x );
	Float32 x2z3 = ( lv.x * rv.z );
	Float32 x2y3 = ( lv.x * rv.y );
	Float32 y2x3 = ( lv.y * rv.x );

	return Vector4( ( y2z3 - z2y3 ), ( z2x3 - x2z3 ), ( x2y3 - y2x3 ), 1.0f );
}

Vector4 Vector4::Rand( void )
{
	Mix::Vector4 v;

	v.x = Mix::RandF();
	v.y = Mix::RandF();
	v.z = Mix::RandF();
	v.w = Mix::RandF();

	return v;
}

Vector4 Vector4::Rand( const Mix::Vector4& v )
{
	Mix::Vector4 vv;

	vv.x = Mix::RandF( v.x );
	vv.y = Mix::RandF( v.y );
	vv.z = Mix::RandF( v.z );
	vv.w = Mix::RandF( v.w );

	return vv;
}

Vector4 Vector4::Rand( const Mix::Vector4& v0, const Mix::Vector4& v1 )
{
	Mix::Vector4 v;

	v.x = Mix::RandF( v0.x, v1.x );
	v.y = Mix::RandF( v0.y, v1.y );
	v.z = Mix::RandF( v0.z, v1.z );
	v.w = Mix::RandF( v0.w, v1.w );

	return v;
}

Vector4 Vector4::Min( const Mix::Vector4& v0, const Mix::Vector4& v1 )
{
	Mix::Vector4 v;

	v.x = min( v0.x, v1.x );
	v.y = min( v0.y, v1.y );
	v.z = min( v0.z, v1.z );
	v.w = min( v0.w, v1.w );

	return v;
}

Vector4 Vector4::Max( const Mix::Vector4& v0, const Mix::Vector4& v1 )
{
	Mix::Vector4 v;

	v.x = max( v0.x, v1.x );
	v.y = max( v0.y, v1.y );
	v.z = max( v0.z, v1.z );
	v.w = max( v0.w, v1.w );

	return v;
}

Vector4 Vector4::Lerp( const Mix::Vector4& vs, const Mix::Vector4& ve, Float32 t )
{
	Float32 tt = MIX_CLAMP( t, 0.0f, 1.0f );

	return ( vs * ( 1.0f - tt ) ) + ( ve * tt );
}

Mix::Vector4 Vector4::Slerp( const Mix::Vector4& vs, const Mix::Vector4& ve, Float32 t )
{
	Mix::Vector4 s = vs.ToNormalize();
	Mix::Vector4 e = ve.ToNormalize();

	float angle = acosf( min( 1.0f, Mix::Vector4::Dot( s, e ) ) );
	float sinTh = sinf( angle );
	float invSinTh = MIX_FLOAT_RECIPROCAL( sinTh );

	float Ps = sinf( angle * ( 1.0f - t ) );
	float Pe = sinf( angle * t );

	return ( ( Ps * s + Pe * e ) * invSinTh ).ToNormalize();
}

}